//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AlignTextMeasureContext.h"
#include <Candera/TextEngine/Style.h>

namespace Candera { namespace TextRendering {

    AlignTextMeasureContext::AlignTextMeasureContext(const Style& style, const TextRenderContext* referenceContext) :
        TextMeasureContext(referenceContext),
        m_isWhiteSpaceLeading(true),
        m_ignoreTrailingSpaces(false),
        m_ascender(style.GetMetrics().ascender),
        m_descender(style.GetMetrics().descender),
        m_textRectangle(TextRect::GetMin())
    {
    }

    void AlignTextMeasureContext::Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph)
    {
        if ((m_ignoreTrailingSpaces) && (glyph.width <= 0) && (m_isWhiteSpaceLeading == false)) {
            return;
        }
        m_isWhiteSpaceLeading = false;

        // compute bounds
        PixelPosition left = x - glyph.left;
        PixelPosition right = left + glyph.xadvance;

        PixelPosition glyphBaseLine = y + glyph.top;
        PixelPosition top = (glyphBaseLine - m_ascender) + 1;
        PixelPosition bottom = glyphBaseLine - m_descender;

        m_textRectangle = m_textRectangle.Union(
            TextRect(left, top, right, bottom));
    }

    const TextRect& AlignTextMeasureContext::GetTextRectangle() const
    {
        return m_textRectangle.IsNegative() ? TextRect::GetDefault() : m_textRectangle;
    }

    AlignLayoutStrategy::AlignLayoutStrategy(AlignTextMeasureContext& measureContext, TextLayoutStrategy* strategy) :
        m_alignMeasureContext(measureContext),
        m_next(strategy)
    {
    }

    TextLayoutStrategy::Action AlignLayoutStrategy::OnInit(Pass pass)
    {
        m_alignMeasureContext.SignalNewLine();
        return (m_next != 0) ? m_next->OnInit(pass) : Continue;
    }

    TextLayoutStrategy::Action AlignLayoutStrategy::OnLineBreak(const Char* textCursor, LineBreakType type)
    {
        m_alignMeasureContext.SignalNewLine();
        return (m_next != 0) ? m_next->OnLineBreak(textCursor, type) : Continue;
    }

    TextLayoutStrategy::Action AlignLayoutStrategy::OnGlyphProcessing(TextPosition textPosition)
    {
        return (m_next != 0) ? m_next->OnGlyphProcessing(textPosition) : Continue;
    }

    }    // namespace TextRendering
}    // namespace Candera
