//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_UPDATESYSTEM_H
#define CANDERA_UPDATESYSTEM_H

#include <Candera/System/EntityComponentSystem/ComponentSystem.h>
#include <Candera/System/UpdateSystem/UpdateComponent.h>

namespace Candera {

namespace EntityComponentSystem { class EntitySystem; }

/** @addtogroup UpdateSystem
 *  @{
 */

/**
 * @brief  The UpdateSystem is a component system that is globally accessible via the EntitySystem.
 *         The UpdateSystem dispatches a series of update calls to entities that have an UpdateComponent
 *         attached. Which particular call is delegated to an entity is configured in the attached
 *         UpdateComponent using UpdateDelegates. An UpdateDelegate is a wrapper for the entity's
 *         member function that is called by the UpdateSystem.
 *
 *         Every render frame, the UpdateSystem dispatches the following calls in that order:
 *         FixedUpdate() (the number of times depends on the fixed time step),
 *         Update() or Update(applicationTimeInSeconds, deltaTimeInSeconds) or Update(worldTimeInMilliseconds, deltaTimeInMilliseconds),
 *         and LateUpdate().
 *
 *         The UpdateSystem also provides the following time measurements that should be applied application wide to achieve a
 *         consistent usage of frame time:
 *         Application time and delta time (including smoothed values) in seconds measured at the beginning of the render frame,
 *         Fixed application time and fixed delta time in seconds measured using the configurable fixed time step,
 *         World time and delta time (including smoothed values) in milliseconds measured at the beginning of the render frame.
 *
 *         Please note that the world time is only supplied for legacy reasons. Application time should be preferred by all means.
 *
 *         Please see UpdateComponent and UpdateDelegate for further information.
 *         Also see the CGI Studio documentation for a code example.
 */

class UpdateSystem : public EntityComponentSystem::ComponentSystem<UpdateComponent, UpdateEntity>
{
    typedef EntityComponentSystem::ComponentSystem<UpdateComponent, UpdateEntity> Base;

public:
    typedef UpdateDelegate Delegate;

    FEATSTD_RTTI_DECLARATION();

    /**
     *  Destructor
     */
    virtual ~UpdateSystem();

    /**
     *  If the component system is enabled, this function is called regularly by the entity system.
     */
    virtual void Update() override;

    /**
     *  Set the delegate of the FixedUpdate callback for the handle. The FixedUpdate is called
     *  1/FixedTimeStep times per second independent of the actual render frame rate.
     *  @param handle    The handle of the component to set the delegate for.
     *  @param delegate  The delegate wrapping the user FixedUpdate function to be called
     *                   1/FixedTimeStep times per second.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool SetComponentFixedUpdateDelegate(Handle handle, Delegate delegate) const;

    /**
     *  Set the delegate of the Update callback for the handle.
     *  @param handle    The handle of the component to set the delegate for.
     *  @param delegate  The delegate wrapping the user Update function to be called every render frame.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool SetComponentUpdateDelegate(Handle handle, Delegate delegate) const;

    /**
     *  Set the delegate of the LateUpdate callback for the handle.
     *  @param handle    The handle of the component to set the delegate for.
     *  @param delegate  The delegate wrapping the user LateUpdate function to be called every render frame
     *                   after all other Update callbacks have been executed.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    bool SetComponentLateUpdateDelegate(Handle handle, Delegate delegate) const;

    /**
     *  Set the priority of the update component.
     *  @param handle    The handle of the component to set the priority for.
     *  @param priority  The priority of the update component to be set.
     */
    bool SetComponentPriority(Handle handle, Float priority);

    /**
     *  Set the fixed time step in seconds. The UpdateSystem performs 1/FixedTimeStep callbacks per second on
     *  its components via their FixedUpdate delegates.
     *  @param fixedTimeStepInSeconds  The fixed time step to use for FixedUpdate callbacks.
     */
    void SetFixedTimeStep(Double fixedTimeStepInSeconds) { m_fixedTimeStepInSeconds = fixedTimeStepInSeconds; }

    /**
     *  Get the fixed time step in seconds. The UpdateSystem performs 1/FixedTimeStep callbacks per second on
     *  its components via their FixedUpdate delegates.
     *  @return  The fixed time step used to perform FixedUpdate callbacks.
     */
    Double GetFixedTimeStep() const { return m_fixedTimeStepInSeconds; }

    /**
     *  Get the world time of this update frame in milliseconds.
     *  It is measured once at the beginning of every update frame.
     *  @return  The world time of this update frame in milliseconds.
     */
    TimeType GetWorldTimeInMilliseconds() const { return m_worldTimeInMilliseconds; }

    /**
     *  Get the time that has passed since last update in milliseconds.
     *  @return  The time that passed since last update in milliseconds.
     */
    TimeType GetDeltaTimeInMilliseconds() const { return m_deltaTimeInMilliseconds; }

    /**
     *  Get the smoothed world time of this update frame in milliseconds.
     *  It is measured once at the beginning of every update frame.
     *  @return  The world time of this update frame in milliseconds.
     */
    TimeType GetSmoothedWorldTimeInMilliseconds() const { return m_smoothedWorldTimeInMilliseconds; }

    /**
     *  Get the smoothed time that has passed since last update in milliseconds.
     *  @return  The time that passed since last update in milliseconds.
     */
    TimeType GetSmoothedDeltaTimeInMilliseconds() const { return m_smoothedDeltaTimeInMilliseconds; }

    /**
     *  Get the application time of this update frame in seconds.
     *  It is measured once at the beginning of every update frame, starting with the construction
     *  of the UpdateSystem. The application time excludes intervals in which it was paused.
     *  @return  The application time of this update frame in seconds.
     */
    Double GetApplicationTime() const;

    /**
     *  Get the time that has passed since last update in seconds.
     *  @return  The time that passed since last update in seconds.
     */
    Double GetDeltaTime() const;

    /**
     *  Get the smoothed time that has passed since last update in seconds.
     *  @return  The time that passed since last update in seconds.
     */
    Double GetSmoothedDeltaTime() const;

    /**
     *  Get the application time of current fixed frame in seconds.
     *  It is measured once at the beginning of every fixed frame, starting with the construction
     *  of the UpdateSystem. The fixed application time excludes all intervals in which it was paused.
     *  @return  The application time of the current fixed frame in seconds.
     */
    Double GetFixedApplicationTime() const;

    /**
     *  Get the fixed delta time (i.e. fixed time step) in seconds. The UpdateSystem
     *  performs 1/FixedTimeStep callbacks per second on its components via their FixedUpdate
     *  delegates.
     *  @return  The fixed time step used to perform FixedUpdate callbacks.
     */
    Double GetFixedDeltaTime() const;

    /**
     *  Get the application realtime in seconds.
     *  For most application cases GetApplicationTime() is a better choice due to the consistent nature
     *  of frame time instead of realtime.
     *  @return  The application realtime in seconds.
     */
    Double GetRealtime() const;

    /**
     *  Enable or disable time smoothing.
     *  @param isSmoothTimeEnabled  'true' enables time smoothing, 'false' disables time smoothing.
     */
    void SetSmoothTimeEnabled(bool isSmoothTimeEnabled) { m_isSmoothTimeEnabled = isSmoothTimeEnabled; }

    /**
     *  Check if smoothing time is enabled.
     *  @return  True, if smoothing time is enabled. False, otherwise.
     */
    bool IsSmoothTimeEnabled() const { return m_isSmoothTimeEnabled; }

    /**
     *  Pause the application time. This also pauses callbacks on update delegates that receive application time.
     */
    void PauseApplicationTime();

    /**
     *  Check if the application time is paused ('true') or not ('false').
     *  @return  True, if the application time is paused. False, otherwise.
     */
    bool IsApplicationTimePaused() const { return m_isApplicationTimePaused; }

    /**
     *  Reset the application time.
     */
    void ResetApplicationTime();

protected:
    /**
     *  Callback after a component was attached.
     *  @param handle  The handle of the component that was attached.
     *  @param entity  The entity the component was attached to.
     *  @return  True, if the operation was successful. False, otherwise.
     */
    virtual bool OnAttachComponent(Handle handle, UpdateEntity* const entity) override;

    /**
     * Calculate world, delta, and application time.
     */
    void CalculateTime();

private:
    /**
     *  Constructor
     */
    UpdateSystem();

    void SortComponents();
    Double GetTimeInSecondsSinceTimeInPerfCounterUnits(TimeType sinceTime) const;

    friend class EntityComponentSystem::EntitySystem;

    TimeType m_startTimeInPerfCounterUnits;
    TimeType m_pausedTimeInPerfCounterUnits;
    Double m_totalPausedTimeInSeconds;

    // time in milliseconds
    TimeType m_worldTimeInMilliseconds;
    TimeType m_deltaTimeInMilliseconds;

    // smoothed time in milliseconds
    TimeType m_smoothedWorldTimeInMilliseconds;
    TimeType m_smoothedDeltaTimeInMilliseconds;

    // application time in seconds
    Double m_applicationTimeInSeconds;
    Double m_deltaTimeInSeconds;

    Double m_previousApplicationTimeInSeconds;

    // smoothed application time in seconds
    Double m_smoothedDeltaTimeInSeconds;

    // fixed frame rate time in seconds
    Double m_fixedTimeInSeconds;
    Double m_fixedTimeStepInSeconds;
    Double m_fixedTimeStepRemainder;

    bool m_isSmoothTimeEnabled;
    bool m_isApplicationTimePaused;
    bool m_isApplicationTimeFirstMeasurement;
};

/** @} */ // end of UpdateSystem

} // namespace Candera

#endif
