//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "UpdateSystem.h"

#include <CanderaPlatform/OS/TimePlatform.h>
#include <FeatStd/Platform/PerfCounter.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(UpdateSystem, Base)

UpdateSystem::UpdateSystem()
    :
    m_startTimeInPerfCounterUnits(FeatStd::Internal::PerfCounter::Now()),
    m_pausedTimeInPerfCounterUnits(0),
    m_totalPausedTimeInSeconds(0.0),
    m_worldTimeInMilliseconds(0),
    m_deltaTimeInMilliseconds(0),
    m_smoothedWorldTimeInMilliseconds(0),
    m_smoothedDeltaTimeInMilliseconds(0),
    m_applicationTimeInSeconds(0.0),
    m_deltaTimeInSeconds(0.0),
    m_previousApplicationTimeInSeconds(0.0),
    m_smoothedDeltaTimeInSeconds(0.0),
    m_fixedTimeInSeconds(0.0),
    m_fixedTimeStepInSeconds(1.0 / 60.0),
    m_fixedTimeStepRemainder(0.0),
    m_isSmoothTimeEnabled(false),
    m_isApplicationTimePaused(false),
    m_isApplicationTimeFirstMeasurement(true)
{
}

UpdateSystem::~UpdateSystem()
{
}

void UpdateSystem::Update()
{
    CalculateTime();

    // FixedUpdate calls
    while (m_fixedTimeStepRemainder < m_deltaTimeInSeconds) {
        for (SizeType i = 0; i < m_components.Size(); ++i) {
            UpdateComponent* component = m_components[i];
            UpdateEntity* entity = component->GetEntity();
            if (0 != entity)
            {
                component->m_fixedUpdate(entity);
            }
        }

        m_fixedTimeStepRemainder += m_fixedTimeStepInSeconds;
        m_fixedTimeInSeconds += m_fixedTimeStepInSeconds;
    }

    m_fixedTimeStepRemainder -= m_deltaTimeInSeconds;

    // Update calls
    TimeType worldTimeInMilliseconds =  GetWorldTimeInMilliseconds();
    TimeType deltaTimeInMilliseconds = GetDeltaTimeInMilliseconds();
    Double applicationTimeInSeconds = GetApplicationTime();
    Double deltaTimeInSeconds = GetDeltaTime();

    if (m_isSmoothTimeEnabled) {
        worldTimeInMilliseconds = GetSmoothedWorldTimeInMilliseconds();
        deltaTimeInMilliseconds = GetSmoothedDeltaTimeInMilliseconds();
        deltaTimeInSeconds = GetSmoothedDeltaTime();
    }

    for (SizeType i = 0; i < m_components.Size(); ++i) {
        UpdateComponent* component = m_components[i];
        UpdateEntity* entity = component->GetEntity();
        if (0 != entity) {
            switch (component->m_update.m_functionType)
            {
            case Delegate::NoArgs:
                component->m_update(entity);
                break;

            case Delegate::DoubleDoubleArgs:
                component->m_update(entity, applicationTimeInSeconds, deltaTimeInSeconds);
                break;

            case Delegate::TimeTypeTimeTypeArgs:
                component->m_update(entity, worldTimeInMilliseconds, deltaTimeInMilliseconds);
                break;

            default:
                break;
            }
        }
    }

    // LateUpdate calls
    for (SizeType i = 0; i < m_components.Size(); ++i) {
        UpdateComponent* component = m_components[i];
        UpdateEntity* entity = component->GetEntity();
        if (0 != entity) {
            component->m_lateUpdate(entity);
        }
    }
}

bool UpdateSystem::SetComponentFixedUpdateDelegate(Handle handle, Delegate delegate) const
{
    UpdateComponent* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    component->m_fixedUpdate = delegate;
    return true;
}

bool UpdateSystem::SetComponentUpdateDelegate(Handle handle, Delegate delegate) const
{
    UpdateComponent* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    component->m_update = delegate;
    return true;
}

bool UpdateSystem::SetComponentLateUpdateDelegate(Handle handle, Delegate delegate) const
{
    UpdateComponent* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    component->m_lateUpdate = delegate;
    return true;
}

bool UpdateSystem::SetComponentPriority(Handle handle, Float priority)
{
    UpdateComponent* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
    if (priority != component->m_priority) {
        component->m_priority = priority;
        SortComponents();
    }

    return true;
}

Double UpdateSystem::GetApplicationTime() const
{
    if (IsApplicationTimePaused()) {
        UInt64 timeInMicroSec = static_cast<UInt64>(m_pausedTimeInPerfCounterUnits - m_startTimeInPerfCounterUnits) * FEATSTD_PERFCOUNTER_RESOLUTION;
        return (static_cast<Double>(timeInMicroSec) / (1000.0 * 1000.0)) - m_totalPausedTimeInSeconds;
    }

    return m_applicationTimeInSeconds - m_totalPausedTimeInSeconds;
}

Double UpdateSystem::GetDeltaTime() const
{
    return (IsApplicationTimePaused()) ? 0.0 : m_deltaTimeInSeconds;
}

Double UpdateSystem::GetSmoothedDeltaTime() const
{
    return (IsApplicationTimePaused()) ? 0.0 : m_smoothedDeltaTimeInSeconds;
}

Double UpdateSystem::GetFixedApplicationTime() const
{
    if (IsApplicationTimePaused()) {
        UInt64 timeInMicroSec = static_cast<UInt64>(m_pausedTimeInPerfCounterUnits - m_startTimeInPerfCounterUnits) * FEATSTD_PERFCOUNTER_RESOLUTION;
        return (static_cast<Double>(timeInMicroSec) / (1000.0 * 1000.0)) - m_totalPausedTimeInSeconds;
    }

    return m_fixedTimeInSeconds - m_totalPausedTimeInSeconds;
}

Double UpdateSystem::GetFixedDeltaTime() const
{
    return (IsApplicationTimePaused()) ? 0.0 : GetFixedTimeStep();
}

Double UpdateSystem::GetRealtime() const
{
    return GetTimeInSecondsSinceTimeInPerfCounterUnits(m_startTimeInPerfCounterUnits) - m_totalPausedTimeInSeconds;
}

void UpdateSystem::PauseApplicationTime()
{
    m_isApplicationTimePaused = !m_isApplicationTimePaused;

    if (IsApplicationTimePaused()) {
        m_pausedTimeInPerfCounterUnits = FeatStd::Internal::PerfCounter::Now();
    }
    else {
        m_totalPausedTimeInSeconds += GetTimeInSecondsSinceTimeInPerfCounterUnits(m_pausedTimeInPerfCounterUnits);
    }
}

void UpdateSystem::ResetApplicationTime()
{
    m_isApplicationTimeFirstMeasurement = true;
    m_totalPausedTimeInSeconds = 0.0;
    m_startTimeInPerfCounterUnits = FeatStd::Internal::PerfCounter::Now();
}

bool UpdateSystem::OnAttachComponent(Handle handle, UpdateEntity* const entity)
{
    FEATSTD_UNUSED(entity);
    const UpdateComponent* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    SortComponents();
    return true;
}

Double UpdateSystem::GetTimeInSecondsSinceTimeInPerfCounterUnits(TimeType sinceTime) const
{
    // If the size of TimeType changes, the overflow handling needs to be adjusted.
    FEATSTD_COMPILETIME_ASSERT(sizeof(TimeType) == sizeof(UInt32));

    const UInt timeTypeBitCount = sizeof(TimeType) * 8;
    static UInt64 s_overflowCount = 0;
    static TimeType s_previousTime = sinceTime;
    TimeType currentTime = FeatStd::Internal::PerfCounter::Now();
    if (s_previousTime > currentTime) {
        // An overflow happens every ~23 hours.
        // There are 32-log2(FEATSTD_PERFCOUNTER_RESOLUTION) bits available to count overflows, and correct the time accordingly.
        // With the current resolution of 20 micro seconds, this is 2^27 overflows, before an overflow of overflows happens.
        ++s_overflowCount;
    }

    s_previousTime = currentTime;

    // Resolution depends on PerformanceCount, which is FEATSTD_PERFCOUNTER_RESOLUTION micro seconds.
    UInt64 elapsedTimeInMicroSec = ((s_overflowCount << timeTypeBitCount) + currentTime - sinceTime) * FEATSTD_PERFCOUNTER_RESOLUTION;
    return static_cast<Double>(elapsedTimeInMicroSec) / (1000.0 * 1000.0);
}

void UpdateSystem::CalculateTime()
{
    // World time and world delta time in milliseconds.
    // Overflow of world time happens every ~49 days, delta times are not affected by it.
    static TimeType s_previousWorldTimeInMilliseconds = TimePlatform::GetMilliSeconds();
    m_worldTimeInMilliseconds = TimePlatform::GetMilliSeconds();

    m_deltaTimeInMilliseconds = m_worldTimeInMilliseconds - s_previousWorldTimeInMilliseconds;
    s_previousWorldTimeInMilliseconds = m_worldTimeInMilliseconds;

    // Application time and application delta time in seconds.
    m_applicationTimeInSeconds = GetTimeInSecondsSinceTimeInPerfCounterUnits(m_startTimeInPerfCounterUnits);
    if (m_isApplicationTimeFirstMeasurement) {
        m_previousApplicationTimeInSeconds = m_applicationTimeInSeconds;
        m_fixedTimeInSeconds = m_applicationTimeInSeconds;
    }

    m_deltaTimeInSeconds = m_applicationTimeInSeconds - m_previousApplicationTimeInSeconds;
    m_previousApplicationTimeInSeconds = m_applicationTimeInSeconds;

    // Smooth time
    const UInt16 frameCount = 10;
    static UInt16 s_index = 0;

    // Smoothed world time and world delta time in milliseconds.
    static TimeType s_previousSmoothedWorldTimeInMilliseconds = s_previousWorldTimeInMilliseconds;
    static TimeType s_frameTimeInMilliseconds[frameCount] = { 0 };
    static TimeType s_sumOfFrameDeltasInMilliseconds = 0;
    if (m_isApplicationTimeFirstMeasurement) {
        s_index = 0;
        s_sumOfFrameDeltasInMilliseconds = 0;
        MemoryPlatform::Set(&s_frameTimeInMilliseconds, 0, frameCount * sizeof(TimeType));
    }
    s_sumOfFrameDeltasInMilliseconds = s_sumOfFrameDeltasInMilliseconds - s_frameTimeInMilliseconds[s_index];
    s_frameTimeInMilliseconds[s_index] = m_deltaTimeInMilliseconds;
    s_sumOfFrameDeltasInMilliseconds = s_sumOfFrameDeltasInMilliseconds + m_deltaTimeInMilliseconds;
    m_smoothedDeltaTimeInMilliseconds = (s_sumOfFrameDeltasInMilliseconds / frameCount);
    m_smoothedWorldTimeInMilliseconds = s_previousSmoothedWorldTimeInMilliseconds + m_smoothedDeltaTimeInMilliseconds;
    s_previousSmoothedWorldTimeInMilliseconds = m_smoothedWorldTimeInMilliseconds;

    // Smoothed application time and application delta time in seconds.
    static Double s_frameTimeInSeconds[frameCount] = { 0.0 };
    static Double s_sumOfFrameDeltasInSeconds = 0.0;
    if (m_isApplicationTimeFirstMeasurement) {
        s_sumOfFrameDeltasInSeconds = 0.0;
        MemoryPlatform::Set(&s_frameTimeInSeconds, 0, frameCount * sizeof(Double));
    }
    s_sumOfFrameDeltasInSeconds = s_sumOfFrameDeltasInSeconds - s_frameTimeInSeconds[s_index];
    s_frameTimeInSeconds[s_index] = m_deltaTimeInSeconds;
    s_sumOfFrameDeltasInSeconds = s_sumOfFrameDeltasInSeconds + m_deltaTimeInSeconds;
    m_smoothedDeltaTimeInSeconds = (s_sumOfFrameDeltasInSeconds / static_cast<Double>(frameCount));

    s_index = (s_index + 1) % frameCount;

    m_isApplicationTimeFirstMeasurement = false;
}

struct ComponentPriorityComparator
{
    bool operator()(const UpdateComponent* a, const UpdateComponent* b) const {
        return (a->GetPriority() > b->GetPriority());
    }
};

void UpdateSystem::SortComponents()
{
    ComponentPriorityComparator comparator;
    m_components.Sort(comparator);
}

} // namespace Candera
