//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_UPDATEDELEGATE_H
#define CANDERA_UPDATEDELEGATE_H

#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera {

class UpdateSystem;

/** @addtogroup UpdateSystem
 *  @{
 */

/**
 * @brief  The UpdateDelegate wraps a member function of an entity class to be called by the UpdateSystem.
 *         It can wrap the following 3 types of functions (which are tailored to the needs of the UpdateSystem):
 *         Functions with no arguments (e.g. FixedUpate(), Update(), LateUpdate()),
 *         Functions with 2 Double arguments (e.g. Update(Double applicationTimeInSeconds, Double deltaTimeInSeconds)),
 *         Functions with 2 TimeType arguments (e.g. Update(TimeType worldTimeInMilliseconds, TimeType deltaTimeInMilliseconds)).
 *
 *         Please see UpdateSystem and UpdateComponent for further information.
 *         See also the CGI Studio documentation for a code example.
 */

class UpdateDelegate
{
public:
    UpdateDelegate()
        :
        m_stub(0),
        m_functionType(None)
    {}

    /**
     *  Create a delegate encapsulating a pointer to the member function with no arguments called
     *  TFunction of class T.
     *  @return  The delegate encapsulating a pointer to the member function TFunction of class T.
     */
    template<class T, void (T::*TFunction)()>
    static UpdateDelegate Update()
    {
        UpdateDelegate delegate;
        delegate.m_stub = &Stub<T, TFunction>;
        delegate.m_functionType = NoArgs;
        return delegate;
    }

    /**
     *  Calls the member function of the given object with no arguments.
     *  @param object  The object whose member function is called.
     */
    void operator()(void* object) const
    {
        if ((0 != object) && (0 != m_stub)) {
            FEATSTD_DEBUG_ASSERT(NoArgs == m_functionType);
            (*m_stub)(object);
        }
    }

    /**
     *  Create a delegate encapsulating a pointer to the member function with 2 Double arguments called
     *  TFunction of class T.
     *  @return  The delegate encapsulating a pointer to the member function TFunction of class T.
     */
    template<class T, void (T::*TFunction)(Double, Double)>
    static UpdateDelegate UpdateWithSeconds()
    {
        UpdateDelegate delegate;
        delegate.m_stubDoubleDouble = &StubDoubleDouble<T, TFunction>;
        delegate.m_functionType = DoubleDoubleArgs;
        return delegate;
    }

    /**
     *  Calls the member function of the given object with two arguments of type Double.
     *  @param object     The object whose member function is called.
     *  @param argument1  The first argument that is given to the member function.
     *  @param argument2  The second argument that is given to the member function.
     */
    void operator()(void* object, Double argument1, Double argument2) const
    {
        if ((0 != object) && (0 != m_stubDoubleDouble)) {
            FEATSTD_DEBUG_ASSERT(DoubleDoubleArgs == m_functionType);
            (*m_stubDoubleDouble)(object, argument1, argument2);
        }
    }

    /**
     *  Create a delegate encapsulating a pointer to the member function with 2 TimeType arguments called
     *  TFunction of class T.
     *  @return  The delegate encapsulating a pointer to the member function TFunction of class T.
     */
    template<class T, void (T::*TFunction)(TimeType, TimeType)>
    static UpdateDelegate UpdateWithMilliseconds()
    {
        UpdateDelegate delegate;
        delegate.m_stubTimeTypeTimeType = &StubTimeTypeTimeType<T, TFunction>;
        delegate.m_functionType = TimeTypeTimeTypeArgs;
        return delegate;
    }

    /**
     *  Calls the member function of the given object with two arguments of type TimeType.
     *  @param object     The object whose member function is called.
     *  @param argument1  The first argument that is given to the member function.
     *  @param argument2  The second argument that is given to the member function.
     */
    void operator()(void* object, TimeType argument1, TimeType argument2) const
    {
        if ((0 != object) && (0 != m_stubTimeTypeTimeType)) {
            FEATSTD_DEBUG_ASSERT(TimeTypeTimeTypeArgs == m_functionType);
            (*m_stubTimeTypeTimeType)(object, argument1, argument2);
        }
    }

private:
    friend class UpdateSystem;
    enum FunctionType {
        NoArgs,                     // No arguments.
        DoubleDoubleArgs,           // Double applicationTime, Double deltaTime.
        TimeTypeTimeTypeArgs,       // TimeType worldTime, TimeType deltaTime.
        None
    };

    typedef void(*StubType)(void* object);
    typedef void(*StubTypeTimeTypeTimeType)(void* object, TimeType, TimeType);
    typedef void(*StubTypeDoubleDouble)(void* object, Double, Double);

    union {
        StubType m_stub;
        StubTypeTimeTypeTimeType m_stubTimeTypeTimeType;
        StubTypeDoubleDouble m_stubDoubleDouble;
    };

    FunctionType m_functionType;

    // no argument
    template<class T, void (T::*TFunction)()>
    static void Stub(void* object)
    {
        T* p = static_cast<T*>(object);
        return (p->*TFunction)();
    }

    // TimeType argument
    template<class T, void (T::*TFunction)(TimeType, TimeType)>
    static void StubTimeTypeTimeType(void* object, TimeType argument1, TimeType argument2)
    {
        T* p = static_cast<T*>(object);
        return (p->*TFunction)(argument1, argument2);
    }

    // Double argument
    template<class T, void (T::*TFunction)(Double, Double)>
    static void StubDoubleDouble(void* object, Double argument1, Double argument2)
    {
        T* p = static_cast<T*>(object);
        return (p->*TFunction)(argument1, argument2);
    }
};

/** @} */ // end of UpdateSystem

} // namespace Candera

#endif
