//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DISPOSABLEPROPERTY_H)
#define CANDERA_DISPOSABLEPROPERTY_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/Disposer.h>

namespace Candera {

    /**
     *  @brief DisposableProperty is used for reference type properties where the destruction of the value is handled by a disposer function.
     */
    template <typename T>
    class DisposableProperty {
        public:
            typedef void(*DisposerFn)(const T*);

            /**
             *  Constructor
             */
            DisposableProperty();

            /**
             *  Destructor
             */
            ~DisposableProperty();

            /**
             *  Sets the value and the disposer.
             *  @param value        The value that is assigned.
             *  @param disposerFn   The disposer function that is assigned.
             */
            void Set(const T& value, DisposerFn disposerFn);

            /**
             *  Gets the current value.
             *  @return The current value.
             */
            const T& Get() const;

            /**
             *  Gets the current value.
             *  @return The current value.
             */
            T& Get();

            operator T();
            operator const T() const;

            T& operator()();
            const T& operator()() const;

        private:
            T m_value;
            DisposerFn m_disposerFn;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(DisposableProperty);
    };

    template<typename T>
    DisposableProperty<T>::DisposableProperty() :
        m_value(T()),
        m_disposerFn(0)
    {
    }

    template<typename T>
    DisposableProperty<T>::~DisposableProperty()
    {
        if (m_disposerFn != 0) {
            m_disposerFn(m_value);
        }
    }

    template<typename T>
    void DisposableProperty<T>::Set(const T& value, DisposerFn disposerFn)
    {
        if (m_disposerFn != 0) {
            m_disposerFn(m_value);
        }

        m_value = value;
        m_disposerFn = disposerFn;
    }

    template<typename T>
    inline const T& DisposableProperty<T>::Get() const
    {
        return m_value;
    }

    template<typename T>
    inline T& DisposableProperty<T>::Get()
    {
        return m_value;
    }

    template<typename T>
    inline DisposableProperty<T>::operator T()
    {
        return Get();
    }

    template<typename T>
    inline DisposableProperty<T>::operator const T() const
    {
        return Get();
    }

    template<typename T>
    inline T& DisposableProperty<T>::operator()()
    {
        return Get();
    }

    template<typename T>
    inline const T& DisposableProperty<T>::operator()() const
    {
        return Get();
    }

}   // namespace Candera

#endif  // CANDERA_DISPOSABLEPROPERTY_H
