//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(PUBLIC_MACROS_H)
#define PUBLIC_MACROS_H

#include <Candera/System/MetaInfo/GenericInternalMacros.h>

/// @addtogroup MetaInfo
/// @{
/**
 *  Defines the version of the meta info element.
 */
#define CdaVersion(version) \
    virtual Candera::Int32 GetVersion() const { return version; }

/** Opens a property definition section within a properties definition section.
    @param name name of the property (e.g. TextColor)
    @param type type of the property (e.g. Color)
    @param prop property (e.g. m_color) */
#define CdaFieldProperty(name, type, prop) \
    _CdaFieldPropertyDef(name, type, prop)

/**
 *  Closes a property definition section within a properties definition section.
 */
#define CdaPropertyEnd() \
    _CdaPropertyEpilog()

/**
 *  Defines the validity test method.
 *  Whenever the test returns a not null string, the property will be highlighted in SceneComposer property list and a string
 *   will be displayed as an error message.
 */
#define CdaValidityTest(validityTest)    \
    virtual const Candera::Char* GetValueValidityMessage(const BaseHostType* object) const { return (object == 0) ? NotAnObjectMessage() : static_cast<const Candera::Char*>(CdaCastOperator(object, (CurrentType*)0)->validityTest); }

/**
 * Defines the type to be used as return type of a property value validity method.
 * Valid values should return CdaValidValue()
 * Invalid values should return CdaInvalidValue(ErrorMessage)
 */
#define CdaPropertyMetaInfoErrorType const Candera::Char*

/**
 * Defines CdaPropertyMetaInfoErrorType value for valid property values.
 */
#define CdaValidValue() 0

#ifdef CANDERA_META_DESCRIPTION
    /**
     * Defines CdaPropertyMetaInfoErrorType value for invalid property values.
     */
    #define CdaInvalidValue(message) message

    /**
     *  Defines a readable name for a meta information object
     *  Readable names will be displayed by SceneComposer wherever applicable.
     */
    #define CdaReadableName(name) \
        virtual const Candera::Char* GetReadableName() const { return name; }

    /**
     *  Defines a description element in the meta information object
     */
    #define CdaDescription(descr) \
        virtual const Candera::Char* GetDescription() const { return descr; }

    /**
     *  Defines an editor for the current property. If an editor is not
     *  explicitly defined, the one from the DataType is taken.
     *  See PropertyMetaInfo::GetEditor for more information. This macro may
     *  be used only within a CdaProperty definition.
     *  @param editorString The editor string of this property.
     */
    #define CdaEditor(editorString) \
        _CdaCustomEditorInfo(editorString)

    /**
     *  Defines a category element in the meta information object
     */
    #define CdaCategory(category)    \
        virtual const Candera::Char* GetCategory() const { return category; }

    /**
     *  Defines the visibility test method or code.
     *  Whenever the test returns false, the property will be hidden from SceneComposer property list.
     */
    #define CdaVisibilityTest(visibilityTest)    \
        virtual PropertyVisibility GetPropertyVisibility(const BaseHostType* object) const { return ((object != 0) && (CdaCastOperator(object, (const CurrentType*)0)->visibilityTest())) ? VisibleProperty : InvisibleProperty; }
    #define CdaFieldVisibilityTest(visibilityTest)    \
        virtual PropertyVisibility GetPropertyVisibility(const BaseHostType* object) const { return ((object != 0) && (CdaCastOperator(object, (const CurrentType*)0)->visibilityTest)) ? VisibleProperty : InvisibleProperty; }

    /**
     *  Defines the update type of the property.
     *  The update type information is used by SceneComposer to correctly update the property.
     */
    #define CdaUpdateType(updateType)    \
        enum {c_updateType = updateType}; \
        virtual UpdateType GetUpdateType() const { return static_cast<UpdateType>(c_updateType); }

    /**
     *  Defines a value range for a property.
     *  Valid only in CdaProperty sections.
     *  @param rangeMin minimum value of property range
     *  @param rangeMax maximum value of property range
     */
    #define CdaRange(rangeMin, rangeMax) \
        virtual bool GetRangeMin(Candera::Char *buf, Candera::UInt bufLen) const { \
            return Candera::MetaInfo::Internal::DataType<TPropertyType>::ConvertToString(rangeMin, buf, bufLen); \
        } \
        virtual bool GetRangeMax(Candera::Char *buf, Candera::UInt bufLen) const { \
            return Candera::MetaInfo::Internal::DataType<TPropertyType>::ConvertToString(rangeMax, buf, bufLen); \
        }

    /**
     *  Makes a MetaInfoBase object deprecated.
     */
    #define CdaDeprecated() \
        virtual bool IsDeprecated() const { return true; }

#else
// expand macros to nothing
    #define CdaInvalidValue(message) ""
    #define CdaReadableName(name)
    #define CdaDescription(descr)
    #define CdaEditor(editorString)
    #define CdaCategory(cat)
    #define CdaRange(rangeMin, rangeMax)
    #define CdaVisibilityTest(visibilityTest)
    #define CdaFieldVisibilityTest(visibilityTest)
    #define CdaUpdateType(updateType) \
        enum {c_updateType = updateType};
    #define CdaDeprecated()
#endif
/// @}
#endif
