//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_PROPERTYMETAINFO_H)
    #define CANDERA_PROPERTYMETAINFO_H

#include <Candera/EngineBase/Animation/AnimationPropertySetter.h>
#include <Candera/System/MetaInfo/MetaInfoBase.h>

namespace Candera { namespace MetaInfo {
/// @addtogroup MetaInfo
/// @{

/**
 *  Extension point for properties.
 *  Specific properties can provide a capability object for enhanced property functionality.
 */
class PropertyCapability {
    public:
        /**
         *  Capability object type detection
         *  @return the type of capability object
         */
        virtual const Char* CapabilityName() const = 0;
};

/**
 *  Defines the meta information base class of a object property.
 */
class PropertyMetaInfoBase : public MetaInfoBase
{
    public:
        /**
         *  Constructs a property meta information object with the given name.
         *  @param name Name to assign to ObjectMetaInfo object.
         */
        PropertyMetaInfoBase(const Char *name) : MetaInfoBase(name) { }

        /**
         * Destructor.
         */
        virtual ~PropertyMetaInfoBase() {};

        /**
         * Specific properties may provide additional functionality through a capability object.
         * @return capability object or 0 (null) if no additional function is defined for the property
         */
        virtual const PropertyCapability* GetCapability() const {
            return 0;
        }

        /**
         * Value validity flag.
         *
         * This flag specifies if the current value of the property is a valid one.
         */
        enum ValueValidity {
            AlwaysValidValue,           ///< The value is always valid.
            ValidValue,                 ///< The current value is valid.
            InvalidValue                ///< The current value is not valid.
        };

        /**
         * Update type flag.
         *
         * This flag specifies if a certain action has to be executed to successfully update the property value.
         */
        enum UpdateType {
            ContinuousUpdateType,       ///< The property value is automatically updated.
            OnApplyUpdateType,          ///< The property value is updated only upon calling "ApplyChange" or equivalent method.
            OnUploadUpdateType          ///< The property value is updated only on uploading the object.
        };

        enum {c_updateType = ContinuousUpdateType};

#ifdef CANDERA_META_DESCRIPTION
        
        /**
         * Property visibility flag.
         *
         * This flag determines if the property should be visible or not in SceneComposer.
         */
        enum PropertyVisibility {
            AlwaysVisibleProperty,      ///< Property is always visible in SceneComposer.
            VisibleProperty,            ///< Property is currently visible in SceneComposer.
            InvisibleProperty           ///< Property is currently invisible in SceneComposer.
        };

        /**
         *  Return property value range information
         * This method must not be used outside scene composer context.
         *  @param buf character buffer to receive the property value
         *  @param bufLen size of buf
         *  @return false if no range information is defined or conversion to
         *          character buffer failed, true otherwise
         */
        virtual bool GetRangeMin(Char* buf, UInt bufLen) const { (void) buf; (void) bufLen; return false; } // scene composer only

        /**
         *  Return property value range information
         *  This method must not be used outside scene composer context.
         *  @param buf character buffer to receive the property value
         *  @param bufLen size of buf
         *  @return false if no range information is defined or conversion to
         *          character buffer failed, true otherwise
         */
        virtual bool GetRangeMax(Char* buf, UInt bufLen) const { (void) buf; (void) bufLen; return false; } // scene composer only

        /**
         *  Return property editor to be used in SceneComposer
         *  The DataType of the property defines the editor that shall be used for this
         *  property. This function returns either the name of a built-in editor or the name
         *  and class of the assembly that contains the editor for this property type. Scene
         *  composer will load the assembly dynamically and use the embedded editor.
         *  @return editor to be used
         */
        virtual const Char* GetEditor() const = 0;    // scene composer only

        /**
         * Get UpdateType flag.
         * This method must not be used outside scene composer context.
         * @return UpdateType flag.
         */
        virtual UpdateType GetUpdateType() const { return ContinuousUpdateType; }

        /**
         * Retrieve number of dimensions of this property.
         *
         * A dimension of 0 means that the property accepts only one value to be set.
         * A dimension of 1 means that the property accepts a custom number of 0 dimension values to be set.
         * A dimension of 2 means that the property accepts a custom number of 1 dimension values to be set.
         * ...
         * @return number of dimensions.
         * @see Candera::ArrayProperty
         * @see Candera::ArrayDataType
         */
        virtual UInt8 GetDimensionCount() const { return 0; }
#endif

};

/**
 * Defines the meta information object of a object property.
 *
 * @tparam HostType Type of property host, e.g. WidgetBase, GraphicDeviceUnit etc.
 */
template<typename HostType>
class PropertyMetaInfo : public PropertyMetaInfoBase
{
    public:
        /**
         *  Constructs a property meta information object with the given name.
         *  @param name Name to assign to ObjectMetaInfo object.
         */
        PropertyMetaInfo(const Char *name) : PropertyMetaInfoBase(name) { }

        /**
         * Destructor.
         */
        virtual ~PropertyMetaInfo() {}

        /**
         *  Property value getter.
         *  This method must not be used outside scene composer context.
         *  Gets the property value from the given object and transforms to the given
         *  string buffer. If the conversion fails, the string buffer is valid and might
         *  contain some useful information about the property value, but it cannot
         *  be sent back to the Set method.
         *  @param object object instance from which property shall be retrieved
         *  @param buf character buffer to hold the property value
         *  @param buf_size size of the character buffer
         *  @return true if conversion succeeded, false otherwise
         */
        virtual bool Get(const HostType* object, Char *buf, UInt buf_size) = 0;

        /**
         *  Property value setter
         *  This method must not be used outside scene composer context
         *  sets the property value in the given widget.
         *  @param object object instance in which property shall be set
         *  @param value character buffer holding the property value
         *  @return true if conversion succeeded, false otherwise
         */
        virtual bool Set(HostType* object, const Char *value) = 0;

        /**
         *  Create an AnimationProertySetter.
         *  
         *  The caller is responsible for the new created object lifetime.
         *  An AnimationPropertySetter is created, if the conversion from Float to the property type is
         *  supported.
         *  
         *  @param object object instance for which the property setter should be retrieved.
         *  @param dataSize size of Float array that will be used when calling the setter.
         *  @param channels dataSize sized array of channels to be set (e.g. only Alpha channel in a Color property).
         *  @return an AnimationPropertySetter for the property if supported, 0 otherwise.
         *  @see FloatConverter. 
         */
        virtual Animation::AnimationPropertySetter::SharedPointer CreateAnimationPropertySetter(HostType* object, Int dataSize, Int* channels) = 0;

        /**
         * Get ValueValidity flag.
         * @param object object instance for which the property value validity should be retrieved.
         * @param outMessage pointer to a string containing an output message in case of invalid value (available only if CANDERA_META_DESCRIPTION is enabled)
         * @return current ValueValidity flag.
         * 
         * @see PropertyMetaInfoBase::ValueValidity
         */
        PropertyMetaInfoBase::ValueValidity GetValueValidity(const HostType* object, const Char*& outMessage) const;

#ifdef CANDERA_META_DESCRIPTION
        /**
         * Get PropertyVisibility flag.
         * This method must not be used outside scene composer context.
         * @return current PropertyVisibility flag.
         * 
         * @see PropertyMetaInfoBase::PropertyVisibility
         */
        virtual PropertyMetaInfoBase::PropertyVisibility GetPropertyVisibility(const HostType* object) const; // scene composer only
#endif
    protected:
        static const Char* NotAnObjectMessage() { static const Char* nao = "Not an Object"; return nao; }
    private:
        virtual const Char* GetValueValidityMessage(const HostType* object) const;
        static const Char* AlwaysValidValueMessage() { static Char c; return &c; }
};

#ifdef CANDERA_META_DESCRIPTION
template<typename HostType>
inline PropertyMetaInfoBase::PropertyVisibility PropertyMetaInfo<HostType>::GetPropertyVisibility(const HostType* /*object*/) const
{ 
    return PropertyMetaInfoBase::AlwaysVisibleProperty; 
}
#endif

template<typename HostType>
PropertyMetaInfoBase::ValueValidity PropertyMetaInfo<HostType>::GetValueValidity(const HostType* object, const Char*& outMessage) const
{
    outMessage = GetValueValidityMessage(object);

    if (outMessage == AlwaysValidValueMessage()) {
        outMessage = 0;
        return AlwaysValidValue;
    }

    return (outMessage == 0) ? ValidValue : InvalidValue;
}

template<typename HostType>
inline const Char* PropertyMetaInfo<HostType>::GetValueValidityMessage(const HostType* /*object*/) const
{
    return AlwaysValidValueMessage();
}

/// @}
}}
#endif
