//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/System/MetaInfo/EnumDataTypeBase.h>
#include <Candera/System/MetaInfo/DataType.h>

/// @addtogroup MetaInfo
/// @{
/**
 *  \def ENUM_DATA_TYPE
 *  \file EnumDataType.h
 *
 *  This file may be used to generate an enumeration and a Candera::MetaInfo::DataType specialization with the given
 *  enumeration, that uses the default enumeration editor builtin://EnumEditor.
 *
 *  When using this file, \a ENUM_DATA_TYPE must be defined as a block beginning with \a ENUM_DATA_TYPE_BEGIN and ending
 *  with \a ENUM_DATA_TYPE_END, holding at least one item defined with \a ENUM_DATA_TYPE_ITEM or \a ENUM_DATA_TYPE_ITEM_VALUE.
 *  If an item has to be defined which is not visible in Scene Composer, \a ENUM_DATA_TYPE_ITEM_INVISIBLE or
 *  \a ENUM_DATA_TYPE_ITEM_VALUE_INVISIBLE has to be used.
 *
 *  \note
 *  <b>It is mandatory to use \a ENUM_DATA_TYPE_ITEM_VALUE after each invisible item!</b>
 *
 *  Example 1:
 *  \code
 *  #define ENUM_DATA_TYPE_NAMESPACE MyNamespace
 *  
 *  #define ENUM_DATA_TYPE \
 *      ENUM_DATA_TYPE_BEGIN(SomeEnum) \
 *          ENUM_DATA_TYPE_ITEM(SomeItem0) \
 *          ENUM_DATA_TYPE_ITEM_VALUE(SomeItem1, 4) \
 *          ENUM_DATA_TYPE_ITEM(SomeItem2) \
 *          ENUM_DATA_TYPE_ITEM_INVISIBLE(InvisibleItem) \
 *          ENUM_DATA_TYPE_ITEM_VALUE(SomeItem3, 7) \
 *      ENUM_DATA_TYPE_END(SomeEnum)
 *  
 *  #include <Candera/System/MetaInfo/EnumDataType.h>
 *  \endcode
 *  
 *  This file should be included it the default namespace.
 *  The enumeration is defined in the struct MyNamespace::SomeEnum. The enumeration type is MyNamespace::SomeEnum::Enum.
 *  Enumeration items can be accessed by e.g. MyNamespace::SomeEnum::SomeItem0.
 *  The DataType specialization is defined the the namespace Candera::MetaInfo.
 *
 *  Example 2 (no definition for ENUM_DATA_TYPE_NAMESPACE):
 *  \code
 *  #define ENUM_DATA_TYPE \
 *      ENUM_DATA_TYPE_BEGIN(SomeEnum) \
 *          ENUM_DATA_TYPE_ITEM(SomeItem0) \
 *          ENUM_DATA_TYPE_ITEM_VALUE(SomeItem1, 4) \
 *          ENUM_DATA_TYPE_ITEM(SomeItem2) \
 *          ENUM_DATA_TYPE_ITEM_INVISIBLE(InvisibleItem) \
 *          ENUM_DATA_TYPE_ITEM_VALUE(SomeItem3, 7) \
 *      ENUM_DATA_TYPE_END(SomeEnum)
 *
 *  #include <Candera/System/MetaInfo/EnumDataType.h>
 *  \endcode
 *
 *  This file should be included it the default namespace.
 *  The enumeration is defined in the namespace Candera.
 *  The DataType specialization is defined the the namespace Candera::MetaInfo.
 */

#ifndef ENUM_DATA_TYPE

    #error "ENUM_DATA_TYPE should be defined when using this file. See description in file header."

#else //defined ENUM_DATA_TYPE

    // handle namespace helper macros

    #if defined(ENUM_DATA_TYPE_NAMESPACE)
        #define ENUM_DATA_TYPE_IMPL_BEGIN(ENUM) \
            namespace ENUM_DATA_TYPE_NAMESPACE { \
                struct ENUM { \
                    enum Enum {

        #define ENUM_DATA_TYPE_IMPL_END() \
                    }; \
                }; \
            }

        #define ENUM_DATA_TYPE_ENUM_TYPE(ENUM) \
            ENUM_DATA_TYPE_NAMESPACE::ENUM::Enum
    #else
        #define ENUM_DATA_TYPE_IMPL_BEGIN(ENUM) \
            namespace Candera { \
                enum ENUM {

        #define ENUM_DATA_TYPE_IMPL_END() \
                }; \
            }

        #define ENUM_DATA_TYPE_ENUM_TYPE(ENUM) \
            Candera::ENUM
    #endif

    // define enumeration

    #define ENUM_DATA_TYPE_BEGIN(ENUM) \
        ENUM_DATA_TYPE_IMPL_BEGIN(ENUM)

    #define ENUM_DATA_TYPE_ITEM(ITEM)                           ITEM,

    #define ENUM_DATA_TYPE_ITEM_VALUE(ITEM, VALUE)              ITEM = VALUE,

    #define ENUM_DATA_TYPE_ITEM_INVISIBLE(ITEM)                 ENUM_DATA_TYPE_ITEM(ITEM)

    #define ENUM_DATA_TYPE_ITEM_VALUE_INVISIBLE(ITEM, VALUE)    ENUM_DATA_TYPE_ITEM_VALUE(ITEM, VALUE)

    #define ENUM_DATA_TYPE_END(ENUM) \
            _ ##ENUM##_LAST_ITEM /* to avoid compile error 'trailing comma is nonstandard' */ \
        ENUM_DATA_TYPE_IMPL_END()

    ENUM_DATA_TYPE

    #undef ENUM_DATA_TYPE_BEGIN
    #undef ENUM_DATA_TYPE_ITEM
    #undef ENUM_DATA_TYPE_ITEM_VALUE
    #undef ENUM_DATA_TYPE_ITEM_INVISIBLE
    #undef ENUM_DATA_TYPE_ITEM_VALUE_INVISIBLE
    #undef ENUM_DATA_TYPE_END

    // define data type for enumeration

    #define ENUM_DATA_TYPE_BEGIN(ENUM)                                          \
        namespace Candera {                                                     \
            namespace MetaInfo {                                                \
                                                                                \
                inline const Char * FEATSTD_CONCAT3(Get, ENUM, Editor)() {      \
                    return ENUM_DATA_TYPE_EDITOR_BEGIN(ENUM)                    \

    #define ENUM_DATA_TYPE_ITEM(ITEM) \
                        ENUM_DATA_TYPE_EDITOR_ITEM(ITEM)
    #define ENUM_DATA_TYPE_ITEM_VALUE(ITEM, VALUE) \
                        ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(ITEM, VALUE)

    #define ENUM_DATA_TYPE_ITEM_INVISIBLE(ITEM)
    #define ENUM_DATA_TYPE_ITEM_VALUE_INVISIBLE(ITEM, VALUE)

    #define ENUM_DATA_TYPE_END(ENUM) \
                    ENUM_DATA_TYPE_EDITOR_END();                                \
                }                                                               \
                ENUM_DATA_TYPE_DEF(ENUM_DATA_TYPE_ENUM_TYPE(ENUM), &FEATSTD_CONCAT3(Get, ENUM, Editor));  \
            } \
        }

    ENUM_DATA_TYPE

    #undef ENUM_DATA_TYPE_NAMESPACE
    #undef ENUM_DATA_TYPE_IMPL_BEGIN
    #undef ENUM_DATA_TYPE_IMPL_END
    #undef ENUM_DATA_TYPE_ENUM_TYPE

    #undef ENUM_DATA_TYPE_BEGIN
    #undef ENUM_DATA_TYPE_ITEM
    #undef ENUM_DATA_TYPE_ITEM_VALUE
    #undef ENUM_DATA_TYPE_ITEM_INVISIBLE
    #undef ENUM_DATA_TYPE_ITEM_VALUE_INVISIBLE
    #undef ENUM_DATA_TYPE_END

    #undef ENUM_DATA_TYPE
#endif //ENUM_DATA_TYPE


/// @}
