//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DATATYPE_H)
#define CANDERA_DATATYPE_H

#include <Candera/Environment.h>
#include <CanderaPlatform/OS/SerializationPlatform.h>
#include <FeatStd/Util/FixedSizeString.h>
#include <FeatStd/Util/Traits.h>

namespace Candera { namespace MetaInfo {
/// @addtogroup DataTypesSystem
/// @{

template<typename T> struct DataTypeDimensions{ enum { Count = 0 }; };

struct DataTypeBase {
    enum { MinimumBufSize = 128 }; ///< 128, minimum buffer size in bytes
};

/**
    *  DataType is used for converting the generic type from and to various representations:
    *  - conversion from / to string presentation (only needed for scene composer, not needed for target device)
    *  - conversion from AnimationPropertyValue
    *  - conversion from serialized data in a byte stream (asset data initialization vectors)
    *
    *  The default template implements none of the conversions above. For specific data types, template
    *  specializations must be provided.
    *  To simplify overall string conversion handling, the minimum size of the target string buffer is defined
    *  by MinimumBufSize. The callee of the conversion functions must guarantee, that the buffers supplied have
    *  at least MinimumBufSize bytes.
    */
FEATSTD_LINT_SYMBOL(1577, Candera::MetaInfo::DataType<*>, "DataType can be specialized for application-defined types")
template<class T> struct DataType : public DataTypeBase
{
    //static bool ConvertToString(const T &, Char*, UInt) { return false; }
    //static bool ConvertFromString(T&, const Char*) { return false; }
    //static bool ConvertFrom(T&, const AnimationPropertyValue&) { return false; }
    //static bool ConvertFrom(T&, ByteStream&) { return false; }
    //static const Char* GetEditor();                                                    // scene composer only
};

namespace Internal {
    FEATSTD_LINT_SYMBOL(1577, Candera::MetaInfo::Internal::DataType<*>, "DataType can be specialized for application-defined types")
    template<typename T> struct DataType {
        static inline bool ConvertToString(T val, Char *buf, UInt size) {
            return Candera::MetaInfo::DataType<T>::ConvertToString(val, buf, size);
        }

        static inline bool ConvertFromString(T &val, const Char *buf) {
            return Candera::MetaInfo::DataType<T>::ConvertFromString(val, buf);
        }

        static inline const Char* GetEditor() {
            return Candera::MetaInfo::DataType<T>::GetEditor();
        }
    };

    template<typename T, typename C> struct ScalarCastType {
        static inline T Cast(C v) { return static_cast<T>(v); }
    };

    template<typename C> struct ScalarCastType<bool, C> {
        static inline bool Cast(C v) { return v != C(); }
    };

    template<typename T> struct ScalarEditorType;

    template<> struct ScalarEditorType<FeatStd::Internal::UIntSpecializationReplacement> {
        static inline const Char* Get() { return "builtin://UIntEditor"; }
    };

    template<> struct ScalarEditorType<UInt64> {
        static inline const Char* Get() { return "builtin://ULongEditor"; }
    };

    template<> struct ScalarEditorType<UInt32> {
        static inline const Char* Get() { return "builtin://UIntEditor"; }
    };

    template<> struct ScalarEditorType<UInt16> {
        static inline const Char* Get() { return "builtin://UShortEditor"; }
    };

    template<> struct ScalarEditorType<UInt8> {
        static inline const Char* Get() { return "builtin://ByteEditor"; }
    };

    template<> struct ScalarEditorType<FeatStd::Internal::IntSpecializationReplacement> {
        static inline const Char* Get() { return "builtin://IntEditor"; }
    };

    template<> struct ScalarEditorType<Int64> {
        static inline const Char* Get() { return "builtin://LongEditor"; }
    };
    
    template<> struct ScalarEditorType<Int32> {
        static inline const Char* Get() { return "builtin://IntEditor"; }
    };

    template<> struct ScalarEditorType<Int16> {
        static inline const Char* Get() { return "builtin://ShortEditor"; }
    };

    template<> struct ScalarEditorType<Int8> {
        static inline const Char* Get() { return "builtin://SByteEditor"; }
    };

    template<> struct ScalarEditorType<Float> {
        static inline const Char* Get() { return "builtin://FloatEditor"; }
    };

    template<> struct ScalarEditorType<bool> {
        static inline const Char* Get() { return "builtin://BoolEditor"; }
    };
}    // namespace Internal

/**
    *  DataType conversion functions for scalar types.
    */
template<typename T, typename C> struct ScalarDataType {
    static inline bool ConvertToString(T val, Char *buf, UInt size) {
        return SerializationPlatform::FromBinary(buf, size, static_cast<C>(val)) == 1;
    }

    static inline bool ConvertFromString(T &val, const Char *buf) {
        C v;
        int n = SerializationPlatform::ToBinary(buf, v);
        val = Internal::ScalarCastType<T, C>::Cast(v);
        return n == 1;
    }

    static inline const Char* GetEditor() {
        return Internal::ScalarEditorType<T>::Get();
    }
};

/**
    *  @brief DataType specialization for Int8.
    */
template<> struct DataType<Int8> : public ScalarDataType<Int8, Int> { };

/**
    *  @brief DataType specialization for Int16.
    */
template<> struct DataType<Int16> : public ScalarDataType<Int16, Int> { };

/**
    *  @brief DataType specialization for Int32.
    */
template<> struct DataType<Int32> : public ScalarDataType<Int32, Int> { };

/**
    *  @brief DataType specialization for Int64.
    */
template<> struct DataType<Int64> : public ScalarDataType<Int64, Int> { };

/**
    *  @brief DataType specialization for UInt8.
    */
template<> struct DataType<UInt8> : public ScalarDataType<UInt8, UInt> { };

/**
    *  @brief DataType specialization for UInt16.
    */
template<> struct DataType<UInt16> : public ScalarDataType<UInt16, UInt> { };

/**
    *  @brief DataType specialization for UInt32.
    */
template<> struct DataType<UInt32> : public ScalarDataType<UInt32, UInt> { };

/**
    *  @brief DataType specialization for UInt32.
    */
template<> struct DataType<UInt64> : public ScalarDataType<UInt64, UInt> { };

/**
    *  @brief DataType specialization for Int.
    */
template<> struct DataType<FeatStd::Internal::IntSpecializationReplacement> : public ScalarDataType<Int, Int> { };

/**
    *  @brief DataType specialization for UInt.
    */
template<> struct DataType<FeatStd::Internal::UIntSpecializationReplacement> : public ScalarDataType<UInt, UInt> { };

/**
*  @brief DataType specialization for SizeType.
*/
template<> struct DataType<FeatStd::Internal::SizeTypeSpecializationReplacement> : public ScalarDataType<SizeType, UInt>{};

/**
    *  @brief DataType specialization for bool.
    */
template<> struct DataType<bool> : public ScalarDataType<bool, Int> { };

/**
    *  @brief DataType specialization for Float.
    */
template<> struct DataType<Float> : public ScalarDataType<Float, Float> { };

/**
    *  @brief DataType specialization for const Char*.
    */
template<> struct DataType<const Char*> {
    /**
        *  Copies the string into the given buffer.
        *
        *  @param val string to copy.
        *  @param buf Buffer to write the string.
        *  @param bufLen Buffer length.
        *  @return true if conversion successful, false otherwise.
        */
    static bool ConvertToString(const Char *val, Char *buf, UInt bufLen) {
        bool ok = true;
        if (val != 0) {
            ok = (StringPlatform::Length(val) + 1) < bufLen;
            if (ok) {
                StringPlatform::Copy(buf, val);
            }
        } else {
            ok = false;
            buf[0] = '\0';
        }
        return ok;
    }

    /**
        *  Return the given string.
        *
        *  @param val returned string.
        *  @param buf Buffer that contains the initial string.
        *  @return true.
        */
    static bool ConvertFromString(const Char *&val, const Char *buf) {
        val = buf;
        return true;
    }

    /**
        *  Get the PropertyEditor name as known by SceneComposer.
        *
        *  @return string containing the SceneComposer property editor name.
        */
    static const Char* GetEditor() { return "builtin://CharArrayEditor"; }
};

/**
    *  @brief DataType specialization for const FeatStd::Internal::FixedSizeString<>.
    */
template<UInt32 N> struct DataType< FeatStd::Internal::FixedSizeString<N> > {
    typedef FeatStd::Internal::FixedSizeString<N> Type;

    /**
        *  Copies the string into the given buffer.
        *
        *  @param val string to copy.
        *  @param buf Buffer to write the string.
        *  @param bufLen Buffer length.
        *  @return true if conversion successful, false otherwise.
        */
    static bool ConvertTo(const Type &val, Char *buf, UInt bufSize) {
        const TChar *str = val.CStr();
        bool ok = UInt(StringPlatform::Length(str)) < bufSize;
        if (ok) {
            StringPlatform::Copy(buf, str);
        }
        return ok;
    }

    /**
        *  Return the given string.
        *
        *  @param val returned string.
        *  @param buf Buffer that contains the initial string.
        *  @return true.
        */
    static bool ConvertFrom(Type &val, const Char *buf) {
        val = buf;
        return true;
    }

    /**
        *  Get the PropertyEditor name as known by SceneComposer.
        *
        *  @return string containing the SceneComposer property editor name.
        */
    static const Char* GetEditor() {
        return DataType<const Char*>::GetEditor();
    }
};
/// @}
}}
#endif    // CANDERA_DATATYPE_H
