//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VECTOR4_H)
#define CANDERA_VECTOR4_H

#include <Candera/Environment.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Math.h>

namespace Candera {
/** @addtogroup MathematicsSystem
 *  @{
 */

class Vector3;
class Matrix4;

/**
 * @brief The default Vector4 class.
 */
class Vector4
{
    FEATSTD_LOG_SET_REALM(Candera::Diagnostics::LogRealm::CanderaSystem);

    public:
        /**
         *  The data of this vector.
         */
        struct Data
        {
            Float x;
            Float y;
            Float z;
            Float w;
        };

        /**
         *  Constructor
         */
        Vector4();

        /**
         *  Destructor
         */
        ~Vector4() {} // non-virtual

        /**
         *  CopyConstructor from another vector.
         *  @param vector The other vector.
         */
        Vector4(const Vector4& vector);

        /**
         *  CopyConstructor from another vector and the fourth value.
         *  @param vector The other vector.
         *  @param wValue The w-value default = 1.0F.
         */
        explicit Vector4(const Vector3& vector, Float wValue  = 1.0F);

        /**
         *  Constructor with given parameters.
         *  @param xValue The x-value of the vector.
         *  @param yValue The y-value of the vector.
         *  @param zValue The z-value of the vector.
         *  @param wValue The w-value of the vector default = 1.0F.
         */
        Vector4(Float xValue, Float yValue, Float zValue, Float wValue = 1.0F);

        /**
         *  Calculates the Length of the vector.
         *  @return         The length of the vector.
         */
        Float GetLength() const;

        /**
         *  Calculates the cross product of this vector with two another vectors.
         *  @param  vector_a  The second vector.
         *  @param  vector_b  The third vector.
         *  @return           A Vector4 that contains the cross product between this vector and vector_a and vector_b.
         */
        Vector4 GetCrossProduct(const Vector4& vector_a, const Vector4& vector_b) const;

        /**
         *  Calculates the dot product of this Vector4 with another Vector4.
         *  @param  vector  The other vector.
         *  @return         The dot product of this Vector4 with another Vector4.
         */
        Float GetDotProduct(const Vector4& vector) const;

        /**
         *  Transforms the vector by the given Matrix.
         *  @param  matrix  The Matrix to transform with.
         */
        void TransformCoordinate(const Matrix4& matrix);

        /**
         *  Retrieves the x-value of this vector.
         *  @return the x-value of this vector.
         */
        Float GetX() const { return m_data.x; }

        /**
         *  Retrieves the y-value of this vector.
         *  @return the y-value of this vector.
         */
        Float GetY() const { return m_data.y; }

        /**
         *  Retrieves the z-value of this vector.
         *  @return the z-value of this vector.
         */
        Float GetZ() const { return m_data.z; }

        /**
         *  Retrieves the w-value of this vector.
         *  @return the w-value of this vector.
         */
        Float GetW() const { return m_data.w; }

        /**
         *  Sets the x-value of this vector.
         *  @param xValue The value that is set.
         */
        void SetX(Float xValue) { m_data.x = xValue; }

        /**
         *  Sets the y-value of this vector.
         *  @param yValue The value that is set.
         */
        void SetY(Float yValue) { m_data.y = yValue; }

        /**
         *  Sets the z-value of this vector.
         *  @param zValue The value that is set.
         */
        void SetZ(Float zValue) { m_data.z = zValue; }

        /**
         *  Sets the w-value of this vector.
         *  @param wValue The value that is set.
         */
        void SetW(Float wValue) { m_data.w = wValue; }

        Vector4 operator + (const Vector4& rhs) const;
        Vector4 operator - (const Vector4& rhs) const;
        Vector4 operator * (const Vector4& rhs) const;

        Vector4& operator = (const Vector4& rhs);
        Vector4& operator += (const Vector4& rhs);
        Vector4& operator -= (const Vector4& rhs);
        Vector4& operator *= (const Vector4& rhs);

        bool operator == (const Vector4& rhs) const;
        bool operator != (const Vector4& rhs) const;

        Vector4& operator = (const Float f);
        Vector4& operator *= (const Float f);
        Vector4 operator * (const Float f) const;
        Vector4& operator /= (const Float f);
        Vector4 operator / (const Float f) const;

        /**
         *  Return the negative of this Vector4
         *  @return The negative of this Vector4
         */
        Vector4 operator - () const;
        const Vector4& operator + () const;

        /**
         *  Provides access to the properties by specifying an index.
         *  Index 0 returns x and accordingly.
         *  @param index The index of the desired element.
         *  @return The desired element.
         */
        Float& operator [] (UInt8 index);

        /**
         *  Provides access to the properties by specifying an index.
         *  Index 0 returns x and accordingly.
         *  @param index The index of the desired element.
         *  @return The desired element.
         */
        Float operator [] (UInt8 index) const;

        /**
         *  Retrieves the data of this vector.
         *  @return The data of this vector.
         */
        const Data* GetData() const { return &m_data; }

    private:
        Vector4::Data m_data;
    };

    /** @} */ // end of MathematicsSystem

    //Inline implementations of Vector4
    inline Float Vector4::GetDotProduct(const Vector4& vector) const
    {
        return m_data.x * vector.m_data.x + m_data.y * vector.m_data.y + m_data.z * vector.m_data.z + m_data.w * vector.m_data.w;
    }

    inline Vector4 Vector4::operator + (const Vector4& rhs) const
    {
        return Vector4(
            m_data.x + rhs.m_data.x,
            m_data.y + rhs.m_data.y,
            m_data.z + rhs.m_data.z,
            m_data.w + rhs.m_data.w);
    }

    inline Vector4 Vector4::operator - (const Vector4& rhs) const
    {
        return Vector4(
            m_data.x - rhs.m_data.x,
            m_data.y - rhs.m_data.y,
            m_data.z - rhs.m_data.z,
            m_data.w - rhs.m_data.w);
    }

    inline Vector4 Vector4::operator * (const Vector4& rhs) const
    {
        return Vector4(
            m_data.x * rhs.m_data.x,
            m_data.y * rhs.m_data.y,
            m_data.z * rhs.m_data.z,
            m_data.w * rhs.m_data.w);
    }

    inline Vector4& Vector4::operator = (const Vector4& rhs)
    {
        //no check for self assignment due to performance reasons
        m_data.x = rhs.m_data.x;
        m_data.y = rhs.m_data.y;
        m_data.z = rhs.m_data.z;
        m_data.w = rhs.m_data.w;
        return *this;
    }

    inline Vector4& Vector4::operator += (const Vector4& rhs)
    {
        m_data.x += rhs.m_data.x;
        m_data.y += rhs.m_data.y;
        m_data.z += rhs.m_data.z;
        m_data.w += rhs.m_data.w;
        return *this;
    }

    inline Vector4& Vector4::operator -= (const Vector4& rhs)
    {
        m_data.x -= rhs.m_data.x;
        m_data.y -= rhs.m_data.y;
        m_data.z -= rhs.m_data.z;
        m_data.w -= rhs.m_data.w;
        return *this;
    }

    inline Vector4& Vector4::operator *= (const Vector4& rhs)
    {
        m_data.x *= rhs.m_data.x;
        m_data.y *= rhs.m_data.y;
        m_data.z *= rhs.m_data.z;
        m_data.w *= rhs.m_data.w;
        return *this;
    }

    inline bool Vector4::operator == (const Vector4& rhs) const
    {
        return (Math::FloatAlmostEqual(m_data.x, rhs.m_data.x) &&
                Math::FloatAlmostEqual(m_data.y, rhs.m_data.y) &&
                Math::FloatAlmostEqual(m_data.z, rhs.m_data.z) &&
                Math::FloatAlmostEqual(m_data.w, rhs.m_data.w));
    }

    inline bool Vector4::operator != (const Vector4& rhs) const
    {
        return !(*this == rhs);
    }

    inline Vector4& Vector4::operator = (const Float f)
    {
        m_data.x = f;
        m_data.y = f;
        m_data.z = f;
        m_data.w = f;
        return *this;
    }

    inline Vector4& Vector4::operator *= (const Float f)
    {
        m_data.x *= f;
        m_data.y *= f;
        m_data.z *= f;
        m_data.w *= f;
        return *this;
    }

    inline Vector4 Vector4::operator * (const Float f) const
    {
        return Vector4(
            m_data.x * f,
            m_data.y * f,
            m_data.z * f,
            m_data.w * f);
    }

    inline Vector4& Vector4::operator /= (const Float f)
    {
        if (f == 0.0F) {
            FEATSTD_LOG_ERROR("Vector4 operator, division by zero.");
            return *this;
        }

        //Inverse computation: 1 division + 4 multiplications is faster then 4 divisions
        Float inv = 1.0F / f;

        m_data.x *= inv;
        m_data.y *= inv;
        m_data.z *= inv;
        m_data.w *= inv;

        return *this;
    }

    inline Vector4 Vector4::operator / (const Float f) const
    {
        if (f == 0.0F) {
            FEATSTD_LOG_ERROR("Vector4 operator, division by zero.");
            return *this;
        }

        //Inverse computation: 1 division + 4 multiplications is faster then 4 divisions
        Float inv = 1.0F / f;

        return Vector4(
            m_data.x * inv,
            m_data.y * inv,
            m_data.z * inv,
            m_data.w * inv);
    }

    inline Vector4 Vector4::operator - () const
    {
        return Vector4(-m_data.x, -m_data.y, -m_data.z, -m_data.w);
    }

    inline const Vector4& Vector4::operator + () const
    {
        return *this;
    }

    inline Float& Vector4::operator [] (UInt8 index)
    {
        switch (index) {
            FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 6-5-3: all paths return")
            case 0:
                return m_data.x;
            case 1:
                return m_data.y;
            case 2:
                return m_data.z;
            case 3:
                return m_data.w;
            default:
                FEATSTD_DEBUG_FAIL();
                return m_data.x;
        }
    }

    inline Float Vector4::operator [] (UInt8 index) const
    {
        switch (index) {
            FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 6-5-3: all paths return")
            case 0:
                return m_data.x;
            case 1:
                return m_data.y;
            case 2:
                return m_data.z;
            case 3:
                return m_data.w;
            default:
                FEATSTD_DEBUG_FAIL();
                return 0.0F;
        }
    }
} // namespace Candera

#endif    // CANDERA_VECTOR4_H
