//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Vector3.h"
#include <Candera/System/Mathematics/Vector4.h>
#include <Candera/System/Mathematics/Matrix4.h>

namespace Candera {

Vector3::Vector3(const Vector4& vector)
{
    m_data.x = vector.GetX();
    m_data.y = vector.GetY();
    m_data.z = vector.GetZ();
}

Float Vector3::GetSquaredLength() const
{
    return (m_data.x * m_data.x) + (m_data.y * m_data.y) + (m_data.z * m_data.z);
}

Float Vector3::GetDistanceTo(const Vector3& vector) const
{
    return (*this - vector).GetLength();
}

Float Vector3::GetSquaredDistanceTo(const Vector3& vector) const
{
    return (*this - vector).GetSquaredLength();
}

Float Vector3::GetDotProduct(const Vector3& vector) const
{
    return (m_data.x * vector.m_data.x) + (m_data.y * vector.m_data.y) + (m_data.z * vector.m_data.z);
}

Float Vector3::Normalize()
{
    if((m_data.x == 0.0F) && (m_data.y == 0.0F) && (m_data.z == 0.0F))
    {
        // Catch exceptional case when vector has no direction, and thus cannot be normalized.
        // Return length of zero to guarantee a robust handling.
        return 0.0F;
    }

    Float length = GetLength();

    //Inverse computation: 1 division + 3 multiplications is faster then 3 divisions
    Float inverse_length = 1.0F / length;
    m_data.x *= inverse_length;
    m_data.y *= inverse_length;
    m_data.z *= inverse_length;

    return length;
}

Vector3 Vector3::GetCrossProduct(const Vector3& vector) const
{
    return Vector3((m_data.y * vector.m_data.z) - (m_data.z * vector.m_data.y),
                   (m_data.z * vector.m_data.x) - (m_data.x * vector.m_data.z),
                   (m_data.x * vector.m_data.y) - (m_data.y * vector.m_data.x));
}

void Vector3::TransformCoordinate(const Matrix4& matrix4)
{
    Float w = (matrix4.Get(0, 3) * m_data.x) + (matrix4.Get(1, 3) * m_data.y) + (matrix4.Get(2, 3) * m_data.z) + matrix4.Get(3, 3);

    if (w != 0.0F) {
        Vector3 vector(*this);
        m_data.x = ((matrix4.Get(0, 0) * vector.m_data.x) + (matrix4.Get(1, 0) * vector.m_data.y) + (matrix4.Get(2, 0) * vector.m_data.z) + matrix4.Get(3, 0)) / w;
        m_data.y = ((matrix4.Get(0, 1) * vector.m_data.x) + (matrix4.Get(1, 1) * vector.m_data.y) + (matrix4.Get(2, 1) * vector.m_data.z) + matrix4.Get(3, 1)) / w;
        m_data.z = ((matrix4.Get(0, 2) * vector.m_data.x) + (matrix4.Get(1, 2) * vector.m_data.y) + (matrix4.Get(2, 2) * vector.m_data.z) + matrix4.Get(3, 2)) / w;
    }
    else {
        m_data.x = 0.0F;
        m_data.y = 0.0F;
        m_data.z = 0.0F;
    }
}

void Vector3::SetZero()
{
    m_data.x = 0.0F;
    m_data.y = 0.0F;
    m_data.z = 0.0F;
}

} // namespace Candera


