//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VECTOR2_H)
    #define CANDERA_VECTOR2_H

#include <Candera/Environment.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/MathStringBufferAppenders.h>

namespace Candera {
/** @addtogroup MathematicsSystem
 *  @{
 */

/**
 * @brief The default Vector2 class.
 */
class Vector2
{
    FEATSTD_LOG_SET_REALM(Candera::Diagnostics::LogRealm::CanderaSystem);

    public:
        /**
         *  The data of this vector.
         */
        struct Data
        {
            Data() : x(0.F), y(0.F) {}
            Data(Float ix, Float iy) : x(ix), y(iy) {}

            Float x;
            Float y;
        };

        /**
         *  Constructor
         */
        Vector2() {};

        /**
         *  CopyConstructor from another vector.
         *  @param vector The other vector.
         */
        Vector2(const Vector2& vector) : m_data(vector.m_data) {}

        /**
         *  Constructor with given parameters.
         *  @param xValue The x-value of the vector.
         *  @param yValue The y-value of the vector.
         */
        Vector2(Float xValue, Float yValue) : m_data(xValue, yValue) {};

        /**
         *  Destructor
         */
        ~Vector2() {}

        /**
         *  Calculates the Length of the vector.
         *  @see Vector2::GetSquaredLength()
         *  @return         The length of the vector.
         */
        Float GetLength() const {return Math::SquareRoot(GetSquaredLength());}

        /**
         *  Calculates the squared Length of the vector.
         *  This method is faster then Vector2::GetLength(),
         *  and can be used e.g. for comparing the length of 2 Vectors.
         *  @see Vector2::GetLength()
         *  @return         The squared length of the vector.
         */
        Float GetSquaredLength() const;

        /**
         *  Calculates the distance to another vector.
         *  @see Vector2::GetSquaredDistanceTo()
         *  @param  vector  The target vector.
         *  @return         The distance to the passed vector.
         */
        Float GetDistanceTo(const Vector2& vector) const;

        /**
         *  Calculates the squared distance to another vector.
         *  This method is faster then Vector2::GetDistanceTo().
         *  Use this if you want to compare distances between vectors
         *  without incurring the square root.
         *  @see Vector2::GetDistanceTo()
         *  @param  vector  The target vector.
         *  @return         The squared distance to the passed vector.
         */
        Float GetSquaredDistanceTo(const Vector2& vector) const;

        /**
         *  Calculates the scalar dot product of this vector with another vector.
         *  @remarks        Can be used to calculate the angle between two vectors.
         *  @param  vector  The other vector.
         *  @return         The dot product value.
         */
        Float GetDotProduct(const Vector2& vector) const;

        /**
         *  Calculates the cross product of this vector with another vector.
         *  @remarks        The cross product operation between two vectors results
         *                  in a third vector perpendicular to the two input vectors.
         *  @param  vector  The other vector.
         *  @return         A vector perpendicular to this vector and the passed vector.
         */
        Vector2 GetCrossProduct(const Vector2& vector) const;

        /**
         *  Normalizes the vector to a length of a 1.0F approximation.
         *  @return         The length of the vector before the normalization.
         */
        Float Normalize();

        /**
         *  Sets all values of the Vector2 to 0.
         */
        void SetZero();

        /**
         *  Return whether this Vector2 is equal to the given Vector2.
         *  NOTE: operator== uses Math::FloatAlmostEqual, meaning '==' does not test for equality.
         *  @param rhs  The Vector2 to test for equality.
         *  @return     True, if the given Vector2 is equal to this Vector2. False, otherwise.
         */
        bool IsEqual(const Vector2& rhs) const {
            FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
            return ((m_data.x == rhs.m_data.x) && (m_data.y == rhs.m_data.y));
        }

        Vector2 operator + (const Vector2& rhs) const;
        Vector2 operator - (const Vector2& rhs) const;
        Vector2 operator * (const Vector2& rhs) const;
        Vector2 operator / (const Vector2& rhs) const;

        Vector2& operator = (const Vector2& rhs);
        Vector2& operator += (const Vector2& rhs);
        Vector2& operator -= (const Vector2& rhs);
        Vector2& operator *= (const Vector2& rhs);
        Vector2& operator /= (const Vector2& rhs);

        bool operator == (const Vector2& rhs) const;
        bool operator != (const Vector2& rhs) const;

        Vector2& operator = (const Float f);
        Vector2& operator *= (const Float f);
        Vector2 operator * (const Float f) const;
        Vector2 operator /= (const Float f);
        Vector2 operator / (const Float f) const;

        /**
         *  Return the negative of this Vector2
         *  @return The negative of this Vector2
         */
        Vector2 operator - () const;
        const Vector2& operator + () const;

        /**
         *  Provides access to the properties by specifying an index.
         *  Index 0 returns x and accordingly.
         *  @param index The index of the desired element.
         *  @return The desired Element.
         */
        Float& operator [] (UInt8 index);
        /**
         *  Provides access to the properties by specifying an index.
         *  Index 0 returns x and accordingly.
         *  @param index The index of the desired element.
         *  @return The desired Element.
         */
        Float operator [] (UInt8 index) const;

        /**
         *  Retrieves the x-value of this vector.
         *  @return the x-value of this vector.
         */
        Float GetX() const { return m_data.x; }

        /**
         *  Retrieves the y-value of this vector.
         *  @return the y-value of this vector.
         */
        Float GetY() const { return m_data.y; }

        /**
         *  Sets the x-value of this vector.
         *  @param xValue The value that is set.
         */
        void SetX(Float xValue) { m_data.x = xValue; }

        /**
         *  Sets the y-value of this vector.
         *  @param yValue The value that is set.
         */
        void SetY(Float yValue) { m_data.y = yValue; }

        /**
         *  Retrieves the data of this vector.
         *  @return The data of this vector.
         */
        const Data* GetData() const { return &m_data; }

    private:
        Vector2::Data m_data;
};

 /** @} */ // end of MathematicsSystem

inline Vector2& Vector2::operator += (const Vector2& rhs)
{
    m_data.x += rhs.m_data.x;
    m_data.y += rhs.m_data.y;
    return *this;
}

inline Vector2& Vector2::operator -= (const Vector2& rhs)
{
    m_data.x -= rhs.m_data.x;
    m_data.y -= rhs.m_data.y;
    return *this;
}

inline Vector2& Vector2::operator *= (const Vector2& rhs)
{
    m_data.x *= rhs.m_data.x;
    m_data.y *= rhs.m_data.y;
    return *this;
}

inline Vector2& Vector2::operator /= (const Vector2& rhs)
{
    m_data.x /= rhs.m_data.x;
    m_data.y /= rhs.m_data.y;
    return *this;
}

inline Vector2 Vector2::operator + (const Vector2& rhs) const
{
    return Vector2(
        m_data.x + rhs.m_data.x,
        m_data.y + rhs.m_data.y);
}

inline Vector2 Vector2::operator - (const Vector2& rhs) const
{
    return Vector2(
        m_data.x - rhs.m_data.x,
        m_data.y - rhs.m_data.y);
}

inline Vector2 Vector2::operator * (const Vector2& rhs) const
{
    return Vector2(
        m_data.x * rhs.m_data.x,
        m_data.y * rhs.m_data.y);
}

inline Vector2 Vector2::operator / (const Vector2& rhs) const
{
    return Vector2(
        m_data.x / rhs.m_data.x,
        m_data.y / rhs.m_data.y);
}

inline Vector2& Vector2::operator = (const Vector2& rhs)
{
    //no check for self assignment due to performance reasons
    m_data.x = rhs.m_data.x;
    m_data.y = rhs.m_data.y;
    return *this;
}

inline bool Vector2::operator == (const Vector2& rhs) const
{
    return Math::FloatAlmostEqual(m_data.x, rhs.m_data.x) && Math::FloatAlmostEqual(m_data.y, rhs.m_data.y);
}

inline bool Vector2::operator != (const Vector2& rhs) const
{
    return !(*this == rhs);
}

inline Vector2& Vector2::operator = (const Float f)
{
    m_data.x = f;
    m_data.y = f;
    return *this;
}

inline Vector2& Vector2::operator *= (const Float f)
{
    m_data.x *= f;
    m_data.y *= f;
    return *this;
}

inline Vector2 Vector2::operator * (const Float f) const
{
    return Vector2(
        m_data.x * f,
        m_data.y * f);
}

inline Vector2 Vector2::operator /= (const Float f)
{
    if (f == 0.0F) {
        FEATSTD_LOG_ERROR("Vector2 operator, division by zero.");
        return *this;
    }

    //Inverse computation: 1 division + 2 multiplications is faster then 2 divisions
    Float inv = 1.0F / f;

    m_data.x *= inv;
    m_data.y *= inv;

    return *this;
}

inline Vector2 Vector2::operator / (const Float f) const
{
    if (f == 0.0F) {
        FEATSTD_LOG_ERROR("Vector2 operator, division by zero.");
        return *this;
    }

    //Inverse computation: 1 division + 2 multiplications is faster then 2 divisions
    Float inv = 1.0F / f;

    return Vector2(
        m_data.x * inv,
        m_data.y * inv);
}

inline Vector2 Vector2::operator - () const
{
    return Vector2(-m_data.x, -m_data.y);
}

inline const Vector2& Vector2::operator + () const
{
    return *this;
}

inline Float& Vector2::operator [] (UInt8 index)
{
    switch (index) {
        FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 6-5-3: all paths return")
        case 0:
            return m_data.x;
        case 1:
            return m_data.y;
        default:
            FEATSTD_DEBUG_FAIL();
            return m_data.x;
    }
}

inline Float Vector2::operator [] (UInt8 index) const
{
    switch (index) {
        FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 6-5-3: all paths return")
        case 0:
            return m_data.x;
        case 1:
            return m_data.y;
        default:
            FEATSTD_DEBUG_FAIL();
            return 0.0F;
    }
}

inline void Vector2::SetZero()
{
    m_data.x = 0.0F;
    m_data.y = 0.0F;
}

} // namespace Candera

#endif    // CANDERA_VECTOR2_H
