//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RECTANGLE_H)
#define CANDERA_RECTANGLE_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/MathStringBufferAppenders.h>

namespace Candera {

    /** @addtogroup MathematicsSystem
    *  @{
    */

    class Matrix3x2;
    class Vector2;

    /**
    * @brief Represents a rectangle.
    */
    class Rectangle  {

    public:
        /**
         *  Constructor
         */
        Rectangle();

        /**
         *  Constructor with given parameters.
         *  @param left   Left side of the rectangle.
         *  @param top    Top side of the rectangle.
         *  @param width  Width of the rectangle.
         *  @param height Height of the rectangle.
         */
        Rectangle(Float left, Float top, Float width, Float height);

        /**
         *  Constructor with given parameters.
         *  @param position The position as a vector (left, top).
         *  @param size     The size as a vector (width, height).
         */
        Rectangle(const Vector2& position, const Vector2& size);

        /**
         *  CopyConstructor from another rectangle.
         *  @param rectangle The other rectangle.
         */
        Rectangle(const Rectangle& rectangle);

        /**
         *  Destructor
         */
        ~Rectangle();

        void Set(Float left, Float top, Float width, Float height) { m_left = left; m_top = top; m_width = width; m_height = height; }

        /**
         *  The left coordinate of this Rectangle.
         *  @param left Value to set as left coordinate of this Rectangle.
         */
        void SetLeft(Float left) { m_left = left; }

        /**
         *  Retrieves the left coordinate of this Rectangle.
         *  @return The left coordinate of this Rectangle.
         */
        Float GetLeft() const { return m_left; }

        /**
         *  The top coordinate of this Rectangle.
         *  @param top Value to set as top coordinate of this Rectangle.
         */
        void SetTop(Float top) { m_top = top; }

        /**
         *  Retrieves the top coordinate of this Rectangle.
         *  @return The top coordinate of this Rectangle.
         */
        Float GetTop() const { return m_top; }

        /**
         *  The width of this Rectangle.
         *  @param width Value to set as width of this Rectangle.
         */
        void SetWidth(Float width) { m_width = width; }

        /**
         *  Retrieves the width of this Rectangle.
         *  @return The width of this Rectangle.
         */
        Float GetWidth() const { return m_width; }

        /**
         *  The height of this Rectangle.
         *  @param height Value to set as height of this Rectangle.
         */
        void SetHeight(Float height) { m_height = height; }

        /**
         *  Retrieves the height of this Rectangle.
         *  @return The height of this Rectangle.
         */
        Float GetHeight() const { return m_height; }

        /**
         *  Sets the position of this Rectangle.
         *  @param left Left position.
         *  @param top  Top position.
         */
        void SetPosition(Float left, Float top) { m_left = left; m_top = top; }

        /**
         *  Sets the position of this Rectangle with a vector.
         *  @param position The position as a vector (left, top).
         */
        void SetPosition(const Vector2& position) { SetPosition(position.GetX(), position.GetY()); }

        /**
         *  Retrieves the position of this Rectangle as a vector.
         *  @return The position as a vector (left, top).
         */ 
        Vector2 GetPosition() const { return Vector2(m_left, m_top); }

        /**
         *  Sets the size of this Rectangle.
         *  @param width  The width that is set.
         *  @param height The height that is set.
         */
        void SetSize(Float width, Float height) { m_width = width; m_height = height; }

        /**
         *  Sets the size of this Rectangle with a vector.
         *  @param size The size as a vector (width, height).
         */
        void SetSize(const Vector2& size) { SetSize(size.GetX(), size.GetY()); }

        /**
         *  Retrieves the size of this Rectangle as a vector.
         *  @return The size as a vector (width, height).
         */
        Vector2 GetSize() const { return Vector2(m_width, m_height); }

        Rectangle& operator = (const Rectangle& rhs);

        bool operator==(const Rectangle& rhs) const;
        bool operator!=(const Rectangle& rhs) const;

        /**
         *  Set the Rectangle to the smallest rectangle bounding
         *  the union between this rectangle and the source.
         *  @param  src     source rectangle
         */
        void Union(const Rectangle& src);

        /**
         *  Set the Rectangle to the smallest rectangle bounding
         *  the intersection between this rectangle and the source.
         *  @param  src     source rectangle.
         */
        void Intersect(const Rectangle& src);

        /**
         *  Set the Rectangle to the smallest rectangle bounding
         *  the intersection between this rectangle and the source.
         *  @param  child     source rectangle with coordinate relative to the destination.
         */
        void IntersectChild(const Rectangle& child);

        /**
         *  Set the Rectangle to the smallest rectangle bounding
         *  the set obtained by transforming this rectangle.
         *  @param  mat     matrix used for transforming the rectangle.
         */
        void Transform(const Matrix3x2& mat);

        /**
         *  Retrieves whether a point is inside this Rectangle or not.
         *  @param point The point that is checked.
         *  @return True, if the Rectangle contains the point, otherwise false.
         */
        bool Contains(const Vector2& point) const;

        /**
          * Provides access to the properties by specifying an index.
          * Index 0 returns left and accordingly.
          * @param index Index to be resolved.Index has to be in the range 0 - 3.
          * @return Float value of indexed Rectangle.
        */
        Float& operator [] (UInt8 index);
        Float operator [] (UInt8 index) const;

    private:
        Float m_left;
        Float m_top;
        Float m_width;
        Float m_height;
    };

    /** @} */ // end of MathematicsSystem

} // namespace Candera

#endif// CANDERA_RECTANGLE_H
