//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_PLANE_H)
#define CANDERA_PLANE_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <Candera/System/Mathematics/Math.h>

namespace Candera {

/** @addtogroup MathematicsSystem
 *  @{
 */

/**
 * @brief The class plane has an infinite extension and is defined by a direction vector (x,y,z) that is the normal to the plane and the 
 *        distance (d) to the origin of the Cartesian coordinate system.
 *        The plane equation is as follows: ax + by + cz + d = 0.
 *        Distance from a point to a plane when x,y,z is normalized is calculated as follows: D = |ax + by + cz + d|. 
 */
class Plane
{
    public:
        /**
         *  Constructor
         */
        Plane();

        /**
         *  Destructor
         */
        ~Plane(); // non-virtual

        /**
         *  CopyConstructor from another plane.
         *  @param plane The other plane.
         */
        Plane(const Plane& plane);       


        /**
         *  Constructs a plane from it's direction and distance to the origin of the Cartesian coordinate system.
         *  @param  direction   A Vector3 which describes the direction of the plane.  
         *  @param  distance    The distance of the Plane from the origin of the Cartesian coordinate system to direction vector.
         */
        Plane(const Vector3& direction, Float distance = 0.0F); 

        /**
         *  Gets the distance to a given point.
         *  @param  point   The target point.
         *  @return         The distance to the given point.
         */
        Float GetDistanceTo(const Vector3& point) const { return m_direction.GetDotProduct(point) + m_distance; }

        /**
         *  Calculates if a given point is in front of the plane.
         *  @param  point   The target point.
         *  @return         TRUE if the point is in front of the plane, otherwise FALSE.
         */
        bool IsInFront(const Vector3& point) const { return (GetDistanceTo(point) > 0.0F) ? true : false;}

        /**
         *  Calculates if a given point is behind the plane.
         *  @param  point   The target point.
         *  @return         TRUE if the point is behind the plane, otherwise FALSE.
         */
        bool IsInBack(const Vector3& point) const {return (GetDistanceTo(point) < 0.0F) ? true : false;}

        /**
         *  Calculates if the plane intersects with a given line.
         *  @param [out] point  If there is an intersection the intersection point will be returned here.
         *                      If there is no intersection this Vector3 is unchanged. (e.g. the line is parallel to the plane)
         *  @param  start       The starting point of the line.
         *  @param  end         The end point of the line.
         *  @param  accuracy    The accepted imprecision for the calculation. Default is Math::EpsilonFloat3D.
         *  @return             TRUE if a intersection if found, FALSE if not.
         */
        bool IsIntersectingLine(Vector3& point, const Vector3& start, const Vector3& end, Float accuracy = Math::EpsilonFloat3D()) const;
        
        /**
         *  Calculates the dot product with a Vector4.
         *  @param  vector   The vector.
         *  @return          The dot product with the vector. 
         */
        Float GetDotProduct(const Vector4& vector) const;

        /**
         *  Calculates the dot product with a Vector3. The w parameter of the Vector3 is assumed to be 1.
         *  @param  vector   The vector.
         *  @return          The dot product with the vector. 
         */
        Float GetDotCoordinate(const Vector3& vector) const { return m_direction.GetDotProduct(vector) + m_distance; }

        
        /**
         *  Set direction of the plane. 
         *  @param direction The direction of the infinite plane
         */
        void SetDirection(const Vector3& direction) { m_direction = direction; }

        /**
         *  Returns the direction of the Plane as a Vector3.
         *  @return         A Vector3 which describes the direction of the plane.        
         */
        Vector3 GetDirection() const { return m_direction; }

        /**
         *  Normalizes the plane
         */
        void Normalize() { static_cast<void>(m_direction.Normalize()); }

        /**
         *  Scales the plane with a given value.
         *  @param value The Scale-Value.
         */
        void Scale(Float value) { m_direction *= value; m_distance *= value; }

        
        /**
         *  Set distance of the plane to origin of the Cartesian coordinate system.
         *  @param distance Distance of the plane to the origin of the Cartesian coordinate system to set.
         */
        void SetDistance(Float distance) { m_distance = distance; }

        /**  
         *  Returns the distance of the Plane to Vector3(0.0F, 0.0F, 0.0F). 
         *  @return The distance of the Plane to Vector3(0.0F, 0.0F, 0.0F).
         */
        Float GetDistance() const { return m_distance; }

        /**
         *  Sets the direction and distance of the Plane.
         *  @param  direction   A Vector3 which describes the direction of the plane.
         *  @param  distance    The distance of the Plane to Vector3(0.0F, 0.0F, 0.0F).
         */
        void Set(const Vector3& direction, Float distance) { m_direction = direction; m_distance = distance; }

        /**
         *  assignment operator
         *  @param rhs The right hand side of the operator.
         *  @return The result of the operator as a Plane.
         */
        Plane& operator = (const Plane& rhs);

        // equal operator
        bool operator==(const Plane& rhs) const;
        // unequal operator
        bool operator!=(const Plane& rhs) const { return !operator==(rhs); }

    private:
        Float m_distance;
        Vector3 m_direction;
};

 /** @} */ // end of MathematicsSystem
 
} // namespace Candera

#endif// CANDERA_PLANE_H
