//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MATRIX4_H)
#define CANDERA_MATRIX4_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Math.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

namespace Candera {
    /** @addtogroup MathematicsSystem
     *  @{
     */

    //Forward declarations to prevent reciprocal dependencies
    class Matrix3;
    class Vector3;

    /**
     * @brief The default representation of a 4x4 Matrix4.
     */
    class Matrix4
    {
    public:
        /**
         *  Constructor
         */
        Matrix4() { SetIdentity(); }

        /**
         *  Destructor
         */
        ~Matrix4() {} // non-virtual

        /**
         *  CopyConstructor from another 4x4 Matrix.
         *  @param matrix4 The other 4x4 Matrix.
         */
        Matrix4(const Matrix4& matrix4);

        /**
         *  Constructor each element of the matrix is set individually
         *  @param m00 Element 0,0
         *  @param m01 Element 0,1
         *  @param m02 Element 0,2
         *  @param m03 Element 0,3
         *  @param m10 Element 1,0
         *  @param m11 Element 1,1
         *  @param m12 Element 1,2
         *  @param m13 Element 1,3
         *  @param m20 Element 2,0
         *  @param m21 Element 2,1
         *  @param m22 Element 2,2
         *  @param m23 Element 2,3
         *  @param m30 Element 3,0
         *  @param m31 Element 3,1
         *  @param m32 Element 3,2
         *  @param m33 Element 3,3
         */
        Matrix4(Float m00, Float m01, Float m02, Float m03,
            Float m10, Float m11, Float m12, Float m13,
            Float m20, Float m21, Float m22, Float m23,
            Float m30, Float m31, Float m32, Float m33);

        /**
         *  Constructs a 4x4 Matrix4 out of a float array.
         *  @param  arr16 The float array consisting of 16 values.
         */
        Matrix4(const Float arr16[]);

        /**
         *  Sets all values of this Matrix4 to 0.
         */
        void SetZero();

        /**
         *  Sets the Identity Matrix
         */
        void SetIdentity();

        /**
         *  Sets the Matrix4 to a rotated Matrix4 around the X-axis.
         *  The Matrix4 is at first initialized to an identity Matrix4
         *  and then the Rotation values are set.
         *  @param  angle       The angle in degree.
         */
        void SetRotationX(Float angle);

        /**
         *  Sets the Matrix4 to a rotated Matrix4 around the Y-axis.
         *  The Matrix4 is at first initialized to an identity Matrix4
         *  and then the Rotation values are set.
         *  @param  angle       The angle in degree.
         */
        void SetRotationY(Float angle);

        /**
         *  Sets the Matrix4 to a rotated Matrix4 around the Z-axis.
         *  The Matrix4 is at first initialized to an identity Matrix4
         *  and then the Rotation values are set.
         *  @param  angle       The angle in degree.
         */
        void SetRotationZ(Float angle);

        /**
         *  Sets the Matrix4 to a rotated Matrix4 around the X, Y and Z-axis.
         *  The Matrix4 is at first initialized to an identity Matrix4
         *  and then the Rotation values are set. This method is faster then
         *  using 3 consecutive calls to SetRotationX, SetRotationY and SetRotationZ.
         *  @param  x_angle       The angle to rotate around the x-axis in degree.
         *  @param  y_angle       The angle to rotate around the y-axis in degree.
         *  @param  z_angle       The angle to rotate around the z-axis in degree.
         */
        void SetRotationXYZ(Float x_angle, Float y_angle, Float z_angle);

        /**
         *  Sets the Matrix4 to a rotated Matrix4 around a specified axis.
         *  The Matrix4 is at first initialized to an identity Matrix4
         *  and then the Rotation values are set.
         *  @param  angle    The angle in degree.
         *  @param  x        The x value of the rotation axis.
         *  @param  y        The y value of the rotation axis.
         *  @param  z        The z value of the rotation axis.
         */
        void SetRotationAxis(Float angle, Float x, Float y, Float z);

        /**
         *  Sets the Matrix4 to a translated Matrix4.
         *  The Matrix4 is at first initialized to an identity Matrix4
         *  and then the translation values are set.
         *  @param  x           The offset on the X-axis.
         *  @param  y           The offset on the Y-axis.
         *  @param  z           The offset on the Z-axis.
         */
        void SetTranslation(Float x, Float y, Float z);

        /**
         *  Sets the Matrix4 to a scaled Matrix4.
         *  The Matrix4 is at first initialized to an identity Matrix4
         *  and then the scaling values are set.
         *  @param  x           The scaling on the X-axis.
         *  @param  y           The scaling on the Y-axis.
         *  @param  z           The scaling on the Z-axis.
         */
        void SetScaling(Float x, Float y, Float z);

        /**  Decomposes this matrix M as M = S *  R *  T.
         *  S (scale) and T (translation) are represented as vectors, R (rotation) as a matrix.
         *  Assumptions:
         *   1. M is in fact decomposable.
         *   2. All scaling factors are positive.
         *  @param s Vector to write scale vector of this matrix into.
         *  @param r Matrix to write rotation matrix of this matrix into.
         *  @param t Vector to write translation vector of this matrix into.
         *  @return false if any scaling factor is zero.
         */
        bool Decompose(Vector3& s, Matrix4& r,Vector3& t) const;

        /**
         *  Creates the inverse version of the Matrix4.
         */
        void Inverse();

        /**
         *  Creates the transpose version of the Matrix4.
         */
        void Transpose();

        /**
         *  Retrieves the determinant of the Matrix3.
         *  @return The Determinant of the Matrix3.
         */
        Float GetDeterminant() const;

        /**
         *  Gets the value at a specific row and column.
         *  The indexes of row and column start with 0.
         *  @param  row     The index of the row.
         *  @param  col     The index of the column.
         *  @return         The value at the intersection of row and col.
         */
        Float Get(UInt8 row, UInt8 col) const;

        /**
         *  Gets the upper 3x3 matrix.
         *  @return         The Matrix3
         */
        Matrix3 GetUpper3x3(void) const;

        /**
         *  Sets a value at a specific row and column.
         *  The indexes of row and column start with 0.
         *  @param  row     The index of the row.
         *  @param  col     The index of the column.
         *  @param  val     The value to set.
         */
        void Set(UInt8 row, UInt8 col, Float val);

        Matrix4 operator * (const Matrix4& rhs) const;
        Matrix4& operator *= (const Matrix4& rhs);
        Matrix4& operator = (const Matrix4& rhs);
        Float& operator () (UInt8 row, UInt8 col);
        Float  operator () (UInt8 row, UInt8 col) const;

        bool operator == (const Matrix4& rhs) const;
        bool operator != (const Matrix4& rhs) const;

        /**
         * Gets the data out of the Matrix4.
         * @return The data out of the Matrix4
         */
        const Float* GetData() const { return &m_data[0][0]; };

    private:
        Float m_data[4][4];
    };

    /** @} */ // end of MathematicsSystem

    inline Matrix4& Matrix4::operator = (const Matrix4& rhs)
    {
        MemoryPlatform::Copy(m_data, &rhs.m_data, sizeof(m_data));
        return *this;
    }

    inline Float Matrix4::Get(UInt8 row, UInt8 col) const
    {
        return m_data[row][col];
    }

    inline void Matrix4::Set(UInt8 row, UInt8 col, Float val)
    {
        m_data[row][col] = val;
    }

    inline Float& Matrix4::operator () (UInt8 row, UInt8 col)
    {
        return m_data[row][col];
    }

    inline Float Matrix4::operator () (UInt8 row, UInt8 col) const
    {
        return m_data[row][col];
    }

    inline bool Matrix4::operator == (const Matrix4& rhs) const
    {
        return MemoryPlatform::Compare(m_data, rhs.m_data, sizeof(m_data)) == 0;
    }

    inline bool Matrix4::operator!= (const Matrix4& rhs) const
    {
        return !(*this == rhs);
    }
} // namespace Candera

#endif    // CANDERA_MATRIX4_H
