//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Matrix4.h"
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <Candera/System/Mathematics/Matrix3.h>

namespace Candera {

using namespace Diagnostics;

Matrix4::Matrix4(const Matrix4& matrix4)
{
    //no check for self assignment due to performance reasons
    MemoryPlatform::Copy(m_data, matrix4.m_data, sizeof(m_data));
}

Matrix4::Matrix4(Float m00, Float m01, Float m02, Float m03,
                 Float m10, Float m11, Float m12, Float m13,
                 Float m20, Float m21, Float m22, Float m23,
                 Float m30, Float m31, Float m32, Float m33)
{
    m_data[0][0] = m00; m_data[0][1] = m01; m_data[0][2] = m02; m_data[0][3] = m03;
    m_data[1][0] = m10; m_data[1][1] = m11; m_data[1][2] = m12; m_data[1][3] = m13;
    m_data[2][0] = m20; m_data[2][1] = m21; m_data[2][2] = m22; m_data[2][3] = m23;
    m_data[3][0] = m30; m_data[3][1] = m31; m_data[3][2] = m32; m_data[3][3] = m33;
}

Matrix4::Matrix4(const Float arr16[])
{
    MemoryPlatform::Copy(m_data, arr16, sizeof(m_data));
}

void Matrix4::SetZero()
{
    MemoryPlatform::Set(m_data, 0, sizeof(m_data));
}

void Matrix4::SetRotationX(Float angle)
{
    SetIdentity();
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    m_data[1][1] = cr;
    m_data[2][2] = cr;
    m_data[1][2] = sr;
    m_data[2][1] = -sr;
}

void Matrix4::SetRotationY(Float angle)
{
    SetIdentity();
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    m_data[0][0] = cr;
    m_data[2][2] = cr;
    m_data[0][2] = -sr;
    m_data[2][0] = sr;
}

void Matrix4::SetRotationZ(Float angle)
{
    SetIdentity();
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    m_data[0][0] = cr;
    m_data[1][1] = cr;
    m_data[0][1] = sr;
    m_data[1][0] = -sr;
}

void Matrix4::SetRotationXYZ(Float x_angle, Float y_angle, Float z_angle)
{
    SetIdentity();

    x_angle = Math::DegreeToRadian((x_angle));
    y_angle = Math::DegreeToRadian((y_angle));
    z_angle = Math::DegreeToRadian((z_angle));

    Float sx = Math::Sine(x_angle);
    Float cx = Math::Cosine(x_angle);
    Float sy = Math::Sine(y_angle);
    Float cy = Math::Cosine(y_angle);
    Float sz = Math::Sine(z_angle);
    Float cz = Math::Cosine(z_angle);

    m_data[0][0] = cy * cz;
    m_data[0][1] = cy * sz;
    m_data[0][2] = -sy;

    m_data[1][0] = (cz * sx * sy) - (cx * sz);
    m_data[1][1] = (cx * cz) + (sx * sy * sz);
    m_data[1][2] = cy * sx;

    m_data[2][0] = (cx * cz * sy) + (sx * sz);
    m_data[2][1] = (cx * sy * sz) - (sx*cz);
    m_data[2][2] = cx * cy;
}

void Matrix4::SetRotationAxis(Float angle, Float x, Float y, Float z)
{
   SetIdentity();
   Float sinAngle;
   Float cosAngle;
   Float mag = Math::SquareRoot((x * x) + (y * y) + (z * z));

   sinAngle = Math::Sine((angle * Math::Pi()) / 180.0F );
   cosAngle = Math::Cosine((angle * Math::Pi()) / 180.0F );
   if (mag > 0.0F)
   {
      Float xx;
      Float yy;
      Float zz;
      Float xy;
      Float yz;
      Float zx;
      Float xs;
      Float ys;
      Float zs;
      Float oneMinusCos;

      x /= mag;
      y /= mag;
      z /= mag;

      xx = x * x;
      yy = y * y;
      zz = z * z;
      xy = x * y;
      yz = y * z;
      zx = z * x;
      xs = x * sinAngle;
      ys = y * sinAngle;
      zs = z * sinAngle;
      oneMinusCos = 1.0F - cosAngle;

      m_data[0][0] = (oneMinusCos * xx) + cosAngle;
      m_data[0][1] = (oneMinusCos * xy) - zs;
      m_data[0][2] = (oneMinusCos * zx) + ys;
      m_data[0][3] = 0.0F;

      m_data[1][0] = (oneMinusCos * xy) + zs;
      m_data[1][1] = (oneMinusCos * yy) + cosAngle;
      m_data[1][2] = (oneMinusCos * yz) - xs;
      m_data[1][3] = 0.0F;

      m_data[2][0] = (oneMinusCos * zx) - ys;
      m_data[2][1] = (oneMinusCos * yz) + xs;
      m_data[2][2] = (oneMinusCos * zz) + cosAngle;
      m_data[2][3] = 0.0F;

      m_data[3][0] = 0.0F;
      m_data[3][1] = 0.0F;
      m_data[3][2] = 0.0F;
      m_data[3][3] = 1.0F;
   }
}

void Matrix4::SetTranslation(Float x, Float y, Float z)
{
    SetIdentity();
    m_data[3][0] = x;
    m_data[3][1] = y;
    m_data[3][2] = z;
}



void Matrix4::SetScaling(Float x, Float y, Float z)
{
    SetIdentity();
    m_data[0][0] = x;
    m_data[1][1] = y;
    m_data[2][2] = z;
}

bool Matrix4::Decompose(Vector3& s, Matrix4& r,Vector3& t) const
{
    // Decompose translation.
    t.SetX(m_data[3][0]);
    t.SetY(m_data[3][1]);
    t.SetZ(m_data[3][2]);

    // Decompose scaling.
    const Float sX = Math::SquareRoot((m_data[0][0] * m_data[0][0]) + (m_data[0][1] * m_data[0][1]) + (m_data[0][2] * m_data[0][2]));
    const Float sY = Math::SquareRoot((m_data[1][0] * m_data[1][0]) + (m_data[1][1] * m_data[1][1]) + (m_data[1][2] * m_data[1][2]));
    const Float sZ = Math::SquareRoot((m_data[2][0] * m_data[2][0]) + (m_data[2][1] * m_data[2][1]) + (m_data[2][2] * m_data[2][2]));

    if ((sX == 0.0F) || (sY == 0.0F) || (sZ == 0.0F)) {
        return false;
    }

    s.SetX(sX);
    s.SetY(sY);
    s.SetZ(sZ);

    // Decompose rotation.
    r = *this;

    r.m_data[0][0] /= sX;
    r.m_data[0][1] /= sX;
    r.m_data[0][2] /= sX;

    r.m_data[1][0] /= sY;
    r.m_data[1][1] /= sY;
    r.m_data[1][2] /= sY;

    r.m_data[2][0] /= sZ;
    r.m_data[2][1] /= sZ;
    r.m_data[2][2] /= sZ;

    r.m_data[3][0] = 0.0F;
    r.m_data[3][1] = 0.0F;
    r.m_data[3][2] = 0.0F;

    return true;
}

void Matrix4::Inverse()
{
    //lint --e{438} keep assignment for better understanding
    //lint --e{838} keep assignment for better understanding
    Float m00 = m_data[0][0]; Float m01 = m_data[0][1]; Float m02 = m_data[0][2]; Float m03 = m_data[0][3];
    Float m10 = m_data[1][0]; Float m11 = m_data[1][1]; Float m12 = m_data[1][2]; Float m13 = m_data[1][3];
    Float m20 = m_data[2][0]; Float m21 = m_data[2][1]; Float m22 = m_data[2][2]; Float m23 = m_data[2][3];
    Float m30 = m_data[3][0]; Float m31 = m_data[3][1]; Float m32 = m_data[3][2]; Float m33 = m_data[3][3];

    Float v0 = (m20 * m31) - (m21 * m30);
    Float v1 = (m20 * m32) - (m22 * m30);
    Float v2 = (m20 * m33) - (m23 * m30);
    Float v3 = (m21 * m32) - (m22 * m31);
    Float v4 = (m21 * m33) - (m23 * m31);
    Float v5 = (m22 * m33) - (m23 * m32);

    Float t00 = + (((v5 * m11) - (v4 * m12)) + (v3 * m13));
    Float t10 = - (((v5 * m10) - (v2 * m12)) + (v1 * m13));
    Float t20 = + (((v4 * m10) - (v2 * m11)) + (v0 * m13));
    Float t30 = - (((v3 * m10) - (v1 * m11)) + (v0 * m12));

    Float invDet = 1.0F / ((t00 * m00) + (t10 * m01) + (t20 * m02) + (t30 * m03));

    Float d00 = t00 * invDet;
    Float d10 = t10 * invDet;
    Float d20 = t20 * invDet;
    Float d30 = t30 * invDet;

    Float d01 = - (((v5 * m01) - (v4 * m02)) + (v3 * m03)) * invDet;
    Float d11 = + (((v5 * m00) - (v2 * m02)) + (v1 * m03)) * invDet;
    Float d21 = - (((v4 * m00) - (v2 * m01)) + (v0 * m03)) * invDet;
    Float d31 = + (((v3 * m00) - (v1 * m01)) + (v0 * m02)) * invDet;

    v0 = (m10 * m31) - (m11 * m30);
    v1 = (m10 * m32) - (m12 * m30);
    v2 = (m10 * m33) - (m13 * m30);
    v3 = (m11 * m32) - (m12 * m31);
    v4 = (m11 * m33) - (m13 * m31);
    v5 = (m12 * m33) - (m13 * m32);

    Float d02 = + (((v5 * m01) - (v4 * m02)) + (v3 * m03)) * invDet;
    Float d12 = - (((v5 * m00) - (v2 * m02)) + (v1 * m03)) * invDet;
    Float d22 = + (((v4 * m00) - (v2 * m01)) + (v0 * m03)) * invDet;
    Float d32 = - (((v3 * m00) - (v1 * m01)) + (v0 * m02)) * invDet;

    v0 = (m21 * m10) - (m20 * m11);
    v1 = (m22 * m10) - (m20 * m12);
    v2 = (m23 * m10) - (m20 * m13);
    v3 = (m22 * m11) - (m21 * m12);
    v4 = (m23 * m11) - (m21 * m13);
    v5 = (m23 * m12) - (m22 * m13);

    Float d03 = - (((v5 * m01) - (v4 * m02)) + (v3 * m03)) * invDet;
    Float d13 = + (((v5 * m00) - (v2 * m02)) + (v1 * m03)) * invDet;
    Float d23 = - (((v4 * m00) - (v2 * m01)) + (v0 * m03)) * invDet;
    Float d33 = + (((v3 * m00) - (v1 * m01)) + (v0 * m02)) * invDet;

    m_data[0][0] = d00;
    m_data[0][1] = d01;
    m_data[0][2] = d02;
    m_data[0][3] = d03;

    m_data[1][0] = d10;
    m_data[1][1] = d11;
    m_data[1][2] = d12;
    m_data[1][3] = d13;

    m_data[2][0] = d20;
    m_data[2][1] = d21;
    m_data[2][2] = d22;
    m_data[2][3] = d23;

    m_data[3][0] = d30;
    m_data[3][1] = d31;
    m_data[3][2] = d32;
    m_data[3][3] = d33;
}

static inline void Swap(Float& f1, Float&f2)
{
    const Float aux = f1;
    f1 = f2;
    f2 = aux;
}

void Matrix4::Transpose()
{
    Swap(m_data[0][1], m_data[1][0]);
    Swap(m_data[0][2], m_data[2][0]);
    Swap(m_data[0][3], m_data[3][0]);

    Swap(m_data[1][2], m_data[2][1]);
    Swap(m_data[1][3], m_data[3][1]);

    Swap(m_data[2][3], m_data[3][2]);
}

Float Matrix4::GetDeterminant() const
{
    Float determinant;
    Vector4 vec1;
    Vector4 vec2;
    Vector4 vec3;
    Vector4 min;

    vec1.SetX(m_data[0][0]);
    vec1.SetY(m_data[1][0]);
    vec1.SetZ(m_data[2][0]);
    vec1.SetW(m_data[3][0]);

    vec2.SetX(m_data[0][1]);
    vec2.SetY(m_data[1][1]);
    vec2.SetZ(m_data[2][1]);
    vec2.SetW(m_data[3][1]);

    vec3.SetX(m_data[0][2]);
    vec3.SetY(m_data[1][2]);
    vec3.SetZ(m_data[2][2]);
    vec3.SetW(m_data[3][2]);

    min = vec1.GetCrossProduct(vec2, vec3);
    determinant = -((m_data[0][3] * min.GetX()) + (m_data[1][3] * min.GetY()) + (m_data[2][3] * min.GetZ()) + (m_data[3][3] * min.GetW()));

    return determinant;
}

Matrix4 Matrix4::operator * (const Matrix4& rhs) const
{
    return Matrix4(((m_data[0][0] * rhs.m_data[0][0]) + (m_data[0][1] * rhs.m_data[1][0]) + (m_data[0][2] * rhs.m_data[2][0]) + (m_data[0][3] * rhs.m_data[3][0])),
                   ((m_data[0][0] * rhs.m_data[0][1]) + (m_data[0][1] * rhs.m_data[1][1]) + (m_data[0][2] * rhs.m_data[2][1]) + (m_data[0][3] * rhs.m_data[3][1])),
                   ((m_data[0][0] * rhs.m_data[0][2]) + (m_data[0][1] * rhs.m_data[1][2]) + (m_data[0][2] * rhs.m_data[2][2]) + (m_data[0][3] * rhs.m_data[3][2])),
                   ((m_data[0][0] * rhs.m_data[0][3]) + (m_data[0][1] * rhs.m_data[1][3]) + (m_data[0][2] * rhs.m_data[2][3]) + (m_data[0][3] * rhs.m_data[3][3])),

                   ((m_data[1][0] * rhs.m_data[0][0]) + (m_data[1][1] * rhs.m_data[1][0]) + (m_data[1][2] * rhs.m_data[2][0]) + (m_data[1][3] * rhs.m_data[3][0])),
                   ((m_data[1][0] * rhs.m_data[0][1]) + (m_data[1][1] * rhs.m_data[1][1]) + (m_data[1][2] * rhs.m_data[2][1]) + (m_data[1][3] * rhs.m_data[3][1])),
                   ((m_data[1][0] * rhs.m_data[0][2]) + (m_data[1][1] * rhs.m_data[1][2]) + (m_data[1][2] * rhs.m_data[2][2]) + (m_data[1][3] * rhs.m_data[3][2])),
                   ((m_data[1][0] * rhs.m_data[0][3]) + (m_data[1][1] * rhs.m_data[1][3]) + (m_data[1][2] * rhs.m_data[2][3]) + (m_data[1][3] * rhs.m_data[3][3])),

                   ((m_data[2][0] * rhs.m_data[0][0]) + (m_data[2][1] * rhs.m_data[1][0]) + (m_data[2][2] * rhs.m_data[2][0]) + (m_data[2][3] * rhs.m_data[3][0])),
                   ((m_data[2][0] * rhs.m_data[0][1]) + (m_data[2][1] * rhs.m_data[1][1]) + (m_data[2][2] * rhs.m_data[2][1]) + (m_data[2][3] * rhs.m_data[3][1])),
                   ((m_data[2][0] * rhs.m_data[0][2]) + (m_data[2][1] * rhs.m_data[1][2]) + (m_data[2][2] * rhs.m_data[2][2]) + (m_data[2][3] * rhs.m_data[3][2])),
                   ((m_data[2][0] * rhs.m_data[0][3]) + (m_data[2][1] * rhs.m_data[1][3]) + (m_data[2][2] * rhs.m_data[2][3]) + (m_data[2][3] * rhs.m_data[3][3])),

                   ((m_data[3][0] * rhs.m_data[0][0]) + (m_data[3][1] * rhs.m_data[1][0]) + (m_data[3][2] * rhs.m_data[2][0]) + (m_data[3][3] * rhs.m_data[3][0])),
                   ((m_data[3][0] * rhs.m_data[0][1]) + (m_data[3][1] * rhs.m_data[1][1]) + (m_data[3][2] * rhs.m_data[2][1]) + (m_data[3][3] * rhs.m_data[3][1])),
                   ((m_data[3][0] * rhs.m_data[0][2]) + (m_data[3][1] * rhs.m_data[1][2]) + (m_data[3][2] * rhs.m_data[2][2]) + (m_data[3][3] * rhs.m_data[3][2])),
                   ((m_data[3][0] * rhs.m_data[0][3]) + (m_data[3][1] * rhs.m_data[1][3]) + (m_data[3][2] * rhs.m_data[2][3]) + (m_data[3][3] * rhs.m_data[3][3])));
}

Matrix3 Matrix4::GetUpper3x3(void) const
{
    return Matrix3(m_data[0][0], m_data[0][1], m_data[0][2],
                   m_data[1][0], m_data[1][1], m_data[1][2],
                   m_data[2][0], m_data[2][1], m_data[2][2]);
}

Matrix4& Matrix4::operator *= (const Matrix4& rhs)
{
    Float tmp[4][4];
    MemoryPlatform::Copy(tmp, m_data, sizeof(m_data));
    m_data[0][0] = ((tmp[0][0] * rhs.m_data[0][0]) + (tmp[0][1] * rhs.m_data[1][0]) + (tmp[0][2] * rhs.m_data[2][0]) + (tmp[0][3] * rhs.m_data[3][0]));
    m_data[0][1] = ((tmp[0][0] * rhs.m_data[0][1]) + (tmp[0][1] * rhs.m_data[1][1]) + (tmp[0][2] * rhs.m_data[2][1]) + (tmp[0][3] * rhs.m_data[3][1]));
    m_data[0][2] = ((tmp[0][0] * rhs.m_data[0][2]) + (tmp[0][1] * rhs.m_data[1][2]) + (tmp[0][2] * rhs.m_data[2][2]) + (tmp[0][3] * rhs.m_data[3][2]));
    m_data[0][3] = ((tmp[0][0] * rhs.m_data[0][3]) + (tmp[0][1] * rhs.m_data[1][3]) + (tmp[0][2] * rhs.m_data[2][3]) + (tmp[0][3] * rhs.m_data[3][3]));

    m_data[1][0] = ((tmp[1][0] * rhs.m_data[0][0]) + (tmp[1][1] * rhs.m_data[1][0]) + (tmp[1][2] * rhs.m_data[2][0]) + (tmp[1][3] * rhs.m_data[3][0]));
    m_data[1][1] = ((tmp[1][0] * rhs.m_data[0][1]) + (tmp[1][1] * rhs.m_data[1][1]) + (tmp[1][2] * rhs.m_data[2][1]) + (tmp[1][3] * rhs.m_data[3][1]));
    m_data[1][2] = ((tmp[1][0] * rhs.m_data[0][2]) + (tmp[1][1] * rhs.m_data[1][2]) + (tmp[1][2] * rhs.m_data[2][2]) + (tmp[1][3] * rhs.m_data[3][2]));
    m_data[1][3] = ((tmp[1][0] * rhs.m_data[0][3]) + (tmp[1][1] * rhs.m_data[1][3]) + (tmp[1][2] * rhs.m_data[2][3]) + (tmp[1][3] * rhs.m_data[3][3]));

    m_data[2][0] = ((tmp[2][0] * rhs.m_data[0][0]) + (tmp[2][1] * rhs.m_data[1][0]) + (tmp[2][2] * rhs.m_data[2][0]) + (tmp[2][3] * rhs.m_data[3][0]));
    m_data[2][1] = ((tmp[2][0] * rhs.m_data[0][1]) + (tmp[2][1] * rhs.m_data[1][1]) + (tmp[2][2] * rhs.m_data[2][1]) + (tmp[2][3] * rhs.m_data[3][1]));
    m_data[2][2] = ((tmp[2][0] * rhs.m_data[0][2]) + (tmp[2][1] * rhs.m_data[1][2]) + (tmp[2][2] * rhs.m_data[2][2]) + (tmp[2][3] * rhs.m_data[3][2]));
    m_data[2][3] = ((tmp[2][0] * rhs.m_data[0][3]) + (tmp[2][1] * rhs.m_data[1][3]) + (tmp[2][2] * rhs.m_data[2][3]) + (tmp[2][3] * rhs.m_data[3][3]));

    m_data[3][0] = ((tmp[3][0] * rhs.m_data[0][0]) + (tmp[3][1] * rhs.m_data[1][0]) + (tmp[3][2] * rhs.m_data[2][0]) + (tmp[3][3] * rhs.m_data[3][0]));
    m_data[3][1] = ((tmp[3][0] * rhs.m_data[0][1]) + (tmp[3][1] * rhs.m_data[1][1]) + (tmp[3][2] * rhs.m_data[2][1]) + (tmp[3][3] * rhs.m_data[3][1]));
    m_data[3][2] = ((tmp[3][0] * rhs.m_data[0][2]) + (tmp[3][1] * rhs.m_data[1][2]) + (tmp[3][2] * rhs.m_data[2][2]) + (tmp[3][3] * rhs.m_data[3][2]));
    m_data[3][3] = ((tmp[3][0] * rhs.m_data[0][3]) + (tmp[3][1] * rhs.m_data[1][3]) + (tmp[3][2] * rhs.m_data[2][3]) + (tmp[3][3] * rhs.m_data[3][3]));
    return *this;
}

void Matrix4::SetIdentity()
{
    MemoryPlatform::Set(m_data, 0, sizeof(m_data));
    m_data[0][0] = 1.0F;
    m_data[1][1] = 1.0F;
    m_data[2][2] = 1.0F;
    m_data[3][3] = 1.0F;
}

} // namespace Candera

