//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MATRIX3X2_H)
#define CANDERA_MATRIX3X2_H

#include <Candera/Environment.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

namespace Candera {

    /** @addtogroup MathematicsSystem
     *  @{
     */

    class Vector2;

    /**
     * @brief Representation of a 3x2 Matrix used for 2d transformations. 
     */
    class Matrix3x2
    {
    public:
        /**
         * Constructs an identity Matrix3x2.
         */
        Matrix3x2();
        
        /**
         * CopyConstructor
         * @param matrix The other 3x2 matrix
         */
        Matrix3x2(const Matrix3x2& matrix);

        /**
         * Constructor each element of the matrix is set individually
         * @param m00 Element 0,0
         * @param m01 Element 0,1
         * @param m10 Element 1,0
         * @param m11 Element 1,1
         * @param m20 Element 2,0
         * @param m21 Element 2,1
         */
        Matrix3x2(Float m00, Float m01,
            Float m10, Float m11,
            Float m20, Float m21);

        /**
         * Constructs a Matrix3x2 out of a float array.
         * @param  arr6 The float array consisting of at least 6 values.
         */
        Matrix3x2(const Float arr6[]);

        /**
         * Destructor
         */
        ~Matrix3x2(); // non-virtual

        /**
         * Sets all values of this Matrix3x2 to 0.
         */
        void SetZero();

        /**
         * Sets the Identity Matrix
         */
        void SetIdentity();

        /**
         * Sets the Matrix3x2 to a rotation matrix.
         * The Matrix is at first initialized to an identity Matrix3x2
         * and then the Rotation values are set.
         * @param  angle       The angle in degree.
         */
        void SetRotation(Float angle);

        /**
         * Applies rotation to the left of the Matrix3x2 (same as *this * rotation).
         * @param  angle       The angle in degree.
         */
        void Rotate(Float angle);

        /**
         * Applies rotation to the right of the Matrix3x2 (same as rotation * *this).
         * @param  angle       The angle in degree.
         */
        void PreRotate(Float angle);

        /**
         * Sets the Matrix3x2 to a translation matrix.
         * The Matrix3x2 is at first initialized to an identity Matrix3x2
         * and then the Translation values are set.
         * @param  x        The x value.
         * @param  y        The y value.
         */
        void SetTranslation(Float x, Float y);

        /**
        * Gets the x Translation value from the Matrix3x2.
        * @return         The x value.
        */
        Float GetTranslationX() const { return m_data[2][0]; }

        /**
        * Gets the y Translation value from the Matrix3x2.
        * @return         The y value.
        */
        Float GetTranslationY() const { return m_data[2][1]; }

        /**
         * Applies translation to the left of the Matrix3x2 (same as *this * translation).
         * @param  x        The x value.
         * @param  y        The y value.
         */
        void Translate(Float x, Float y);

        /**
         * Applies translation to the right of the Matrix3x2 (same as translation * *this).
         * @param  x        The x value.
         * @param  y        The y value.
         */
        void PreTranslate(Float x, Float y);

        /**
         * Sets the Matrix3x2 to a scale Matrix3x2.
         * The Matrix3x2 is at first initialized to an identity Matrix3x2
         * and then the scaling values are set.
         * @param  x           The scaling on the X-axis.
         * @param  y           The scaling on the Y-axis.
         */
        void SetScaling(Float x, Float y);

        /**
         * Applies scale to the left of the Matrix3x2 (same as *this * scale).
         * @param  x        The x value.
         * @param  y        The y value.
         */
        void Scale(Float x, Float y);

        /**
         * Applies scale to the right of the Matrix3x2 (same as scale * *this).
         * @param  x        The x value.
         * @param  y        The y value.
         */
        void PreScale(Float x, Float y);

        /**
         * Creates the inverse version of the Matrix3x2.
         */
        void Inverse();

        /**
         * Computes determinant of the Matrix3x2.
         * @return Determinant of the Matrix3x2
         */
        Float GetDeterminant() const;

        /**
         * Gets the value at a specific row and column.
         * The indexes of row and column start with 0.
         * @param  row     The index of the row.
         * @param  col     The index of the column.
         * @return         The value at the intersection of row and col.
         */
        Float Get(UInt8 row, UInt8 col) const;

        /**
         * Gets the data out of the Matrix3x2.
         * @return         The data out of the Matrix3x2
         */
        const Float* GetData() const { return &m_data[0][0]; }

        /**
         * Sets a value at a specific row and column.
         * The indexes of row and column start with 0.
         * @param  row     The index of the row.
         * @param  col     The index of the column.
         * @param  val     The value to set.
         */
        void Set(UInt8 row, UInt8 col, Float val);

        /**
         * Multiplies the current Matrix3x2 with a Vector2.
         * It first appends 1 to the vector and then it does the multiplication.
         * @param  vec     Vector.
         * @return         Multiplication result.
         */
        Vector2 Multiply(const Vector2& vec) const;

        /**
         * Multiplies the current Matrix3x2 with another Matrix3x2.
         * It first appends a column containing (0, 0, 1) to the current matrix
         * and then it does the multiplication.
         * @param  rhs     The other matrix.
         * @return         Multiplication result.
         */
        Matrix3x2 operator * (const Matrix3x2& rhs) const;

        Matrix3x2& operator *= (const Matrix3x2& rhs);
        Matrix3x2& operator = (const Matrix3x2& rhs);
        Float& operator () (UInt8 row, UInt8 col);
        Float  operator () (UInt8 row, UInt8 col) const;
        bool operator == (const Matrix3x2& rhs) const;
        bool operator != (const Matrix3x2& rhs) const;

    private:
        Float m_data[3][2];
    };

    /** @} */ // end of MathematicsSystem
    
    inline Matrix3x2& Matrix3x2::operator *= (const Matrix3x2& rhs)
    {
        *this = ((*this) * rhs);
        return *this;
    }

    inline Matrix3x2& Matrix3x2::operator = (const Matrix3x2& rhs)
    {
        MemoryPlatform::Copy(m_data, &rhs.m_data, sizeof(m_data));
        return *this;
    }

    inline Float Matrix3x2::Get(UInt8 row, UInt8 col) const
    {
        return m_data[row][col];
    }

    inline void Matrix3x2::Set(UInt8 row, UInt8 col, Float val)
    {
        m_data[row][col] = val;
    }

    inline Float& Matrix3x2::operator () (UInt8 row, UInt8 col)
    {
        return m_data[row][col];    
    }

    inline Float Matrix3x2::operator () (UInt8 row, UInt8 col) const
    {
        return m_data[row][col];
    }

    inline bool Matrix3x2::operator == (const Matrix3x2& rhs) const
    {
        return MemoryPlatform::Compare(m_data, rhs.m_data, sizeof(m_data)) == 0;
    }

    inline bool Matrix3x2::operator!= (const Matrix3x2& rhs) const
    {
        return !(*this == rhs);
    }

} // namespace Candera

#endif// CANDERA_MATRIX3_H
