//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

namespace Candera {


static const Float cInitData[3][2] = {
    { 1.0F, 0.0F },
    { 0.0F, 1.0F },
    { 0.0F, 0.0F }
};

Matrix3x2::Matrix3x2()
{
    MemoryPlatform::Copy(m_data, cInitData, sizeof(m_data));
}

Matrix3x2::~Matrix3x2()
{
}

Matrix3x2::Matrix3x2(const Matrix3x2& matrix)
{
    //no check for self assignment due to performance reasons
    MemoryPlatform::Copy(m_data, matrix.m_data, sizeof(m_data));
}

Matrix3x2::Matrix3x2(Float m00, Float m01,
                     Float m10, Float m11,
                     Float m20, Float m21)
{
    m_data[0][0] = m00; m_data[0][1] = m01;
    m_data[1][0] = m10; m_data[1][1] = m11;
    m_data[2][0] = m20; m_data[2][1] = m21;
}

Matrix3x2::Matrix3x2(const Float arr6[])
{
    MemoryPlatform::Copy(m_data, arr6, sizeof(m_data));
}

void Matrix3x2::SetZero()
{
    MemoryPlatform::Set(m_data, 0, sizeof(m_data));
}

void Matrix3x2::SetIdentity()
{
    MemoryPlatform::Copy(m_data, cInitData, sizeof(m_data));
}

void Matrix3x2::SetRotation(Float angle) {
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    m_data[0][0] = cr;
    m_data[0][1] = sr;
    
    m_data[1][0] = -sr;
    m_data[1][1] = cr;

    m_data[2][0] = 0.0F;
    m_data[2][1] = 0.0F;
}

void Matrix3x2::Rotate(Float angle) {
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    Float copy[2][2] = {{m_data[0][0], m_data[0][1]}, {m_data[1][0], m_data[1][1]}};

    m_data[0][0] =  (cr * copy[0][0]) + (sr * copy[1][0]);
    m_data[1][0] = (-sr * copy[0][0]) + (cr * copy[1][0]);

    m_data[0][1] =  (cr * copy[0][1]) + (sr * copy[1][1]);
    m_data[1][1] = (-sr * copy[0][1]) + (cr * copy[1][1]);
}

void Matrix3x2::PreRotate(Float angle) {
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    Float copy[3][2] = {{m_data[0][0], m_data[0][1]}, 
                        {m_data[1][0], m_data[1][1]},
                        {m_data[2][0], m_data[2][1]}};

    m_data[0][0] = (cr * copy[0][0]) - (sr * copy[0][1]);
    m_data[0][1] = (sr * copy[0][0]) + (cr * copy[0][1]);

    m_data[1][0] = (cr * copy[1][0]) - (sr * copy[1][1]);
    m_data[1][1] = (sr * copy[1][0]) + (cr * copy[1][1]);

    m_data[2][0] = (cr * copy[2][0]) - (sr * copy[2][1]);
    m_data[2][1] = (sr * copy[2][0]) + (cr * copy[2][1]);
}

void Matrix3x2::SetTranslation(Float x, Float y) {
    m_data[0][0] = 1.0F;
    m_data[0][1] = 0.0F;

    m_data[1][0] = 0.0F;
    m_data[1][1] = 1.0F;

    m_data[2][0] = x;
    m_data[2][1] = y;
}

void Matrix3x2::Translate(Float x, Float y) {
    m_data[2][0] += (m_data[0][0] * x) + (m_data[1][0] * y);
    m_data[2][1] += (m_data[0][1] * x) + (m_data[1][1] * y);
}

void Matrix3x2::PreTranslate(Float x, Float y) {
    m_data[2][0] += x;
    m_data[2][1] += y;
}

void Matrix3x2::SetScaling(Float x, Float y) {
    m_data[0][0] = x;
    m_data[0][1] = 0.0F;

    m_data[1][0] = 0.0F;
    m_data[1][1] = y;

    m_data[2][0] = 0.0F;
    m_data[2][1] = 0.0F;
}

void Matrix3x2::Scale(Float x, Float y) {
    m_data[0][0] *= x;
    m_data[0][1] *= x;

    m_data[1][0] *= y;
    m_data[1][1] *= y;
}

void Matrix3x2::PreScale(Float x, Float y) {
    m_data[0][0] *= x;
    m_data[0][1] *= y;

    m_data[1][0] *= x;
    m_data[1][1] *= y;

    m_data[2][0] *= x;
    m_data[2][1] *= y;
}

void Matrix3x2::Inverse() {
    Float d = (m_data[0][0] * m_data[1][1]) - (m_data[0][1] * m_data[1][0]);

    Float copy[3][2] = {{m_data[0][0], m_data[0][1]}, 
                        {m_data[1][0], m_data[1][1]},
                        {m_data[2][0], m_data[2][1]}};

    m_data[0][0] =  copy[1][1] / d;
    m_data[0][1] = -copy[0][1] / d;

    m_data[1][0] = -copy[1][0] / d;
    m_data[1][1] =  copy[0][0] / d;

    m_data[2][0] = (-m_data[0][0] * copy[2][0]) - (m_data[1][0] * copy[2][1]);
    m_data[2][1] = (-m_data[0][1] * copy[2][0]) - (m_data[1][1] * copy[2][1]);
}

Float Matrix3x2::GetDeterminant() const {
    return (m_data[0][0] * m_data[1][1]) - (m_data[0][1] * m_data[1][0]);
}

Vector2 Matrix3x2::Multiply (const Vector2& vec) const{
    return Vector2((m_data[0][0] * vec.GetX()) + (m_data[1][0] * vec.GetY()) + m_data[2][0],
                   (m_data[0][1] * vec.GetX()) + (m_data[1][1] * vec.GetY()) + m_data[2][1]);
}

Matrix3x2 Matrix3x2::operator * (const Matrix3x2& rhs) const {
    return Matrix3x2((m_data[0][0] * rhs.m_data[0][0]) + (m_data[0][1] * rhs.m_data[1][0]),
                     (m_data[0][0] * rhs.m_data[0][1]) + (m_data[0][1] * rhs.m_data[1][1]),

                     (m_data[1][0] * rhs.m_data[0][0]) + (m_data[1][1] * rhs.m_data[1][0]),
                     (m_data[1][0] * rhs.m_data[0][1]) + (m_data[1][1] * rhs.m_data[1][1]),
                     
                     (m_data[2][0] * rhs.m_data[0][0]) + (m_data[2][1] * rhs.m_data[1][0]) + rhs.m_data[2][0],
                     (m_data[2][0] * rhs.m_data[0][1]) + (m_data[2][1] * rhs.m_data[1][1]) + rhs.m_data[2][1]);
}

} // namespace Candera


