//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Matrix3.h"
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Mathematics/Plane.h>

namespace Candera {


Matrix3::Matrix3()
{
    // default is to create an identity matrix at construction
    MemoryPlatform::Set(m_data, 0, sizeof(m_data));
    m_data[0][0] = 1.0F;
    m_data[1][1] = 1.0F;
    m_data[2][2] = 1.0F;
}

Matrix3::~Matrix3()
{
}

Matrix3::Matrix3(const Matrix3& matrix3)
{
    //no check for self assignment due to performance reasons
    MemoryPlatform::Copy(m_data, matrix3.m_data, sizeof(m_data));
}

Matrix3::Matrix3(const Matrix4& matrix4)
{
    for (UInt8 i = 0; i < 3; i++) {
        for (UInt8 j = 0; j < 3; j++) {
           m_data[i][j] = matrix4.Get(i, j);
        }
    }
}

Matrix3::Matrix3(Float m00, Float m01, Float m02,
                 Float m10, Float m11, Float m12,
                 Float m20, Float m21, Float m22)
{
    m_data[0][0] = m00; m_data[0][1] = m01; m_data[0][2] = m02;
    m_data[1][0] = m10; m_data[1][1] = m11; m_data[1][2] = m12;
    m_data[2][0] = m20; m_data[2][1] = m21; m_data[2][2] = m22;
}

Matrix3::Matrix3(const Float arr9[])
{
    MemoryPlatform::Copy(m_data, arr9, sizeof(m_data));
}

void Matrix3::SetZero()
{
    MemoryPlatform::Set(m_data, 0, sizeof(m_data));
}

void Matrix3::SetIdentity()
{
    MemoryPlatform::Set(m_data, 0, sizeof(m_data));
    m_data[0][0] = 1.0F;
    m_data[1][1] = 1.0F;
    m_data[2][2] = 1.0F;
}

void Matrix3::SetRotationX(Float angle)
{
    SetIdentity();
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    m_data[1][1] = cr;
    m_data[2][2] = cr;
    m_data[1][2] = sr;
    m_data[2][1] = -sr;
}

void Matrix3::SetRotationY(Float angle)
{
    SetIdentity();
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    m_data[0][0] = cr;
    m_data[2][2] = cr;
    m_data[0][2] = -sr;
    m_data[2][0] = sr;
}

void Matrix3::SetRotationZ(Float angle)
{
    SetIdentity();
    angle = Math::DegreeToRadian(angle);
    Float sr = Math::Sine(angle);
    Float cr = Math::Cosine(angle);

    m_data[0][0] = cr;
    m_data[1][1] = cr;
    m_data[0][1] = sr;
    m_data[1][0] = -sr;
}

void Matrix3::SetRotationXYZ(Float x_angle, Float y_angle, Float z_angle)
{
    SetIdentity();

    x_angle = Math::DegreeToRadian((x_angle));
    y_angle = Math::DegreeToRadian((y_angle));
    z_angle = Math::DegreeToRadian((z_angle));

    Float sx = Math::Sine(x_angle);
    Float cx = Math::Cosine(x_angle);
    Float sy = Math::Sine(y_angle);
    Float cy = Math::Cosine(y_angle);
    Float sz = Math::Sine(z_angle);
    Float cz = Math::Cosine(z_angle);

    m_data[0][0] = cy * cz;
    m_data[0][1] = cy * sz;
    m_data[0][2] = -sy;

    m_data[1][0] = (cz * sx * sy) - (cx * sz);
    m_data[1][1] = (cx * cz) + (sx * sy * sz);
    m_data[1][2] = cy * sx;

    m_data[2][0] = (cx * cz * sy) + (sx * sz);
    m_data[2][1] = (cx * sy * sz) - (sx * cz);
    m_data[2][2] = cx * cy;
}

void Matrix3::SetRotationAxis(Float angle, Float x, Float y, Float z)
{
   SetIdentity();
   Float sinAngle;
   Float cosAngle;
   Float mag = Math::SquareRoot((x * x) + (y * y) + (z * z));
      
   sinAngle = Math::Sine((angle * Math::Pi()) / 180.0F );
   cosAngle = Math::Cosine((angle * Math::Pi()) / 180.0F );
   if (mag > 0.0F)
   {
      Float xx;
      Float yy;
      Float zz;
      Float xy;
      Float yz;
      Float zx;
      Float xs;
      Float ys;
      Float zs;
      Float oneMinusCos;
   
      x /= mag;
      y /= mag;
      z /= mag;

      xx = x * x;
      yy = y * y;
      zz = z * z;
      xy = x * y;
      yz = y * z;
      zx = z * x;
      xs = x * sinAngle;
      ys = y * sinAngle;
      zs = z * sinAngle;
      oneMinusCos = 1.0F - cosAngle;

      m_data[0][0] = (oneMinusCos * xx) + cosAngle;
      m_data[0][1] = (oneMinusCos * xy) - zs;
      m_data[0][2] = (oneMinusCos * zx) + ys;

      m_data[1][0] = (oneMinusCos * xy) + zs;
      m_data[1][1] = (oneMinusCos * yy) + cosAngle;
      m_data[1][2] = (oneMinusCos * yz) - xs;

      m_data[2][0] = (oneMinusCos * zx) - ys;
      m_data[2][1] = (oneMinusCos * yz) + xs;
      m_data[2][2] = (oneMinusCos * zz) + cosAngle;
   }
}

void Matrix3::SetScaling(Float x, Float y, Float z)
{
    SetIdentity();
    m_data[0][0] = x;
    m_data[1][1] = y;
    m_data[2][2] = z;
}

void Matrix3::Inverse()
{
    //lint --e{438} keep assignment for better understanding
    //lint --e{550} keep assignment for better understanding
    Float m00 = m_data[0][0]; Float m01 = m_data[0][1]; Float m02 = m_data[0][2];
    Float m10 = m_data[1][0]; Float m11 = m_data[1][1]; Float m12 = m_data[1][2]; 
    Float m20 = m_data[2][0]; Float m21 = m_data[2][1]; Float m22 = m_data[2][2];

    Float d00 = +((m11 * m22) - (m21 * m12));
    Float d01 = -((m10 * m22) - (m20 * m12));
    Float d02 = +((m10 * m21) - (m20 * m11));
    Float d10 = -((m01 * m22) - (m21 * m02));
    Float d11 = +((m00 * m22) - (m20 * m02));
    Float d12 = -((m00 * m21) - (m20 * m01));
    Float d20 = +((m01 * m12) - (m11 * m02));
    Float d21 = -((m00 * m12) - (m10 * m02));
    Float d22 = +((m00 * m11) - (m10 * m01));

    Float invDet = 1.0F / ((m00 * d00) + (m01 * d01) + (m02 * d02));

    m_data[0][0] = d00 * invDet;
    m_data[0][1] = d10 * invDet;
    m_data[0][2] = d20 * invDet;

    m_data[1][0] = d01 * invDet;
    m_data[1][1] = d11 * invDet;
    m_data[1][2] = d21 * invDet;

    m_data[2][0] = d02 * invDet;
    m_data[2][1] = d12 * invDet;
    m_data[2][2] = d22 * invDet;
} 

void Matrix3::Transpose()
{
    Int i;
    Int j = 0;
    Matrix3 temp(*this);

    for (i = 0; i < 3; i++)
    {
        for (j = 0; j < 3; j++)
        {
           m_data[i][j] = temp.m_data[j][i];
        }
    }
}

Float Matrix3::GetDeterminant() const
{
    Float determinant;
    Vector3 vec1;
    Vector3 vec2;
    Vector3 vec3;
    Vector3 min;
    
    vec1.SetX(m_data[0][0]);
    vec1.SetY(m_data[1][0]);
    vec1.SetZ(m_data[2][0]);

    vec2.SetX(m_data[0][1]);
    vec2.SetY(m_data[1][1]);
    vec2.SetZ(m_data[2][1]);

    vec3.SetX(m_data[0][2]);
    vec3.SetY(m_data[1][2]);
    vec3.SetZ(m_data[2][2]);

    min = vec1.GetCrossProduct(vec2);
    determinant = min.GetDotProduct(vec3);

    return determinant;
}

Matrix3 Matrix3::operator * (const Matrix3& rhs) const
{
    return Matrix3(((m_data[0][0] * rhs.m_data[0][0]) + (m_data[0][1] * rhs.m_data[1][0]) + (m_data[0][2] * rhs.m_data[2][0])), 
                   ((m_data[0][0] * rhs.m_data[0][1]) + (m_data[0][1] * rhs.m_data[1][1]) + (m_data[0][2] * rhs.m_data[2][1])),
                   ((m_data[0][0] * rhs.m_data[0][2]) + (m_data[0][1] * rhs.m_data[1][2]) + (m_data[0][2] * rhs.m_data[2][2])),

                   ((m_data[1][0] * rhs.m_data[0][0]) + (m_data[1][1] * rhs.m_data[1][0]) + (m_data[1][2] * rhs.m_data[2][0])),
                   ((m_data[1][0] * rhs.m_data[0][1]) + (m_data[1][1] * rhs.m_data[1][1]) + (m_data[1][2] * rhs.m_data[2][1])),
                   ((m_data[1][0] * rhs.m_data[0][2]) + (m_data[1][1] * rhs.m_data[1][2]) + (m_data[1][2] * rhs.m_data[2][2])),

                   ((m_data[2][0] * rhs.m_data[0][0]) + (m_data[2][1] * rhs.m_data[1][0]) + (m_data[2][2] * rhs.m_data[2][0])),
                   ((m_data[2][0] * rhs.m_data[0][1]) + (m_data[2][1] * rhs.m_data[1][1]) + (m_data[2][2] * rhs.m_data[2][1])),
                   ((m_data[2][0] * rhs.m_data[0][2]) + (m_data[2][1] * rhs.m_data[1][2]) + (m_data[2][2] * rhs.m_data[2][2])));
}

} // namespace Candera


