//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MATH_H)
#define CANDERA_MATH_H

#include <Candera/Environment.h>
#include <CanderaPlatform/OS/MathPlatform.h>

namespace Candera {

    /** @addtogroup MathematicsSystem
     *  @{
     */

    class Line;
    class Matrix4;
    class Vector3;
    class Vector4;

    /**
    * @brief The Math class for mathematical helper functions and defined constants.
    */
    class Math {
    public:
        /**
         * Compares two values and returns the minimum one.
         * @param a        First value to compare.
         * @param b        Second value to compare.
         * @return         The minimum of the two values.
         */
        template<typename OperandType> 
        static const OperandType& Minimum(const OperandType& a, const OperandType& b);

        /**
         * Compares two values and returns the maximum one.
         * @param a        First value to compare.
         * @param b        Second value to compare.
         * @return         The maximum of the two values.
         */
        template<typename OperandType> 
        static const OperandType& Maximum(const OperandType& a, const OperandType& b);
        
        /**
         * Determines whether a float has invalid value.
         * @param f         Float to check.
         * @return          True if f is invalid.
         */
        static bool IsNan(Float f);

        /**
         * Calculates the square root for float types.
         * @param f        A non-negative floating-point value.
         * @return         The square-root of f. If f is negative, SquareRoot returns an indefinite value.
         */
        static Float SquareRoot(Float f);

        /**
         * Calculates the square root for double types.
         * @param d        A non-negative floating-point value.
         * @return         The square-root of d. If d is negative, SquareRoot returns an indefinite value.
         */
        static Double SquareRoot(Double d);

        /**
         * Converts degree to radian.
         * @param degree   The degree value to convert.
         * @return         The radian value.
         */
        static Float DegreeToRadian(Float degree);

        /**
         * Converts radian to degree.
         * @param radian   The radian value to convert.
         * @return         The degree value.
         */
        static Float RadianToDegree(Float radian);

        /**
         * Computes the sine of a given value in radians.
         * @param val      A float value in radians.
         * @return         The sine of the given float value.
         */
        static Float Sine(Float val);

        /** 
         * Computes the arcsine of a given value.
         * @param val A float value inside [-1, 1] interval.
         * @return the arcsine of the given value in radians.
         */
        static Float ASine(Float val);

        /**
         * Computes the cosine of a given value in radians.
         * @param val      A float value in radians.
         * @return         The cosine of the given float value.
         */
        static Float Cosine(Float val);

        /**
         * Computes the arccosine of a given value.
         * @param val A float value inside [-1, 1] interval.
         * @return the arccosine of the given value in radians.
         */
        static Float ACosine(Float val);

        /**
         * Computes the tangent of a given value in radians.
         * @param val      A float value in radians.
         * @return         The tangent of the given float value.
         */
        static Float Tangent(Float val);

        /** 
         * Computes the arctangent of a given value.
         * @param val A float value.
         * @return the arctangent of the given value in radians.
         */
        static Float ATangent(Float val);

        /** 
         * Computes the arctangent 2 of the given values.
         * @param y Opposite side.
         * @param x Adjacent side.
         * @return the arctangent 2 of the given values in radians.
         */
        static Float ATangent2(Float y, Float x);

        /**
         * Computes a base raised to the power of a exponent.
         * @param base      The base.
         * @param exponent  The exponent.
         * @return          The base raised to the power of the exponent.
         */
        static Float Power(Float base, Float exponent);

        /**
         * Sets the given value to the specified maximum value.
         * If \a value does not exceed \a max then \a value is not changed.
         * @param max          The limiting maximum value.
         * @param [out] value   The value.
         */
        static void SetToMax(Float max, Float& value);

        /**
         * Sets the given value to the specified minimum value.
         * If \a value is not below \a min then \a value is not changed.
         * @param min          The limiting minimum value.
         * @param [out] value  The value.
         */
        static void SetToMin(Float min, Float& value);

        /**
         * Calculates the absolute value.
         * @param value     The value to compute the absolute value from.
         * @return          The absolute value of the value given.
         */
        static UInt32 Absolute(Int32 value);

        /**
         * Calculates the greatest common divisor of two numbers.
         * As % is used internally, the sign of the result is undefined
         * when only one of the values is negative.
         * @param a         The first number.
         * @param b         The second number.
         * @return          Greatest common divisor of the two numbers.
         */
        static Int32 GreatestCommonDivisor(Int32 a, Int32 b);

        /**
         * Calculates the least common multiple of two numbers.
         * GreatestCommonDivisor is used internaly. See
         * GreatestCommonDivisor for comment on sign of the return value.
         * @param a         The first number.
         * @param b         The second number.
         * @return          Least common multiple of the two numbers.
         */
        static Int32 LeastCommonMultiple(Int32 a, Int32 b);

        /**
         * Calculates the absolute value.
         * @param value     The value to compute the absolute value from.
         * @return          The absolute value of the value given.
         */
        static Float Absolute(Float value);
        
        /**
         * Calculates the largest integer smaller then the value given.
         * @param value    The value to calculate the floor value from.
         * @return         The floor value of the value given.
         */
        static Float Floor(Float value);
        
        /**
         * Calculates the smallest integer larger then the value given.
         * @param value    The value to calculate the ceiling value from.
         * @return         The ceiling value of the value given.
         */
        static Float Ceil(Float value);

        /**
         * Compares two Float values.
         * @param a         The first value.
         * @param b         The second value.
         * @param relativeEpsilon, absoluteEpsilon   The accepted inaccuracy on comparison.
         *                                           If 99.999% accuracy is needed then pass a epsilon value of 0.00001.
         * @return          true if the given Float values are equal in respect to the given epsilon, otherwise false.
         */
        static bool FloatAlmostEqual(Float a, Float b, Float relativeEpsilon = EpsilonFloat(), Float absoluteEpsilon = EpsilonFloat3D());


        static Float Pi();
        static Float PiDouble();
        static Float PiHalf();
        static Float PiFourth();
        static Float PiInverse();
        static Float PiOver180();
        static Float PiOver360();
        static Float MaxFloat();
        static Float MinFloat();
        static Float EpsilonFloat();
        static Float EpsilonFloat3D();
        static Float UInt8Inverse();
        static Float EulerConstant();
    };

        /** @} */ // end of MathematicsSystem

    template<typename OperandType>
    const OperandType& Math::Minimum(const OperandType& a, const OperandType& b)
    {
        return (a < b) ? a : b;
    }

    template<typename OperandType>
    const OperandType& Math::Maximum(const OperandType& a, const OperandType& b)
    {
        return (a > b) ? a : b;
    }

    inline bool Math::IsNan(Float f) {
        return !((f<=0) || (f>0));
    }

    inline Float Math::SquareRoot(Float f)
    {
        return MathPlatform::SquareRoot(f);
    }

    inline Double Math::SquareRoot(Double d)
    {
        return MathPlatform::SquareRoot(d);
    }

    inline Float Math::DegreeToRadian(Float degree)
    {
        return MathPlatform::DegreeToRadian(degree);
    }

    inline Float Math::RadianToDegree(Float radian)
    {
        return MathPlatform::RadianToDegree(radian);
    }

    inline Float Math::Sine(Float val)
    {
        return MathPlatform::Sine(val);
    }

    inline Float Math::ASine(Float val)
    {
        if (val > 1.0F) {
            val = 1.0F;
        }
        if (val < -1.0F) {
            val = -1.0F;
        }
        return MathPlatform::ASine(val);
    }

    inline Float Math::Cosine(Float val)
    {
        return MathPlatform::Cosine(val);
    }

    inline Float Math::ACosine(Float val)
    {
        if (val > 1.0F) {
            val = 1.0F;
        }
        if (val < -1.0F) {
            val = -1.0F;
        }
        return MathPlatform::ACosine(val);
    }

    inline Float Math::Tangent(Float val)
    {
        return MathPlatform::Tangent(val);
    }

    inline Float Math::ATangent(Float val)
    {
        return MathPlatform::ATangent(val);
    }

    inline Float Math::ATangent2(Float y, Float x)
    {
        return MathPlatform::ATangent2(y,x);
    }

    inline Float Math::Power(Float base, Float exponent)
    {
        return MathPlatform::Power(base, exponent);
    }

    inline void Math::SetToMax(Float max, Float& value)
    {
        if (value > max) {
            value = max;
        }
    }

    inline void Math::SetToMin(Float min, Float& value)
    {
        if (value < min) {
            value = min;
        }
    }

    inline Float Math::Pi()
    {
        return 3.14159265358979323846F;
    }

    inline Float Math::PiDouble()
    {
        return 6.28318530717958623200F; // 2.0F * Pi
    }

    inline Float Math::PiHalf()
    {
        return 1.57079632679489655800F; // Pi / 2.0F
    }

    inline Float Math::PiFourth()
    {
        return 0.78539816339744827900F; // Pi / 4.0F
    }

    inline Float Math::PiInverse()
    {
        return 0.31830988618379069122F; // 1.0F / Pi
    }

    inline Float Math::PiOver180()
    {
        return 0.017453292519943295769236907684886F; // Pi / 180.0F
    }

    inline Float Math::PiOver360()
    {
        return 0.0087266462599716478846184538424431F; // Pi / 360.0F
    }

    inline Float Math::MaxFloat()
    {
        return MathPlatform::MaxFloat(); // Max number for Float
    }

    inline Float Math::MinFloat()
    {
        return MathPlatform::MinFloat(); // Min number for Float
    }

    inline Float Math::EpsilonFloat()
    {
        return MathPlatform::EpsilonFloat(); // Epsilon for Floats (smallest positive number for which 1 + epsilon != 1)
    }

    inline Float Math::EpsilonFloat3D()
    {
        return MathPlatform::EpsilonFloat3D(); // accepted imprecision for math functions 
    }

    inline Float Math::UInt8Inverse()
    {
        return 0.003921568627450980392156862745098F; // 1.0F / 255.0F;
    }

    inline Float Math::EulerConstant()
    {
        return 2.71828182845905F;
    }

    inline Int32 Math::GreatestCommonDivisor(Int32 a, Int32 b)
    {
        while(b != 0) {
            Int temp = b;
            b = a % b;
            a = temp;
        }

        return a;
    }

    inline Int32 Math::LeastCommonMultiple(Int32 a, Int32 b)
    {
        return a * (b / GreatestCommonDivisor(a, b));
    }

    inline UInt32 Math::Absolute(Int32 value)
    {
        return MathPlatform::Absolute(value);
    }

    inline Float Math::Absolute(Float value)
    {
        return MathPlatform::Absolute(value);
    }

    inline bool Math::FloatAlmostEqual(Float a, Float b, Float relativeEpsilon, Float absoluteEpsilon)
    {
        return MathPlatform::FloatAlmostEqual(a, b, relativeEpsilon, absoluteEpsilon);
    }

    inline Float Math::Floor(Float value)
    {
        return MathPlatform::Floor(value);
    }

    inline Float Math::Ceil(Float value)
    {
        return MathPlatform::Ceil(value);
    }


} // namespace Candera

#endif    // CANDERA_MATH_H
