//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_LOCALIZABLE_H)
    #define CANDERA_GLOBALIZATION_LOCALIZABLE_H

#include <Candera/System/GlobalizationBase/CultureChangeListener.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/AtomicOp.h>
#include <FeatStd/Platform/Thread.h>
#endif


namespace Candera { namespace Globalization { namespace Internal {

        /** @addtogroup CanderaGlobalizationLocalizable
         *  @{
         */

        /**
         *  Base class for translatable text.
         */
        class Localizable : public CultureChangeListener {
            FEATSTD_TYPEDEF_BASE(CultureChangeListener);

            public:
                typedef SizeType Id;

                /**
                 *  Constructor
                 *  @param  id      The unique identifier for the item.
                 */
                Localizable(Id id);

                /**
                 *  Destructor
                 */
                virtual ~Localizable() override;

                /**
                 *  Retrieves the unique ID from the text.
                 *  @return         The text ID.
                 */
                Id GetId() const { return m_id; }

            protected:

                /**
                 *  This method is invoked whenever the culture changes.
                 *  @param  culture     The new culture.
                 */
                virtual void OnCultureChanged(const Culture& culture) override;

#ifdef FEATSTD_THREADSAFETY_ENABLED
                virtual void Obtain()
                {
                    (void)FeatStd::Internal::AtomicOp::Inc(m_obtainCounter);
                }

                virtual void Release()
                {
                    (void)FeatStd::Internal::AtomicOp::Dec(m_obtainCounter);
                }

                virtual void WaitForRelease()
                {
                    while (!FeatStd::Internal::AtomicOp::TestAndSet(m_obtainCounter, 0, 0)) {
                        FeatStd::Internal::Thread::Sleep(5);
                    }
                }

                FeatStd::Internal::AtomicOp::Atomic m_obtainCounter;
#endif
        private:
            Id m_id;
        };

        /// @}

    }

    typedef Internal::Localizable::Id LocalizableId;

}}   // namespace Candera::Globalization::Internal

#endif  // CANDERA_GLOBALIZATION_LOCALIZABLE_H
