//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_LANGUAGEPACK_H)
    #define CANDERA_GLOBALIZATION_LANGUAGEPACK_H

#include <Candera/Environment.h>
#include <Candera/System/GlobalizationBase/Localizer.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {
    namespace Globalization {
        FEATSTD_LINT_SYMBOL(578, Candera::Globalization::Handle, "Handle is a WIN32 specific type. Hiding it is acceptable.")
        typedef const Char* Handle;

        /**
         *  Represents a language pack as it is loaded from the asset.
         *
         *  TODO: Add addtogroup after implementation in Internal if appropriate.
         */
        class LanguagePack {
            public:

                FEATSTD_TYPEDEF_SHARED_POINTER(LanguagePack);

                /**
                 *  Creates a language pack object.
                 *  Use Dispose() for deletion.
                 *  @param  hdl     Handle to the binary data which must be 4-byte-aligned.
                 *  @return         The pointer to the created object, null if the object could not be created.
                 */
                static SharedPointer Create(const Handle hdl);

                //void Dispose();

                const TChar* GetBaseString(LocalizableId id) const;
                Localizer::TextType GetTextType(LocalizableId id) const;

                const Char* GetLocale() const { return GetStringForOffset(GetCultureData().m_localeNameOffset); }
                const TChar* GetDisplayName() const { return reinterpret_cast<const TChar*>(GetStringForOffset(GetCultureData().m_cultureDisplayNameOffset)); }
                TextDirection GetTextDirection() const { return static_cast<TextDirection>(GetCultureData().m_textDirection); }

            protected:

            private:
                friend struct FeatStd::MemoryManagement::Internal::Destructor<LanguagePack>;

                struct Header {
                    UInt8 m_majorFileTypeVersion;
                    UInt8 m_minorFileTypeVersion;
                    UInt16 m_fileTypeIdentifierOffset;
                    UInt16 m_encodingOffset;
                    UInt16 m_cultureOffset;
                    UInt16 m_cultureSize;
                    UInt16 m_textTableOffset;
                    UInt32 m_textTableSize;
                };

                struct CultureData {
                    UInt16 m_localeNameOffset;
                    UInt16 m_cultureDisplayNameOffset;
                    UInt16 m_int32FormatOffset;
                    UInt16 m_floatFormatOffset;
                    UInt16 m_booleanTrueFormatOffset;
                    UInt16 m_booleanFalseFormatOffset;
                    UInt8 m_textDirection;
                };

                struct TextTableHeader {
                    UInt32 m_textCount;

                    TextTableHeader() : m_textCount(0) {}
                };

                struct TextTableEntry {
                    UInt m_textId;
                    UInt m_textOffset : 24U;
                    UInt m_textType   :  8U;
                };

                Handle m_hdl;

                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Globalization::LanguagePack::LanguagePack, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
                LanguagePack(Handle hdl);
                ~LanguagePack();

                // disallow cloning
                LanguagePack(const LanguagePack&);
                LanguagePack& operator=(const LanguagePack&);

                void InitCulture();
                const TextTableEntry* GetEntry(LocalizableId id) const;

                const Header& GetHeader() const { return *FeatStd::aligned_cast<const Header*>(m_hdl); }

                Handle GetHandleForOffset(const UInt16& offset) const { return m_hdl + offset; }
                const Char* GetStringForOffset(const UInt16& offset) const {
                    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1780, "Address of reference is taken deliberateley, &offset is member of data structur, value of offset is added to retrieve string.")
                    return reinterpret_cast<const Char*>(&offset) + offset;
                }

                const CultureData& GetCultureData() const { return *FeatStd::aligned_cast<const CultureData*>(GetHandleForOffset(GetHeader().m_cultureOffset)); }

                const TextTableHeader& GetTextTableHeader() const { return *FeatStd::aligned_cast<const TextTableHeader*>(GetHandleForOffset(GetHeader().m_textTableOffset)); }

                const TextTableEntry* GetTextTableEntryBase() const  {
                    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(826, "Address of reference + 1 is taken deliberateley - the 'suspicious pointer-to-pointer conversion' is intentional and 'area too small' issue does not apply.")
                    return FeatStd::Internal::PointerToPointer<const TextTableEntry*>(&GetTextTableHeader() + 1);
                }
                const TextTableEntry& GetTextTableEntry(UInt32 index) const { return *(GetTextTableEntryBase() + index); }

                CANDERA_SHARED_POINTER_DECLARATION();
        };

    }   // namespace Globalization
}   // namespace Candera

#endif  // CANDERA_GLOBALIZATION_LANGUAGEPACK_H
