//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_DEFAULTCULTUREMANAGER_H)
    #define CANDERA_GLOBALIZATION_DEFAULTCULTUREMANAGER_H

#include <Candera/System/GlobalizationBase/CultureManager.h>

namespace Candera {
    namespace Globalization {
        /** @addtogroup CanderaGlobalizationCulture
         *  @{
         */

        class Localizer;

        /**
         *  API for accessing and changing the current culture as well as the list of available cultures.
         *  The currently set culture defines, together with other settings, the language and is therefore
         *  responsible for text translation.
         */
        class DefaultCultureManager : public CultureManager
        {
            public:
                virtual ~DefaultCultureManager() override;

                /**
                 *  Sets the current culture.
                 *  @param  locale      The locale string which implicitly defines the culture.
                 *  @return             true, if the culture has been set successfully.
                 */
                virtual bool SetCurrentCulture(const Char* locale) override;

                /**
                 *  Retrieves the current culture.
                 *  @return     The currently set culture.
                 */
                virtual Culture::SharedPointer GetCurrentCulture() const override;

                /**
                 *  Retrieves the amount of cultures which have been registered to the CultureManager.
                 *  @return     The count of currently available cultures.
                 */
                virtual UInt16 GetCultureCount() const override;

                /**
                 *  Retrieves a specific culture within the list of all registered cultures.
                 *  @param  index       The index in the list of available cultures.
                 *  @return             The culture with the specified index, 0 if this culture is not available.
                 */
                virtual Culture::SharedPointer GetCulture(UInt16 index) const override;

                /**
                 *  Retrieves a specific culture within the list of all registered cultures.
                 *  @param  locale      The locale string which implicitly defines the culture.
                 *  @return             The culture for locale, 0 if this culture is not available.
                 */
                virtual Culture::SharedPointer GetCulture(const Char* locale) const override;

                /**
                 * Sets the default culture. This is the culture returned,
                 * if the current culture is not set.
                 * @param  culture      The default culture.
                 */
                virtual void SetDefaultCulture(Culture::SharedPointer culture) override;

            protected:

            private:
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Globalization::DefaultCultureManager::DefaultCultureManager, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
                DefaultCultureManager();

                friend class Candera::Globalization::CultureManager;
        };

        /// @}
    }   // namespace Globalization
}   // namespace Candera

#endif  // CANDERA_GLOBALIZATION_DEFAULTCULTUREMANAGER_H
