//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_CULTUREMANAGER_H)
    #define CANDERA_GLOBALIZATION_CULTUREMANAGER_H

#include <Candera/Environment.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <Candera/System/GlobalizationBase/Culture.h>
#include <Candera/System/GlobalizationBase/CultureChangeListener.h>
#include <FeatStd/Platform/CriticalSection.h>

namespace Candera {
    namespace Globalization {

        /** @addtogroup CanderaGlobalizationCulture
         *  @{
         */

        /**
         *  API for accessing and changing the current culture as well as the list of available cultures.
         *  The currently set culture defines, together with other settings, the language and is therefore
         *  responsible for text translation.
         */
        class CultureManager {
            public:
                /**
                 *  Retrieve the CultureManager instance.
                 *  @return     The singleton instance.
                 */
                static CultureManager& GetInstance();

                /**
                 *  Sets the current culture.
                 *  @param  locale      The locale string which implicitly defines the culture.
                 *  @return             true, if the culture has been set successfully.
                 */
                virtual bool SetCurrentCulture(const Char* locale) = 0;

                /**
                 *  Retrieves the current culture.
                 *  @return     The currently set culture.
                 */
                virtual Culture::SharedPointer GetCurrentCulture() const = 0;

                /**
                 *  Retrieves the amount of cultures which have been registered to the CultureManager.
                 *  @return     The count of currently available cultures.
                 */
                virtual UInt16 GetCultureCount() const = 0;

                /**
                 *  Retrieves a specific culture within the list of all registered cultures.
                 *  @param  index       The index in the list of available cultures.
                 *  @return             The culture with the specified index, 0 if this culture is not available.
                 */
                virtual Culture::SharedPointer GetCulture(UInt16 index) const = 0;

                /**
                 *  Retrieves a specific culture within the list of all registered cultures.
                 *  @param  locale      The locale string which implicitly defines the culture.
                 *  @return             The culture for locale, 0 if this culture is not available.
                 */
                virtual Culture::SharedPointer GetCulture(const Char* locale) const = 0;

                /**
                 *  Adds a listener to the list.
                 *  @param  listener    The listener to be added.
                 */
                void AddCultureChangeListener(CultureChangeListener* listener);

                /**
                 *  Removes a listener from the list.
                 *  @param  listener    The listener to be removed;
                 *  @param  waitForListenerRelease
                 *  @return             true, if the listener could be removed successfully, otherwise false.
                 */
#ifdef FEATSTD_THREADSAFETY_ENABLED
                bool RemoveCultureChangeListener(CultureChangeListener* listener, bool waitForListenerRelease = true);
#else
                bool RemoveCultureChangeListener(CultureChangeListener* listener);
#endif
                
                /**
                 * Sets the default culture. This is the culture returned, 
                 * if the current culture is not set.
                 * @param  culture      The default culture.
                 */
                virtual void SetDefaultCulture(Culture::SharedPointer culture) { m_defaultCulture = culture; }

                /**
                 * Retrieves the default culture. This is the culture returned, 
                 * if the current culture is not set.
                 * @return               The default culture.
                 */
                Culture::SharedPointer GetDefaultCulture() const { return m_defaultCulture; }

            protected:
                CultureManager();
                virtual ~CultureManager();

                virtual void NotifyCultureChangeListeners(Culture::SharedPointer culture);

#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSection& GetCriticalSection() const;
#endif

            private:
                typedef FeatStd::Internal::SingleLinkedList<CultureChangeListener*> CultureChangeListenerList;

                void NotificationLoopBegin();
                void NotificationLoopEnd();
                bool IsListenerRemovalAllowed() const
                {
                    return 0 == m_notificationDepth;
                }

                Culture::SharedPointer m_defaultCulture;
                CultureChangeListenerList m_cultureChangeListenerList;
                CultureChangeListenerList m_pendingCultureChangeListenerList;
                UInt m_notificationDepth;
        };

        /// @}

    }   // namespace Globalization
}   // namespace Candera

#endif  // CANDERA_GLOBALIZATION_CULTUREMANAGER_H
