//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CultureManager.h"
#include <Candera/System/GlobalizationBase/Localizer.h>
#include <CanderaPlatform/OS/StringPlatform.h>
#if defined(FEATSTD_THREADSAFETY_ENABLED)
#include <FeatStd/Platform/CriticalSectionLocker.h>
#include <FeatStd/Util/StaticObject.h>
#endif

namespace Candera {
namespace Globalization {

#ifdef FEATSTD_THREADSAFETY_ENABLED

static FeatStd::Internal::CriticalSection& CultureManagerCriticalSection()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, s_cultureManagerCriticalSection);
    return s_cultureManagerCriticalSection;
}

static FeatStd::Internal::CriticalSection& CultureManagerRemoveListenerCriticalSection()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, s_cultureManagerRemoveListenerCriticalSection);
    return s_cultureManagerRemoveListenerCriticalSection;
}

static FeatStd::Internal::CriticalSection& s_forceInitCultureManagerCriticalSection = CultureManagerCriticalSection();


static FeatStd::Internal::CriticalSection& s_forceInitCultureManagerRemoveListenerCriticalSection = CultureManagerRemoveListenerCriticalSection();

FeatStd::Internal::CriticalSection& CultureManager::GetCriticalSection() const
{
    return CultureManagerCriticalSection();
}

class CultureChangeListenerLocker
{
public:
    CultureChangeListenerLocker(CultureChangeListener& listener) :
        m_listener(listener)
    {
        m_listener.Obtain();
    }

    ~CultureChangeListenerLocker()
    {
        m_listener.Release();
    }

private:
    CultureChangeListenerLocker();
    CultureChangeListenerLocker(const CultureChangeListenerLocker&);
    CultureChangeListenerLocker& operator=(const CultureChangeListenerLocker&);
    CultureChangeListener& m_listener;
};

#endif

/******************************************************************************
 *  Constructor
 ******************************************************************************/
CultureManager::CultureManager() :
    m_notificationDepth(0)
{
}

/******************************************************************************
 *  Destructor
 ******************************************************************************/
CultureManager::~CultureManager()
{
}

/******************************************************************************
 *  AddCultureChangeListener
 ******************************************************************************/
void CultureManager::AddCultureChangeListener(CultureChangeListener* listener)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    
    if (listener != 0) {
        if (IsListenerRemovalAllowed()) {
            static_cast<void>(m_cultureChangeListenerList.Append(listener));
        }
        else {
            static_cast<void>(m_pendingCultureChangeListenerList.Append(listener));
        }
    }
}

/******************************************************************************
 *  RemoveCultureChangeListener
 ******************************************************************************/
#ifdef FEATSTD_THREADSAFETY_ENABLED
bool CultureManager::RemoveCultureChangeListener(CultureChangeListener* listener, bool waitForListenerRelease)
#else
bool CultureManager::RemoveCultureChangeListener(CultureChangeListener* listener)
#endif
{
    bool isSuccessful = false;
    if (0 != listener) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
        if (IsListenerRemovalAllowed()) {
            isSuccessful = m_cultureChangeListenerList.Remove(listener);
        }
        else {
            for (CultureChangeListenerList::Iterator it = m_cultureChangeListenerList.Begin(); it != m_cultureChangeListenerList.End(); ++it) {
                if (listener == *it) {
                    *it = 0;
                    isSuccessful = true;
                }
            }
            static_cast<void>(m_pendingCultureChangeListenerList.Remove(listener));
        }
#ifdef FEATSTD_THREADSAFETY_ENABLED
        lock.Release();
        if (waitForListenerRelease) {
            listener->WaitForRelease();
        }
#endif
    }
    return isSuccessful;
}

/******************************************************************************
 *  NotifyCultureChangeListeners
 ******************************************************************************/
void CultureManager::NotifyCultureChangeListeners(Culture::SharedPointer culture)
{
    // NotifyCultureChangeListeners is not allowed to lock the critical section during OnCultureChanged itself.
    //
    // Reason (problem 1):
    // Thread 1 triggers NotifyCultureChangeListeners (locking sc1 => GetCriticalSection) and
    // locks an additional critical section (let's call it sc2) while in OnCultureChanged.
    // Before Thread 1 calls NotifyCultureChangeListeners another thread (let's call it thread 2) locks already sc2.
    // Therefore, thread 1 has to wait until thread 2 releases sc2. But, if thread 2 now triggers a call of NotifyCultureChangeListeners it
    // needs to lock sc1 which is already locked by thread 1. In this case we end in a deadlock situation.
    //
    // Solution:
    // By unlocking sc1 before calling OnCultureChanged and lock it again afterward we can easily prevent the deadlock situation and thread 2
    // can continue the notification, release afterward sc2 to allow thread 1 to continue its notification.
    //
    // Additional problem in case of destruction of a listener (problem 2):
    // Each listener has to be removed as listener before destruction.
    // So, if thread 1 is currently in OnCultureChanged and thread 2 decides to destruct this listener then the RemoveCultureChangeListener call is not blocked.
    // Thread 2 will return before thread 1 is finished with the call of OnCultureChanged and destroy the listener.
    //
    // Solution:
    // The CultureChangeListener will be informed to keep the instance alive within the scope of the CultureChangeListener method calls Obtain and Release.
    // The CultureChangeListener method WaitForRelease has to be called before destroying the CultureChangeListener instance. For convenience the
    // CultureChangeListener WaitForRelease method is called in CultureManager::RemoveCultureChangeListener if waitForListenerRelease is true.
    // Since the CultureChangeListener has to be removed from the CultureManager before its destruction the thread safety is obtained without further changes
    // than the implementation of the methods CultureChangeListener::Obtain, CultureChangeListener::Release and CultureChangeListener::WaitForRelease
    // in a thread safe way. Proposal: use atomic ops to implement an obtain count and wait for the release with Thread::Sleep(5).
    //
    // Remaining deadlock scenario (unpreventable by framework => has to be prevented in application code):
    // We consider problem 1 and add the removal of a listener to that scenario while thread 2 is calling OnCultureChanged.
    // In that case thread 1 will wait for cs2 in OnCultureChanged and has sc3 locked. Thread 2 is has locked cs2 and waits for cs3.
    // This scenario cannot be prevented by the framework because if thread 2 is not waiting for cs3 it will cause a crash of thread 1.
    // Therefore such a scenario has to be prevented in the application code. To do so the applications is not allowed to remove the listener while still
    // holding the lock of another critical section that may be locked by another listener.

#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    NotificationLoopBegin();
    // Before culture changed events
    for (CultureChangeListenerList::Iterator it = m_cultureChangeListenerList.Begin(); it != m_cultureChangeListenerList.End(); ++it) {
        CultureChangeListener* listener = *it;
        if (0 != listener) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            CultureChangeListenerLocker listenerLock(*listener);
            lock.Release();
#endif
            listener->OnPreCultureChanged(*culture);
#ifdef FEATSTD_THREADSAFETY_ENABLED
            lock.Obtain();
#endif
        }
    }

    // Culture changed events
    for (CultureChangeListenerList::Iterator it = m_cultureChangeListenerList.Begin(); it != m_cultureChangeListenerList.End(); ++it) {
        CultureChangeListener* listener = *it;
        if (0 != listener) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            CultureChangeListenerLocker listenerLock(*listener);
            lock.Release();
#endif
            listener->OnCultureChanged(*culture);
#ifdef FEATSTD_THREADSAFETY_ENABLED
            lock.Obtain();
#endif
        }
    }
    
    // After culture changed events
    for (CultureChangeListenerList::Iterator it = m_cultureChangeListenerList.Begin(); it != m_cultureChangeListenerList.End(); ++it) {
        CultureChangeListener* listener = *it;
        if (0 != listener) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            CultureChangeListenerLocker listenerLock(*listener);
            lock.Release();
#endif
            listener->OnPostCultureChanged(*culture);
#ifdef FEATSTD_THREADSAFETY_ENABLED
            lock.Obtain();
#endif
        }
    }

    NotificationLoopEnd();
}

void CultureManager::NotificationLoopBegin()
{
    ++m_notificationDepth;
}

void CultureManager::NotificationLoopEnd()
{
    --m_notificationDepth;
    if (IsListenerRemovalAllowed()) {
        static_cast<void>(m_cultureChangeListenerList.Remove(0, true)); // remove listeners which were removed during notification
        // add listeners which were added during notification
        for (CultureChangeListenerList::Iterator it = m_pendingCultureChangeListenerList.Begin(); it != m_pendingCultureChangeListenerList.End(); ++it) {
            static_cast<void>(m_cultureChangeListenerList.Append(*it));
        }
        m_pendingCultureChangeListenerList.Clear();
    }
}

}   // namespace Globalization
}   // namespace Candera
