//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_CULTURECHANGELISTENER_H)
    #define CANDERA_GLOBALIZATION_CULTURECHANGELISTENER_H

#include <Candera/Environment.h>
#include <Candera/System/Container/LinkedList.h>

namespace Candera {
    namespace Globalization {

        /** @addtogroup CanderaGlobalizationCulture
         *  @{
         */

        class Culture;

        /**
         *  Supports the publisher / subscriber pattern implementation for culture change notifications.
         */
        class CultureChangeListener {
            public:
                CultureChangeListener() {}
                virtual ~CultureChangeListener() {}

            protected:

                /**
                *  This method is called before the culture has been changed.
                *  @param  culture     The new culture.
                */
                virtual void OnPreCultureChanged(const Culture&) {  /* Method body for backward compatbility reasons */ }
                /**
                 *  This method is called whenever the culture has been changed.
                 *  @param  culture     The new culture.
                 */
                virtual void OnCultureChanged(const Culture& culture) = 0;

                /**
                *  This method is called after the culture has been changed.
                *  @param  culture     The new culture.
                */
                virtual void OnPostCultureChanged(const Culture&) { /* Method body for backward compatbility reasons */ }

#ifdef FEATSTD_THREADSAFETY_ENABLED
                /**
                 * The Obtain method will be called in a thread safe way before calling OnCultureChanged
                 *
                 * IMPORTANT NOTE:
                 * During OnCultureChanged the CultureManage holds no lock of any critical section to prevent deadlock situation.
                 * The CultureChangeListener implementation is responsible to prevent the destruction of the CultureChangeListener instance
                 * while OnCultureChanged is called. It is recommended to lock a critical section within Obtain and
                 * unlock the critical section within Release if no other mutex is locked within OnCultureChanged.
                 * Otherwise the CultureChangeListener implementation has to implement a thread safe obtain counter to enable waiting
                 * on the final release call.
                 */
                virtual void Obtain() = 0;

                /**
                 * The Obtain method will be called before OnCultureChanged in a thread safe way.
                 */
                virtual void Release() = 0;

                /**
                 * This method has to wait for the last call of Release:
                 *
                 * IMPORTANT NOTE:
                 * If an obtain counter is used to prevent destruction of the CultureChangeListener instance during an OnCultureChanged call
                 * then the WaitForRelease is only allowed to return as soon as the last Release call has been performed.
                 * The concrete CultureChangeListener is responsible for the exact implementation details.
                 */
                virtual void WaitForRelease() = 0;
#endif

            private:
                friend class CultureChangeListenerLocker;
                friend class CultureManager;
        };

        /// @}

    }   // namespace Globalization
}   // namespace Candera

#endif  // CANDERA_GLOBALIZATION_CULTURECHANGELISTENER_H
