//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GLOBALIZATION_CULTURE_H)
    #define CANDERA_GLOBALIZATION_CULTURE_H

#include <Candera/Environment.h>
#include <Candera/System/Container/LinkedList.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {
    namespace Globalization {

        /** @addtogroup CanderaGlobalizationCulture
         *  @{
         */

        /**
         *  Text direction enumerator.
         */
        enum TextDirection {
            _undefined  = 0,    ///< Undefined
            LeftToRight = 1,    ///< Left-to-right text direction.
            RightToLeft = 2     ///< Right-to-left text direction.
        };

        /**
         *  Provides information about a specific culture. The information includes the locale (e.g. "en_US")
         *  and the text direction (left-to-right or right-to-left).
         */
        class Culture {
            public:

                FEATSTD_TYPEDEF_SHARED_POINTER(Culture);

                /**
                 *  Destructor
                 */
                ~Culture();

                /**
                 *  Creates an instance of this class.
                 *  @return     Shared pointer to the created object.
                 */
                FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

                /**
                 *  Defines the locale for the Culture, e.g. "en_UK".
                 *  @param  locale  The locale string to be set.
                 */
                void SetLocale(const Char* locale) { m_locale = locale; }

                /**
                 *  Retrieve the locale string from the Culture.
                 *  @return     The locale string.
                 */
                const Char* GetLocale() const { return m_locale; }

                /**
                 *  Defines the display name for the Culture, e.g. "English".
                 *  @param  displayName     The display name string to be set.
                 */
                void SetDisplayName(const TChar* displayName) { m_displayName = displayName; }

                /**
                 *  Retrieve the display name string from the Culture.
                 *  @return     The display name string.
                 */
                const TChar* GetDisplayName() const { return m_displayName; }

                /**
                 *  Defines the text direction for the Culture.
                 *  @param  textDirection   The text direction.
                 */
                void SetTextDirection(TextDirection textDirection) { m_textDirection = textDirection; }

                /**
                 *  Retrieve the text direction from the Culture.
                 *  @return     The text direction.
                 */
                TextDirection GetTextDirection() const { return m_textDirection; }

            protected:

            private:
                friend class CultureManager;

                const Char* m_locale;
                const TChar* m_displayName;
                TextDirection m_textDirection;

                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Globalization::Culture::Culture, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
                Culture();
                Culture(const Culture&);
                Culture& operator=(const Culture&);

                CANDERA_SHARED_POINTER_DECLARATION();
        };


        /// @}

    }   // namespace Globalization
}   // namespace Candera

#endif  // CANDERA_GLOBALIZATION_CULTURE_H
