//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "EntitySystem.h"

#include <Candera/System/EntityComponentSystem/ComponentSystem.h>
#include <CanderaPlatform/OS/CanderaTypes.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {

namespace EntityComponentSystem {

EntitySystem::ComponentSystems& EntitySystem::s_forceInitComponentSystems = GetComponentSystems();
EntitySystem::ComponentSystems& EntitySystem::GetComponentSystems()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(ComponentSystems, componentSystems);
    return componentSystems;
}

EntitySystem::EntityComponents& EntitySystem::s_forceInitEntityComponents = GetEntityComponents();
EntitySystem::EntityComponents& EntitySystem::GetEntityComponents()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(EntityComponents, entityComponents);
    return entityComponents;
}

void EntitySystem::OnEvent(const Candera::Event& event)
{
    ComponentSystems& componentSystems = GetComponentSystems();
    for (UInt i = 0; i < static_cast<UInt>(componentSystems.Size()); ++i) {
        componentSystems[i]->OnEvent(event);
    }
}

bool EntitySystem::DeregisterEntity(const Entity* entity)
{
    ComponentSystems& componentSystems = GetComponentSystems();
    EntityComponents& entityComponents = GetEntityComponents();
    Components* components = entityComponents.Find(entity);
    if (0 != components) {
        for (UInt i = 0; i < static_cast<UInt>(components->Size()); ++i) {
            for (UInt j = 0; j < static_cast<UInt>(componentSystems.Size()); ++j) {
                if (componentSystems[j]->IsComponentTypeId((*components)[i].m_handle, (*components)[i].m_typeId)) {
                    componentSystems[j]->DeregisterEntity(entity);
                    break;
                }
            }
        }

        static_cast<void>(entityComponents.Remove(entity));
    }

    return false;
}

void EntitySystem::HeartBeat()
{
    ComponentSystems& componentSystems = GetComponentSystems();
    for (UInt i = 0; i < static_cast<UInt>(componentSystems.Size()); ++i) {
        componentSystems[i]->HeartBeat();
    }
}

void EntitySystem::Update()
{
    ComponentSystems& componentSystems = GetComponentSystems();
    for (UInt i = 0; i < static_cast<UInt>(componentSystems.Size()); ++i) {
        AbstractComponentSystem* const componentSystem = componentSystems[i];
        if (componentSystem->IsEnabled()) {
            componentSystem->Update();
        }
    }
}

bool EntitySystem::AttachComponent(const Entity* entity, const UInt32 handle, const TypeId typeId)
{
    EntityComponents& entityComponents = GetEntityComponents();
    Components* components = entityComponents.Find(entity);
    if (0 != components) {
        for (UInt i = 0; i < static_cast<UInt>(components->Size()); ++i) {
            const ComponentInfo& componentInfo = (*components)[i];
            if ((typeId == componentInfo.m_typeId) && (handle == componentInfo.m_handle)) {
                return false; // This component already is attached to this entity
            }
        }

        return components->Add(ComponentInfo(handle, typeId));
    }
    else {
        Components newComponents;
        if (newComponents.Reserve(1)) {
            if (newComponents.Add(ComponentInfo(handle, typeId))){
                return entityComponents.Insert(entity, newComponents);
            }
        }
    }

    return false;
}

bool EntitySystem::DetachComponent(const Entity* entity, const UInt32 handle, const TypeId typeId)
{
    EntityComponents& entityComponents = GetEntityComponents();
    Components* components = entityComponents.Find(entity);
    if (0 != components) {
        for (UInt i = 0; i < static_cast<UInt>(components->Size()); ++i) {
            const ComponentInfo& componentInfo = (*components)[i];
            if ((typeId == componentInfo.m_typeId) && (handle == componentInfo.m_handle)) {
                if (components->Size() > 1) {
                    return (components->Remove(i));
                }

                return entityComponents.Remove(entity);
            }
        }
    }

    return false;
}

struct ComponentSystemPriorityComparator
{
    bool operator()(const AbstractComponentSystem* a, const AbstractComponentSystem* b) const {
        return (a->GetPriority() > b->GetPriority());
    }
};

void EntitySystem::OnComponentSystemPriorityChanged()
{
    ComponentSystemPriorityComparator comparator;
    GetComponentSystems().Sort(comparator);
}

bool EntitySystem::Destroy(const TypeId typeId)
{
    ComponentSystems& componentSystems = GetComponentSystems();
    for (UInt i = 0; i < static_cast<UInt>(componentSystems.Size()); ++i) {
        if (componentSystems[i]->IsTypeOf(typeId)) {
            AbstractComponentSystem* componentSystem = componentSystems[i];
            bool success = componentSystems.Remove(i);
            FEATSTD_DELETE(componentSystem);

            if (0 == componentSystems.Size()) {
                // Free memory to avoid triggering reports of memory leak
                // in unit tests due EntitySystem's static nature.
                componentSystems.Free();
            }

            return success;
        }
    }

    return false;
}

AbstractComponentSystem* EntitySystem::Get(const TypeId typeId)
{
    ComponentSystems& componentSystems = GetComponentSystems();
    for (UInt i = 0; i < static_cast<UInt>(componentSystems.Size()); ++i) {
        if (componentSystems[i]->IsTypeOf(typeId)) {
            return componentSystems[i];
        }
    }

    return 0;
}

}

}
