//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_ENTITYCOMPONENTSYSTEM_COMPONENTPOINTER_H
#define CANDERA_ENTITYCOMPONENTSYSTEM_COMPONENTPOINTER_H

#include <Candera/System/EntityComponentSystem/ComponentSystem.h>

namespace Candera {

namespace EntityComponentSystem {

/** @addtogroup EntityComponentSystem
  *  @{
  */

/**
 * @brief  ComponentPointer encapsulates a pointer to a component in order to prohibit copying and storing the pointer.
 *         As a result, pointers to components can only be used in a local scope, and have to be resolved from their handles to
 *         avoid dangling pointers.
 *         For general information about the Entity Component System see EntityComponentSystem.h
 *
 * @tparam ComponentType  Type of the Component derived object this component pointer can resolve handles to pointers for.
 * @tparam EntityType     Type of the Entity derived object the associated ComponentType can attach to.
 */

template<typename ComponentType, typename EntityType>
class ComponentPointer
{
public:
    /**
     *  Constructor creating a pointer from a handle
     *  @param componentSystem  The system corresponding to the component type referred to by the handle.
     *  @param componentHandle  The handle identifying the component to create the pointer for.
     */
    inline ComponentPointer(ComponentSystem<ComponentType, EntityType>* componentSystem, ComponentHandle<ComponentType> componentHandle);

    /**
     *  Check if the pointer is 0.
     *  @return  True, if the pointer is 0.
     *           False, if the pointer is not 0.
     */
    bool IsNull() const { return (0 == m_component); }

    /**
     *  Dereferencing.
     *  Use the methods of the underlying instance like this:
     *  @code componentPointer->MemberCall(); @endcode
     *  or this:
     *  @code componentPointer->MemberVar = value; @endcode
     *  @return The dereferenced pointer to the component.
     */
    ComponentType* operator->() const { return m_component; }

    /**
     *  Compare the component pointers for equality.
     *  @param other  The second component pointer
     *  @return  True, if equal, false otherwise.
     */
    bool operator== (const ComponentPointer& other) const { return (m_component == other.m_component); }

    /**
     *  Compare the component pointers for inequality.
     *  @param other  The second component pointer
     *  @return  True, if not equal, false otherwise.
     */
    bool operator!= (const ComponentPointer& other) const { return (m_component == other.m_component); }

private:
    FEATSTD_MAKE_CLASS_UNCOPYABLE(ComponentPointer);
    ComponentType* m_component;
};

template<typename ComponentType, typename EntityType>
ComponentPointer<ComponentType, EntityType>::ComponentPointer(ComponentSystem<ComponentType, EntityType>* componentSystem, ComponentHandle<ComponentType> componentHandle)
{
    m_component = (0 != componentSystem) ? componentSystem->GetPointer(componentHandle) : 0;
}

/** @} */ // end of EntityComponentSystem

} // namespace EntityComponentSystem

} // namespace Candera

#endif
