//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_Diagnostics_ScreenshotTool_h)
#define Candera_Diagnostics_ScreenshotTool_h

#include <FeatStd/Event/EventListener.h>
#include <FeatStd/Event/EventSource.h>
#include <FeatStd/Event/Event.h>
#include <FeatStd/Util/Rtti.h>

#include <Candera/Environment.h>
#include <Candera/System/UpdateSystem/UpdateSystem.h>
#include <Candera/System/EntityComponentSystem/Entity.h>

namespace FeatStd { class String; }
namespace Candera {
    namespace Diagnostics {
        /**
        * The ScreenshotTool is a tool to create screenshots from the current render target (context).
        * The default ScreenshotTool is asynchronous. It triggers the creation of the screenshot.
        * A callback checks whether the screenshot is done or not. This is done by the UpdateSystem.
        * The class can be overridden to extend the feature set.
        * Possible changes:
        * - Add own callback check/dispatcher.
        * - Handle screenshots synchronously.
        * - The direct callback to handle a screenshot can be overridden.
        * Other: An own event listener can be added. It will be triggered as soon as the screenshot is done and after the
        *        internal callback to handle a screenshot has been called.
        * To override the dispatcher - implement an own "Initialize()" method - as there the UpdateSystem is attached.
        * 
        * Note: The screenshot seems to be vertically 'flipped'. This is intended because of the coordinate system.
        * Requirements: 3D and OpenGL (ES) 3.0 is in use
        */
        class ScreenshotTool : public EntityComponentSystem::Entity {
        public:
            typedef SizeType TScreenshotID;
        
            /**
            * ScreenshotData contains the resulting image buffer and the user defined identification for
            * further progress. It is not guaranteed that the buffer still exists after the callbacks and 
            * events were handled.
            */
            struct ScreenshotData
            {
                ScreenshotData(TScreenshotID const& id, Int width, Int height);

                TScreenshotID m_id; //< User-defined identification token  
                UInt8 * m_data; //< image buffer
                Int m_width; //< width of the image
                Int m_height; //< height of the image

                /**
                * Calculates the total size of the image buffer. 
                * Note: The screenshot was made in the RGBA format (4 Byte per pixel)
                * @return total size of the image buffer
                */
                Int GetDataSize() const
                {
                    return m_height * m_width * 4;
                }

            };

            /**
            * Take a screenshot with a custom ScreenshotTool.
            * @param customScreenshotTool
            * @param id Is a token for the screenshot identification as the screenshots are taken asynchronously.
            *           It has no impact on ScreenshotTool itself. It is up to the user of how this value can be used.
            */
            static void TakeScreenshot(ScreenshotTool& customScreenshotTool, TScreenshotID const& id);

            /**
            * Take a screenshot with the default ScreenshotTool instance. The default instance handles screenshots asynchronously and 
            * is attached to the UpdateSystem.
            * @param id Is a token for the screenshot identification as the screenshots are taken asynchronously.
            *           It has no impact on ScreenshotTool itself. It is up to the user of how this value can be used.
            */
            static void TakeScreenshot(TScreenshotID const& id);

            /**
            * HandleScreenshotCallback is called when the screenshot was made and the data can be handled.
            * This implies that HandleFinishedScreenshots noticed the finished screenshot if screenshot were taken asynchronously
            * and no error occurred. This method allows to directly derive from ScreenshotTool and handle the data. No further event handling is 
            * required but possible.
            * It is not guaranteed that the image buffer still exists after the callback and bound events were handled.
            * @param data Contains the screenshotdata.
            * @return     True, if handling the data were successful otherwise false.
            */
            virtual bool HandleScreenshotCallback(ScreenshotData const& /*data*/) { return true; }
        
            /**
            * Destructs the ScreenshotTool and frees the allocated resources.
            */
            virtual ~ScreenshotTool();
           
            /**
            * Default ScreenshotTool instance. The screenshot is taken asynchronously. The callback to check whether the screenshot is done or not
            * is called during the Update() phase of the render loop.
            */
            static ScreenshotTool& GetInstance();

            /**
            *  Adds an EventListener to EventSource. Event listeners receive OnEvent calls.
            *  The OnEvent is called when the screenshot is done - regardless of being asynchronous or not.
            *  @see ScreenshotToolEvent for the event.
            *  The OnEvent is called after HandleScreenshotCallback was called.
            *  @param  eventListener  The event listener to be added.
            *  @return                True, if the listener could be added successfully. False, otherwise.
            */
            bool AddEventListener(FeatStd::EventListener* eventListener);

            /**
            *  Removes an EventListener from EventSource.
            *  @param  eventListener  The event listener to be removed.
            *  @param  waitForListenerRelease
            *  @return                True, if the listener could be removed successfully. False, otherwise.
            */
            bool RemoveEventListener(FeatStd::EventListener* eventListener, bool waitForListenerRelease = true);
            
            /**
            * Checks whether a pending screenshot is done or not and handles it accordingly.
            * This method is normally called by the underlying dispatching system.
            * When a user like to use their own dispatching system, it is suggested to derive from the ScreenshotTools and
            * override Attach/DetachDispatcher and call this function. A simpler use case is, call: DetachDispatcher to remove any
            * dispatcher and call HandleFinishedScreenshots when it is appropriate.
            * Note: This method has no return value which indicates whether a screenshot was handled or not. 
            * This is intended to support the UpdateSystem of the EntityComponentSystem.
            */
            void HandleFinishedScreenshots();

        protected:
    
            /**
            * Customized constructor
            * @param isAsync Defines whether the screenshot tool has to become async or not by the user
            */
            ScreenshotTool(bool isAsync);

            /**
            * Default constructor
            * When PBOs are available the screenshot will be taken asynchronously - otherwise it will be  synchronously
            */
            ScreenshotTool();

            /**
            * AttachDispatcher attaches the ScreenshotTool to the UpdateSystem.
            * This will be called by Initialize() which is called whilst taking the screenshot at latest.
            * @return True, if attaching the dispatcher was successful.
            */
            bool AttachDispatcher();

            /**
            * DetachDispatcher detaches the ScreenshotTool from the UpdateSystem.
            * This method is only here to detach the default dispatcher (UpdateSystem).
            * If it were never attached by calling Initialize() or AttachDispatcher() nothing happens.
            * @return True, if detaching the dispatcher was successful or no dispatcher was attached.
            */
            bool DetachDispatcher();

            /**
            * Initialize attaches the dispatcher. This method is called whilst taking the first screenshot. 
            * Further screenshots will not call this method.
            * If not the default dispatcher is used, this method has to be overridden, so it will not call 
            * AttachDispatcher() anymore.
            */
            virtual void Initialize();
        private:

            class ScreenshotPrivateData;
            FeatStd::EventSource m_eventSource;
            ScreenshotPrivateData* m_screenshotData[2];
            UpdateSystem::Handle m_updateHandle;
            UInt8 m_currentIndex;
            bool m_isAsync;
            bool m_isInitialized;

            bool IsTakeScreenshotFinished(UInt8 screenshotIdx) const;

            bool TakeScreenshotStart(TScreenshotID const& id, UInt8 screenshotIdx);

            bool TakeScreenshotResult(UInt8 screenshotIdx);


        };
        /**
        * ScreenshotToolEvent is the event type which will be received by all event listeners bound to a ScreenshotTool.
        */
        class ScreenshotToolEvent : public FeatStd::Event {
        public:
            FEATSTD_RTTI_DECLARATION();
            explicit ScreenshotToolEvent(ScreenshotTool::ScreenshotData const* data) :m_data(data) {}

            /**
            * Retrieves the event data. This contains the image buffer and the user defined id.
            * It is not guaranteed that the image buffer still exists after the event was handled.
            * @return Event data of the ScreenshotToolEvent.
            */
            ScreenshotTool::ScreenshotData const* GetScreenshotData() const
            {
                return m_data;
            }
        protected:
            ScreenshotTool::ScreenshotData const* m_data;
        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(ScreenshotToolEvent);
        };
    }
}


#endif // Candera_Diagnostics_ScreenshotTool_h
