//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_CONTROLLERSYSTEM_H
#define CANDERA_CONTROLLERSYSTEM_H

#include <Candera/System/EntityComponentSystem/ComponentSystem.h>
#include <Candera/System/ControllerSystem/ControllerComponent.h>

namespace Candera {

namespace EntityComponentSystem { class EntitySystem; }

/** @addtogroup ControllerSystem
 *  @{
 */

/**
 * @brief  The ControllerSystem is a component system that is globally accessible via the EntitySystem.
 *         It serves as a factory for user derived classes of the ControllerComponent. These derived
 *         classes can override Update functions. When instances of these classes are attached to an
 *         entity, their overridden functions will be called via the UpdateSystem.
 */

class ControllerSystem : public EntityComponentSystem::ComponentSystem<ControllerComponent, ControllerEntity>
{
    typedef EntityComponentSystem::ComponentSystem<ControllerComponent, ControllerEntity> Base;

public:
    FEATSTD_RTTI_DECLARATION();

    /**
     *  Destructor
     */
    virtual ~ControllerSystem() {}

    /**
     *  Create a component of type DerivedControllerComponent and return its handle.
     *  Note: DerivedControllerComponent must be derived from ControllerComponent otherwise compilation will fail.
     *  @return  The handle of the created component. If the creation failed, a null handle will be returned.
     */
    FEATSTD_LINT_NEXT_EXPRESSION(1511, "Member hides non-virtual member ComponentSystem<>::CreateComponent() [MISRA C++ Rule 2-10-2]."
        " This is on purpose, because CreateComponent of the base class cannot create abstract ControllerComponent instances.")
    template<typename DerivedControllerComponent>
    Handle CreateComponent();

    /**
     *  Set the priority of the update component.
     *  @param handle    The handle of the component to set the priority for.
     *  @param priority  The priority of the update component to be set.
     */
    bool SetComponentPriority(Handle handle, Float priority);

private:

    /**
     *  Constructor
     */
    ControllerSystem();

    friend class EntityComponentSystem::EntitySystem;

    void FixedUpdateDelegate();
    void UpdateDelegate(Double applicationTime, Double deltaTime);
    void LateUpdateDelegate();

    void SortComponents();
};

/** @} */ // end of ControllerSystem

template<typename DerivedControllerComponent>
ControllerSystem::Handle ControllerSystem::CreateComponent()
{
    FEATSTD_DEBUG_ASSERT(m_wasCreatedbyEntitySystem); // A component system must be created using EntitySystem::Create().

    Handle handle;
    ControllerComponent* component = FEATSTD_NEW(DerivedControllerComponent);
    if (0 != component) {
        handle = CreateHandle(component);
        if (handle.IsNullHandle()) {
            FEATSTD_DELETE(component);
            return handle;
        }

        if (!m_components.Add(component)) {
            Remove(handle);
            handle = Handle();
            FEATSTD_DELETE(component);
            component = 0;
        }
    }

    return handle;
}

} // namespace Candera

#endif
