//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MAP_H)
#define CANDERA_MAP_H

#include <Candera/Environment.h>
#include <FeatStd/Util/RBTree.h>
#include <FeatStd/MemoryManagement/Heap.h>
#include <FeatStd/MemoryManagement/Allocator.h>

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
    #define CANDERA_MAP_ALLOC(nBytes)    Allocator::Alloc(FEATSTD_MEMORYPOOL_CALL_CONTEXT(nBytes))
#else
    #define CANDERA_MAP_ALLOC(nBytes)    Allocator::Alloc(nBytes, FILE_LINE)
#endif
#define CANDERA_MAP_FREE(pData) Allocator::Free(pData)

namespace Candera {
    namespace Internal {

/** @addtogroup CommonBase
 *  @{
 */

/**
 * @brief This class defines the default comparing function used with Map.
 * @tparam Key The type of the compared values.
 */
template<typename Key>
struct MapLess {
    /** Comparator function for two keys. Compares left with right.
     *  @param left The left operand for comparison.
     *  @param right The right operand for comparison.
     *  @return 0 if left == right.
     *  @return -1 if left < right.
     *  @return 1 if left > right.
     */
    static Int32 Compare(const Key &left, const Key &right);
};
template <typename Key>
Int32 MapLess<Key>::Compare(const Key &left, const Key &right)
    {
        if (left < right) {
            return -1;
        }
        else if (right < left) {
            return 1;
        }
        else {
            return 0;
        }
    }

/**
 * @brief This is a container containing mappings between keys and values.
 *
 * Keys within the map are unique.
 * Key and value types need to provide nullary constructor and assignment
 * operator.
 *
 * @tparam Key The type of the key of the map entry. This may not be const.
 * @tparam Type The type of value of the map entry. This may not be const.
 * @tparam Comparator The type of the comparison object used to identify keys
 *         within the map. See MapLess for information on the required
 *         interface.
 * @tparam Allocator The type of the memory allocator used to create internal
 *         objects.
 */
template<
    typename Key,
    typename Type,
    typename Comparator = MapLess<Key>,
    typename Allocator = FeatStd::MemoryManagement::DefaultAllocator>
class Map
{
public:

    /**
     * @brief This is used for visiting the objects within the map.
     */
    class Visitor {
    public:
        /**
         * This function is called for each map entry during visiting.
         * @param key Key of the map entry.
         * @param value Value of the map entry. This may be modified by the
         *      visitor.
         * @return True if visiting shall continue, false to abort.
         */
        virtual bool Visit(const Key& key, Type& value) = 0;
    };

    /**
     * @brief This is used for visiting the objects within an immutable map.
     */
    class ConstantVisitor {
    public:
        /**
         * This function is called for each map entry during visiting.
         * @param key Key of the map entry.
         * @param value Immutable value of the map entry.
         * @return True if visiting shall continue, false to abort.
         */
        virtual bool Visit(const Key& key, const Type& value) = 0;
    };

    /**
     * Constructor.
     */
    Map() : 
        m_root(0)
    {
    }

    /**
     * Copy constructor.
     */
    Map(const Map<Key, Type, Comparator, Allocator>& source) : 
        m_root(0)
    {
        CopyFrom(source);
    }

    /**
     * Copy constructor from other type.
     */
    template <typename OtherComparator, typename OtherAllocator>
    Map(const Map<Key, Type, OtherComparator, OtherAllocator>& source) : 
        m_root(0)
    {
        CopyFrom(source);
    }

    /**
     * Destructor.
     */
    ~Map() 
    {
        Clear();
    }

    /**
     * Assignment operator.
     */
    Map<Key, Type, Comparator, Allocator>& operator = (
        const Map<Key, Type, Comparator, Allocator>& source)
    {
        if (&source != this) {
            Clear();
            CopyFrom(source);
        }
        return *this;
    }

    /**
     * Assignment operator from other type.
     */
    template <typename OtherComparator, typename OtherAllocator>
    Map<Key, Type, Comparator, Allocator>& operator = (
        const Map<Key, Type, OtherComparator, OtherAllocator>& source)
    {
        Clear();
        CopyFrom(source);
        return *this;
    }

    /**
     * Insert an entry into the map. The entry will be inserted in a sorted
     * manner.
     * If an entry with the same key already exists its value is preserved, and
     * insertion fails.
     * @param key The key by which the entries are sorted.
     * @param value The value assigned to the entry.
     * @return True if a new entry has been added. False if a memory allocation
     *      failed, or if the key was already present. Use Find to differentiate
     *      between the two.
     */
    bool Insert(const Key& key, const Type& value)
    {
        NodeSelector selector(key);
        MapNode* node = CastNode(
            FeatStd::Internal::RBTree::Functions::Find(m_root, selector));
        if (node != 0) {
            return false;
        }

        return InsertImpl(key, value);
    }

    /**
     * Update an entry from the map. If the entry key is not found, 
     * the entry will be inserted in a sorted manner.
     * @param key The key of the entry to be updated.
     * @param value The value assigned to the entry.
     * @return True if a new entry has been added. False if a memory allocation
     *      failed, or if the key was already present. Use Find to differentiate
     *      between the two.
     */
    bool Update(const Key& key, const Type& value)
    {
        NodeSelector selector(key);
        MapNode* node = CastNode(
            FeatStd::Internal::RBTree::Functions::Find(m_root, selector));
        if (node != 0)
        {
            node->m_value = value;
            return true;
        }

        return InsertImpl(key, value);
    }

    /**
     * Find an entry with a given key.
     * @param key The key to search for.
     * @return A pointer to the value within the map if the the key was found.
     *         0 otherwise.
     */
    const Type* Find(const Key& key) const { return FindValue(key); }
    Type* Find(const Key& key) { return FindValue(key); }

    /**
     * Remove the entry with the give key.
     * @param key The key of the entry to remove.
     * @return True if an entry was successfully removed. False if no such entry
     *      was found.
     */
    bool Remove(const Key& key) 
    {
        NodeSelector selector(key);
        MapNode* node = CastNode(
            FeatStd::Internal::RBTree::Functions::Remove(m_root, selector));
        if (node == 0) {
            return false;
        }
        DestroyNode(node);
        return true;
    }

    /**
     * Clear all map entries.
     */
    void Clear()
    {
        FeatStd::Internal::RBTree::Functions::UnlinkAllNodes(
            m_root,
            &VisitNodeForDestruction,
            0);
    }

    /**
     * Filter map entries.
     * The return value of the visitor is interpreted as true keep, false
     * to release.
     * @param Visitor function that filters the nodes.
     */
    void Filter(Visitor& visitor)
    {
        FilterData filter = { 0, &visitor };
        FeatStd::Internal::RBTree::Functions::UnlinkAllNodes(
            m_root,
            &VisitNodeForRebuild,
            &filter);
        m_root = filter.m_root;
    }

    /**
     * Visit each entry within the map with the given visitor.
     * @return True if all entries have been visited, false on visitor abort.
     */
    bool Visit(Visitor& visitor)
    {
        return FeatStd::Internal::RBTree::Functions::Visit(
            m_root,
            &VisitNodeForVisting,
            &visitor);
    }

    /**
     * Visit each entry within an immutable map with the given visitor.
     * @return True if all entries have been visited, false on visitor abort.
     */
    bool Visit(ConstantVisitor& visitor) const
    {
        return FeatStd::Internal::RBTree::Functions::Visit(
            m_root,
            &VisitNodeForConstVisting,
            &visitor);
    }

private:
    typedef FeatStd::Internal::RBTree::Node RBTreeNode;

    /**
     * @brief Class that acts as map entry.
     *
     * Map entries are stored within an Red-Black tree.
     */
    struct MapNode : public RBTreeNode
    {
        Key m_key;
        Type m_value;

        MapNode(const Key& key, const Type& value) :
            RBTreeNode(),
            m_key(key),
            m_value(value)
        {
        }
    };

    // Casts from Red-Black node to Map node.
    static MapNode& CastNode(RBTreeNode& node) 
    {
        return static_cast<MapNode&>(node);
    }
    static const MapNode& CastNode(const RBTreeNode& node)
    {
        return static_cast<const MapNode&>(node);
    }

    static MapNode* CastNode(RBTreeNode* node) 
    {
        return static_cast<MapNode*>(node);
    }
    static const MapNode* CastNode(const RBTreeNode* node)
    {
        return static_cast<const MapNode*>(node);
    }

    bool InsertImpl(const Key& key, const Type& value)
    {
        MapNode* node = CreateNode(key, value);
        if (node == 0) {
            return false;
        }

        FeatStd::Internal::RBTree::Functions::Insert(
            m_root, 
            node,
            &CompareNodes);

        return true;
    }

    /**
     * @brief Class used for selecting nodes on removal and searching.
     */
    class NodeSelector : public FeatStd::Internal::RBTree::NodeSelector {
        public:
            NodeSelector(const Key& key) :
                m_key(&key)
            {
            }
            ~NodeSelector()
            {
                m_key = 0;
            }
            virtual Int32 operator()(const RBTreeNode &node)
            {
                return Comparator::Compare(*m_key, CastNode(node).m_key);
            }
        private:
            const Key* m_key;
    };

    /**
     * Function used for comparing nodes on insertion.
     */
    static Int32 CompareNodes(const RBTreeNode& left, const RBTreeNode& right)
    {
        return Comparator::Compare(CastNode(left).m_key, CastNode(right).m_key);
    }

    /**
     * Function used for destroying nodes during clearing.
     */
    static bool VisitNodeForDestruction(RBTreeNode& node, void* /*userData*/)
    {
        DestroyNode(&CastNode(node));
        return true;
    }

    /**
     * @brief Structure used to store filter data.
     */
    struct FilterData {
        RBTreeNode* m_root;
        Visitor* m_visitor;
    };

    /**
     * Function used for filtering node base on a visitor return value.
     */
    static bool VisitNodeForRebuild(RBTreeNode& node, void* userData)
    {
        FilterData* data = static_cast<FilterData*>(userData);

        if (data->m_visitor->Visit(CastNode(node).m_key, CastNode(node).m_value)){
            // Keep data.
            FeatStd::Internal::RBTree::Functions::Insert(
                data->m_root, 
                &node,
                &CompareNodes);
        }
        else {
            // Release data.
            DestroyNode(&CastNode(node));
        }
        return true;
    }

    /**
     * Function used for visiting nodes.
     */
    static bool VisitNodeForVisting(RBTreeNode& node, void* untypedVisitor)
    {
        Visitor* visitor = static_cast<Visitor*>(untypedVisitor);
        return visitor->Visit(CastNode(node).m_key, CastNode(node).m_value);
    }

    /**
     * Function used for visiting nodes.
     */
    static bool VisitNodeForConstVisting(RBTreeNode& node, void* untypedVisitor)
    {
        ConstantVisitor* visitor = static_cast<ConstantVisitor*>(untypedVisitor);
        return visitor->Visit(CastNode(node).m_key, CastNode(node).m_value);
    }

    /**
     * Create a new map node.
     */
    static MapNode* CreateNode(const Key& key, const Type& value)
    {
        MapNode *node = static_cast<MapNode*>(
            CANDERA_MAP_ALLOC(sizeof(MapNode)));
        if (node != 0) {
            FeatStd::MemoryManagement::Construct(node, MapNode(key, value));
        }
        return node;
    }

    /**
     * Release a map node created by CreateNode.
     */
    static void DestroyNode(MapNode* node)
    {
        if (node != 0) {
            FeatStd::MemoryManagement::Destruct(node);
            CANDERA_MAP_FREE(node);
        }
    }

    /**
     * @brief Visitor class that inserts visited values to another map.
     */
    template <typename OtherComparator, typename OtherAllocator>
    class CopyVisitor : 
        public Map<Key, Type, OtherComparator, OtherAllocator>::ConstantVisitor
    {
    public:
        CopyVisitor(Map<Key, Type, Comparator, Allocator>* map) : m_map(map) {}

        virtual bool Visit(const Key& key, const Type& value)
        {
            m_map->Insert(key, value);
            return true;
        }
    private:
        Map<Key, Type, Comparator, Allocator> *m_map;
    };

    /**
     * Copy values from another map.
     */
    template <typename OtherComparator, typename OtherAllocator>
    void CopyFrom(const Map<Key, Type, OtherComparator, OtherAllocator>& source)
    {
        CopyVisitor<OtherComparator, OtherAllocator> visitor(this);
        source.Visit(visitor);
    }

    /**
     * Same as Find, but breaks map constantness.
     */
    Type* FindValue(const Key& key) const
    {
        NodeSelector selector(key);
        MapNode* node = CastNode(
            FeatStd::Internal::RBTree::Functions::Find(m_root, selector));
        if (node != 0) {
            return &node->m_value;
        }
        return 0;
    }

    RBTreeNode* m_root;
};

/** @} */ // end of CommonBase

    } // namespace Internal
} // namespace Candera


#endif  // CANDERA_SHARED_OBJECT_CLONE_STRATEGY_H
