//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_FORWARDITERATOR_H)
#define CANDERA_FORWARDITERATOR_H

namespace Candera {
    /** @addtogroup CommonBase
     *  @{
     */

    /**
     * @brief Forward iterator wrapper.
     *
     * Forward iterator exposes the sequential access operators from within an
     * existing iterator implementation.
     * The internal iterator implementation needs to supply the following types:
     *   - ValueType - the type of the values iterated over by the iterator.
     *   - PointerType - the type of pointers to iterated values.
     *   - ReferenceType - the type of references to iterated values.
     * @tparam I Internal iterator that supports sequential access.
     */
template <typename I>
class ForwardIterator {
    public:
        typedef typename I::ValueType ValueType;
        typedef typename I::PointerType PointerType;
        typedef typename I::ReferenceType ReferenceType;

        ForwardIterator() {}

        /**
         * @param iterator Iterator whose sequential access operators are exposed.
         * @param endIterator Invalid iterator. When the current iterator has 
         *  reached the endIterator, it will become invalid.
         */
        ForwardIterator(const I& iterator, const I& endIterator);

        /**
         * @brief Check iterator validity.
         * 
         * @return whether the iterator is valid and can be dereferenced or not.
         */
        inline bool IsValid() const;

        /**
         * @brief Indirection operator.
         *
         * @return Referenced object.
         */
        inline ReferenceType operator*();

        /**
         * @brief Dereference operator.
         *
         * @return Dereferenced object.
         */
        inline PointerType operator->();

        /**
         * @brief Increment iterator.
         *
         * @return Iterator in the state found after being incremented.
         */
        ForwardIterator& operator++();

        /**
         * @brief Increment iterator.
         *
         * @return Iterator in the state found before being incremented.
         */
        ForwardIterator operator++(int);

        /**
         * @brief Equal to operator.
         *
         * @param other Operand to compare this iterator to.
         * @result true if operators are equal.
         * @result false otherwise.
         */
        bool operator==(const ForwardIterator<I>& other) const;

        /**
         * @brief Not equal to operator.
         *
         * @param other Operand to compare this iterator to.
         * @result false if operators are equal.
         * @result true otherwise.
         */
        bool operator!=(const ForwardIterator<I>& other) const;

    private:
        I m_internal;
        I m_end;
};

 /** @} */ // end of CommonBase

template <typename I>
ForwardIterator<I>::ForwardIterator(const I& iterator, const I& endIterator)
    :m_internal(iterator), m_end(endIterator)
{

}

template <typename I>
bool Candera::ForwardIterator<I>::IsValid() const
{
    return m_internal != m_end;
}

template <typename I>
typename ForwardIterator<I>::ReferenceType ForwardIterator<I>::operator*()
{
    return *m_internal;
}

template <typename I>
typename ForwardIterator<I>::PointerType ForwardIterator<I>::operator->()
{
    return &(*m_internal);
}

template <typename I>
ForwardIterator<I>& ForwardIterator<I>::operator++()
{
    ++m_internal;
    return *this;
}

template <typename I>
ForwardIterator<I> ForwardIterator<I>::operator++(int)
{
    ForwardIterator result = *this;
    ++m_internal;
    return result;
}

template <typename I>
bool ForwardIterator<I>::operator==(const ForwardIterator& other) const
{
    return (m_internal == other.m_internal);
}

template <typename I>
bool ForwardIterator<I>::operator!=(const ForwardIterator& other) const
{
    return !operator==(other);
}

}

#endif
