//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MACROS_H)
#define CANDERA_MACROS_H

#include <FeatStd/Config.h>
#include <FeatStd/Util/ConvenienceMacros.h>
#include <FeatStd/Util/WarningMacros.h>

/*
 * Macros dedicated to suppress warnings.
 */

#define CANDERA_SUPPRESS_LINT_FOR_SYMBOL(message_number, symbol, reason) /*lint -esym(message_number, symbol)*/

#define CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(message_number, reason) /*lint --e(message_number) */

#define CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(message_number, reason) /*lint --e{message_number}*/

#define CANDERA_SUPPRESS_LINT_FOR_MACRO(message_number, macroname, reason) /*lint -emacro(message_number, macroname)*/

#define CANDERA_LINT_DECLARE_INITIALIZER_FUNCTION(symbol) /*lint -sem(symbol, initializer )*/

#define CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(symbol) /*lint -sem(symbol, cleanup)*/

/// 534: Ignoring return value of function is without problems.
#define CANDERA_LINT_IGNORE_RETURN_VALUE(symbol) /*lint -esym(534, symbol)*/

/// For some reason the case block within a switch statement needs no break statement by design.
#define CANDERA_LINT_FALLTHROUGH() /*lint --e{1960} -fallthrough */

/******************************************************************************
 *  This is a list of shorthand reasons that can be given in certain typical
 *  situations (see also coding guidelines).
 ******************************************************************************/

/** The pointer member in question is not "class data", but an associated external
    object whose lifetime is not controlled by the object with the pointer member.
    The object having the pointer member does not own the associated object the
    pointer points to. The pointer is merely used to store the association to the
    (externally managed) referenced object and thus does not represent internal data.
    For the same reason the referencing object cannot be held responsible for
    managing the pointer member in any way upon destruction, copy-creation or assignment.
    Providing a non-const pointer to an associated object is OK because
    the const-correctness of the pointer-owning object is semantically not
    affected even if the associated object changes. */
#define CANDERA_LINT_REASON_ASSOCIATION

/** Instances of the class in question are not intended to be copied.
    There are strong semantic reasons (by the very nature of the class it would
    be an error to allow copying) or strong performance reasons (it would be
    prohibitively expensive to copy; or, copying would entail surprising extra
    memory consumption) against copying. */
#define CANDERA_LINT_REASON_NONCOPYABLE

/** A way of obtaining instances of the class is in fact provided, usually in the form
    of a creation function, or a function providing access to instances created elsewhere
    (as in applications of the Singleton pattern). */
#define CANDERA_LINT_REASON_INSTANCESOBTAINABLE

/** To access the type of an object at run-time in a unique way, a static method
    with fixed name "GetTypeId" is provided by every derived class in a class hierarchy. */
#define CANDERA_LINT_REASON_RTTI

/** The declaration "typedef <TypeOfBaseClass> Base" is used in a derived class
    to access it's base by a standard term, similar to "super" or "base" in
    other languages. Due to the nature of this declaration, it is statically
    used within the declaring class and may hide inherited "Base". */
#define CANDERA_LINT_REASON_BASEDECLARATION

/** The symbol in question is not declared as "const" by intention because it
    typically modifies the object in a direct, indirect, or virtual way. */
#define CANDERA_LINT_REASON_NONCONST

/** There is a LINT misinterpretation of a function call in a destructor.
    Lint analyzes it as a possible call to free resources, in fact the function
    server other purposes. The real disposing of resources happens elsewhere
    and is guaranteed anyway. */
#define CANDERA_LINT_REASON_DESTRUCTORMISINTERPRETATION

/** The hiding of a non-virtual function is intended by design. */
#define CANDERA_LINT_REASON_EXPLICITHIDING

/** The chosen operator is valid. */
#define CANDERA_LINT_REASON_OPERATORVALID

/** The enum to int conversion is done deliberately in a mapping table. */
#define CANDERA_LINT_REASON_ENUMCONVERSION

/** The comparison of this floating point numbers is chosen deliberately and is valid. */
#define CANDERA_LINT_REASON_FLOATCOMPARING

/** Usually, operator new cannot return 0 (it will throw an exception instead);
    so, checks against 0 are meaningless. However, it is possible to have an
    operator new that does return 0, justifying the 0 checks. */
#define CANDERA_LINT_REASON_CANDERANEW

/** The operand is used deliberately. */
#define CANDERA_LINT_REASON_OPERANDNULL

/** Lint reports 1788, no other usage of object besides constructor/destructor call but this
    is the intended use as the sole functionality is executed in constructor/destructor. */
#define CANDERA_LINT_REASON_CONSTRUCTORDESTRUCTOREFFECTS

/** The parameter is not used by the base class but it is required for sub classes.
    For documentation purposes the parameter shall be named in the member signature. */
#define CANDERA_LINT_REASON_PARAMETERRELEVANT

/** Deliberate test of boolean equality since this is the only way to efficiently
    achieve equivalence or exclusive or. */
#define CANDERA_LINT_REASON_BOOLEANEQUALITY

/** The break in a switch/case is omitted on purpose. It would be unreachable because of a return */
#define CANDERA_LINT_REASON_RETURNINCASE

/** The local declaration of typedef CdaMetaInfoTemplate<CdaMetaInfoTemplateBase> CdaMetaInfo; exposes
    the current Widgets meta info. It is meant by intention that exactly the MetaInfo of the
    dynamic type is exposed, similar to RTTI::GetTypeId. Therefore the dynamic class
    overwrites the CdaMetaInfo of it's base class. Meta Info of base class is furthermore 
    handled using Candera::MetaInfo::MetaInfoTypeExposer<baseClass>::CdaMetaInfo. */
#define CANDERA_LINT_REASON_WIDGET_CDAMETAINFO

/** Lint 665 requires macro parameters to be parenthesized if used inside an expression.
    This is of course very useful in the case described for message 665.
    However, in the suppressed case no expression occurs, moreover parentheses should 
    even be set, where C++ doesn't allow it. This is considered as false positive, so here this is suppressed.
 */
#define CANDERA_LINT_REASON_NO_PARENTHESES_ALLOWED

/**
 * Lint 1551 requires that function calls from a destructor have to be
 * surrounded by try{}catch{} block.This is not supported 
 * as there are compilers / compiler configurations where exception handling might
 * be disabled. 
 * Exemplary the MULTI Arm compiler for ARM Cortex R4 is given:
 * Default is setting --no_exceptions, which disables exception handling.
 * Exception handling is supported though only if flag --exceptions is set.
 */
#define CANDERA_LINT_REASON_EXCEPTIONS_NOT_SUPPORTED


/**
 * Lint 1735 requires that default parameters are used. In order to prevent unnecessary function overloads it 
 * can be intended that default paraemter are used to provive a parameterless version of the same function.
 */
#define CANDERA_LINT_REASON_DEFAULT_PARAMETER_REQUIRED

/**
 * Lint 1960 checks whether a chain of if/else-if statements is terminated by an else.
 * This is not required in cases when checking a value against boundary limits i.e.:
 *    if (value < minimal) {
 *        // reset/log/quit
 *    } else if (value > maximal) {
 *        // reset/log/quit
 *    } // No else { } is required here.
 */
#define CANDERA_LINT_REASON_BOUNDARY_CHECK_ON_VALUE

/** Macro helps to avoid unreferenced parameter warning with MSC-Warning Level 4 */
//#define CANDERA_UNUSED_PARAMETER(a) FEATSTD_UNUSED(a)
#if defined(CANDERA_DEBUG)
    #define CANDERA_DEBUG_UNUSED_PARAMETER(x) FEATSTD_UNUSED(x)
    #define CANDERA_RELEASE_UNUSED_PARAMETER(x)
#else
    #define CANDERA_DEBUG_UNUSED_PARAMETER(x)
    #define CANDERA_RELEASE_UNUSED_PARAMETER(x) FEATSTD_UNUSED(x)
#endif


/******************************************************************************
 *  macros to allow unit test class as friend of Candera classes
 *
 *  usage in .h file:
 *  @code
 *  CANDERA_UNIT_TEST_CLASS_DECLARATION(TestClass);     // optional, if needed
 *  CANDERA_UNIT_TEST_TESTCASE_DECLARATION(TestCaseName, TestName);
 *
 *  namespace Candera {
 *
 *      class A {
 *          ...
 *          private:
 *              CANDERA_UNIT_TEST_CLASS_FRIEND(TestClass);  // optional, if needed
 *              CANDERA_UNIT_TEST_TESTCASE_FRIEND(TestCaseName, TestName);
 *              ...
 *      };
 *
 *      ....
 *  }
 *  @endcode
 ******************************************************************************/
#define CANDERA_UNIT_TEST_CLASS_DECLARATION(className) \
    namespace Candera { \
        namespace UnitTest { \
            class className; \
        } \
    }

#define CANDERA_UNIT_TEST_TESTCASE_DECLARATION(testCaseName, testName) \
    namespace Candera { \
        namespace UnitTest { \
            class _FEATSTD_UNIT_TEST_CLASS_NAME(testCaseName, testName); \
        } \
    }

#define CANDERA_UNIT_TEST_CLASS_FRIEND(className) \
    friend class Candera::UnitTest::className

#define CANDERA_UNIT_TEST_TESTCASE_FRIEND(testCaseName, testName) \
    friend class Candera::UnitTest::_FEATSTD_UNIT_TEST_CLASS_NAME(testCaseName, testName)

/******************************************************************************
 *  CANDERA_DEPRECATED_[CURRENT_VERSION]
 *  Macro to mark a function as deprecated. Note, that this interface will be
 *  removed in future version!
 *
 *  Usage (example):
 *
 *  CANDERA_DEPRECATED_2_8_0("Please use NewMethod() instead.",
 *      void DeprecatedMethod()
 *  );
 *
 ******************************************************************************/

// Only this macro has to be used in Candera code and
// it should be removed if all according deprecated interfaces are removed.
#define CANDERA_DEPRECATED_3_0_0 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_0_2 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_1_1 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_2_0 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_2_1 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_3_0 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_4_0 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_4_2 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_5_0 _FEATSTD_DEPRECATED

#define CANDERA_DEPRECATED_3_6_0 _FEATSTD_DEPRECATED
#define CANDERA_DEPRECATED_FILE_3_6_0 _FEATSTD_DEPRECATED_FILE
#define CANDERA_DEPRECATED_CONST_3_6_0 _FEATSTD_DEPRECATED_CONST

// Deprecated warning for candera log macros, will be removed after 3.0.0 
#define CANDERA_LOG_DEPRECATED "DEPRECATED! - Please use FEAT_STD macros instead of CANDERA_LOG macros."

#endif    // CANDERA_MACROS_H
