//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_STACKLAYOUTER_H)
    #define CANDERA_STACKLAYOUTER_H

#include <Candera/EngineBase/Layout/Layouter.h>
#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>

namespace Candera {

/** @addtogroup Layout
 *  @{
 */

    /**
     *  @brief A StackLayouter stacks its elements either in horizontal or vertical order.
     *
     *  - Vertical stack:
     *      The preferred width of the stack is defined by the widest element.
     *
     *  - Horizontal stack:
     *      The preferred height of the stack is defined by the highest element.
     */
    class StackLayouter : public Layouter {
        FEATSTD_TYPEDEF_BASE(Layouter);

        public:
            /**
             *  Defines the layout arrangement.
             */
            enum Arrangement {
                Horizontal, ///< Horizontal stack: The preferred height of the stack is defined by the highest element.
                Vertical    ///< Vertical stack: The preferred width of the stack is defined by the widest element.
            };

            /**
             *  Creates an instance of this class.
             *  @return A pointer to the created object.
             */
            static StackLayouter* Create();

            /**
             *  Sets the arrangement of this layout.
             *  @param arrangement The arrangement to which the layout is set.
             */
            void SetArrangement(Arrangement arrangement);

            /**
             *  Retrieves the arrangement of this layout.
             *  @return The arrangement of this layout.
             */
            Arrangement GetArrangement() const { return m_arrangement; }

            /**
             *  Provides the parent of a dynamic property host for dynamic property value propagation.
             *  This function always returns 0, since layouters are no nodes and as such are
             *  not directly integrated in the child/parent relationships of scene graphs.
             *  This function is still required for general dynamic property support.
             *  @param host The dynamic property host, whose parent (considering property value
             *              propagation) is sought. This parameter is ignored.
             *  @return Always 0.
             */
            FEATSTD_LINT_NEXT_EXPRESSION(1511, "base class method made inaccessible with using statement")
            static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost * host) {
                FEATSTD_UNUSED(host);
                return 0;
            }

            // overrides Layouter::Clone
            virtual Layouter* Clone() const override;

        protected:
            StackLayouter();
            explicit StackLayouter(const StackLayouter& rhs);
            virtual ~StackLayouter() override;

            /**
            *  Implements virtual pure method from Layouter. @see Layouter::OnMeasure.
            *  @param node The node which child nodes shall be layouted.
            *  @param clientArea Area where layout shall be applied.
            *  @return Vector describing height and width of whole layouted rectangle.
            */
            virtual Vector2 OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea) override;

            /**
            *  Implements virtual pure method from Layouter. @see Layouter::OnArrange.
            *  @param node Node which children shall be arranged.
            *  @param clientArea Rectangle where layout shall be applied.
            */
            virtual void OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea) override;

        private:
            Arrangement m_arrangement;

            using Layouter::ParentProvider;

            /**
             * Dynamic property definitions.
             */
            CdaDynamicProperties(Candera::StackLayouter, Candera::Layouter);
            CdaDynamicPropertiesEnd();
    };

 /** @} */ // end of Layout

}   // namespace Candera

#endif  // CANDERA_STACKLAYOUTER_H
