//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StackLayouter.h"
#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Node2D.h>
#endif
#include <Candera/System/MemoryManagement/MemoryManagement.h>

#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"

namespace Candera {
    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    StackLayouter::StackLayouter() :
        m_arrangement(Horizontal)
    {
    }

    StackLayouter::StackLayouter(const StackLayouter& rhs) :
        Base(rhs),
        m_arrangement(rhs.m_arrangement)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    StackLayouter::~StackLayouter()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    StackLayouter* StackLayouter::Create()
    {
        return FEATSTD_NEW(StackLayouter);
    }

    /******************************************************************************
     *  SetArrangement
     ******************************************************************************/
    void StackLayouter::SetArrangement(Arrangement arrangement) 
    {
        // TODO invalidate scene layout
        m_arrangement = arrangement; 
    }

    /******************************************************************************
     *  OnMeasure
     ******************************************************************************/
    Vector2 StackLayouter::OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea)
    {
        Float max = 0.0F;
        Float sum = 0.0F;
        Vector2 childArea(clientArea);

        AbstractNodePointer child = node.GetFirstChild();
        while (child.IsValid()) {
            Layouter* childLayouter = child.GetLayouter();
            if (childLayouter != 0) {
                childLayouter->Measure(child, childArea);
                Vector2 preferredChildSize = childLayouter->GetClientSize(child);

                const Float width = preferredChildSize.GetX();
                const Float height = preferredChildSize.GetY();
                max = (Math::Maximum(max, (m_arrangement == Horizontal) ? height : width));
                sum += (m_arrangement == Horizontal) ? width : height;

                if (m_arrangement == Horizontal) {
                    childArea.SetX(childArea.GetX() - width);
                }
                else {
                    childArea.SetY((childArea.GetY() - height));
                }
            }

            child = child.GetNextSibling();
        }

        if (m_arrangement == Horizontal) {
            return Vector2(sum, max);
        }
        else {
            return Vector2(max, sum);
        }
    }

    /******************************************************************************
     *  OnArrange
     ******************************************************************************/
    void StackLayouter::OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea)
    {
        Node2D* node2D = node.ToNode2D();
        if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
            /** ---- Patched Layout ---- **/

            const Float clientWidth = clientArea.GetWidth();
            const Float clientHeight = clientArea.GetHeight();
            Float childMaxWidth = (clientWidth >= (Math::MaxFloat() - 1.0F)) ? GetClientSize(node).GetX() : clientWidth;
            Float childMaxHeight = (clientHeight >= (Math::MaxFloat() - 1.0F)) ? GetClientSize(node).GetY() : clientHeight;
            const LayoutAlignment::LayoutDirection::Enum orientation = GetLanguageSensitiveDirection(node);
            Float childLeft = Float(0);
            if ((m_arrangement == Horizontal) && (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection)) {
                childLeft = GetPreferredSize(node).GetX();
                if (childLeft > clientWidth) {
                    childLeft = clientWidth;
                }
            }
            Float childTop = 0.0F;
            Float actualWidth = 0.0F;
            Float actualHeight = 0.0F;

            AbstractNodePointer child = node.GetFirstChild();
            while (child.IsValid()) {
                Layouter* childLayouter = child.GetLayouter();
                if (childLayouter != 0) {
                    Vector2 clientSize = childLayouter->GetClientSize(child);

                    Float childWidth = (m_arrangement == Horizontal) ? clientSize.GetX() : clientWidth;
                    childWidth = Math::Minimum(childWidth, childMaxWidth);

                    Float childHeight = (m_arrangement == Vertical) ? clientSize.GetY() : clientHeight;
                    childHeight = Math::Minimum(childHeight, childMaxHeight);

                    if (m_arrangement == Horizontal) {
                        actualWidth += childWidth;
                        if (actualHeight < childHeight) {
                            actualHeight = childHeight;
                        }
                    }
                    else {
                        actualHeight += childHeight;
                        if (actualWidth < childWidth) {
                            actualWidth = childWidth;
                        }
                    }

                    if ((m_arrangement == Horizontal) && (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection)) {
                        childLeft -= childWidth;
                    }

                    Rectangle childArea(childLeft, childTop, childWidth, childHeight);
                    childLayouter->Arrange(child, childArea);

                    if (m_arrangement == Horizontal) {
                        if (orientation == LayoutAlignment::LayoutDirection::LeftToRightDirection) {
                            childLeft += childWidth;
                        }
                        childMaxWidth -= childWidth;
                    }
                    else {
                        childTop += childHeight;
                        childMaxHeight -= childHeight;
                    }
                }

                child = child.GetNextSibling();
            }
            SetArrangeActualSize(Vector2(actualWidth, actualHeight));
            /** ---- Patched Layout end ---- **/
        }
        else {
            /** ---- Old Layout ---- **/
            const Float clientWidth = clientArea.GetWidth();
            const Float clientHeight = clientArea.GetHeight();
            Float childMaxWidth = (clientWidth >= (Math::MaxFloat() - 1.0F)) ? GetClientSize(node).GetX() : clientWidth;
            Float childMaxHeight = (clientHeight >= (Math::MaxFloat() - 1.0F)) ? GetClientSize(node).GetY() : clientHeight;
            const LayoutAlignment::LayoutDirection::Enum orientation = GetLanguageSensitiveDirection(node);
            Float childLeft = ((m_arrangement == Horizontal) && (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection)) ? clientWidth : Float(0);
            Float childTop = 0.0F;

            Node2D* child = node2D->GetFirstChild();
            while (child != 0) {
                Layouter* childLayouter = child->GetLayouter();
                if (childLayouter != 0) {
                    Vector2 clientSize = childLayouter->GetClientSize(*child);

                    Float childWidth = (m_arrangement == Horizontal) ? clientSize.GetX() : clientWidth;
                    childWidth = Math::Minimum(childWidth, childMaxWidth);

                    Float childHeight = (m_arrangement == Vertical) ? clientSize.GetY() : clientHeight;
                    childHeight = Math::Minimum(childHeight, childMaxHeight);

                    if ((m_arrangement == Horizontal) && (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection)) {
                        childLeft -= childWidth;
                    }

                    Rectangle childArea(childLeft, childTop, childWidth, childHeight);
                    childLayouter->Arrange(*child, childArea);

                    if (m_arrangement == Horizontal) {
                        if (orientation == LayoutAlignment::LayoutDirection::LeftToRightDirection) {
                            childLeft += childWidth;
                        }
                        childMaxWidth -= childWidth;
                    }
                    else {
                        childTop += childHeight;
                        childMaxHeight -= childHeight;
                    }
                }

                child = child->GetNextSibling();
            }

            SetNodePosition(node, clientArea.GetPosition());
            /** ---- Old Layout end ---- **/
        }
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Layouter* StackLayouter::Clone() const
    {
        return CANDERA_NEW(StackLayouter)(*this);
    }

}   // namespace Candera
