//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MARGIN_H)
    #define CANDERA_MARGIN_H

#include <Candera/System/MetaInfo/DataType.h>

namespace Candera {

/** @addtogroup Layout
 *  @{
 */

    /**
     *  @brief A non-zero margin applies space outside the element layout's width and height.
     *  The values can be set separately for left, top, right and bottom so the margin can be asymmetric.
     */
    class Margin {
        public:
            /**
             *  Constructor
             */
            Margin() :
                m_left(0),
                m_top(0),
                m_right(0),
                m_bottom(0)
            {
            }

            /**
             *  One margin value for all sides.
             *  @param margin One margin for all sides.
             */
            Margin(Int16 margin) :
                m_left(margin),
                m_top(margin),
                m_right(margin),
                m_bottom(margin)
            {
            }

            /**
             *  Two separate margin values - one for left/right margin and one for top/bottom margin.
             *  @param leftRight Margin of left and right side.
             *  @param topBottom Margin of top and bottom side.
             */
            Margin (Int16 leftRight, Int16 topBottom) :
                m_left(leftRight),
                m_top(topBottom),
                m_right(leftRight),
                m_bottom(topBottom)
            {
            }

            /**
             *  Separate margin values for all sides.
             *  @param left   Margin of left side.
             *  @param top    Margin of top side.
             *  @param right  Margin of right side.
             *  @param bottom Margin of bottom side.
             */
            Margin (Int16 left, Int16 top, Int16 right, Int16 bottom) :
                m_left(left),
                m_top(top),
                m_right(right),
                m_bottom(bottom)
            {
            }

            /**
             *  Retrieves the left value.
             *  @return The left value.
             */
            Int16 GetLeft() const { return m_left; }

            /**
             *  Retrieves the top value.
             *  @return The top value.
             */
            Int16 GetTop() const { return m_top; }

            /**
             *  Retrieves the right value.
             *  @return The right value.
             */
            Int16 GetRight() const { return m_right; }

            /**
             *  Retrieves the buttom value.
             *  @return The buttom value.
             */
            Int16 GetBottom() const { return m_bottom; }

            /**
             *  Comparison operator for margin objects.
             *  @param m The margin to compare to this.
             *  @return true, if m equals this, false otherwise.
             */
            bool operator==(const Margin &m) const {
                return (m_left == m.m_left && m_right == m.m_right && m_top == m.m_top && m_bottom == m.m_bottom);
            }
            /**
            *  Comparison operator for margin objects.
            *  @param m The margin to compare to this.
            *  @return true, if m doesn't equals this, false if m equals this.
            */
            bool operator!=(const Margin &m) const {
                return (m_left != m.m_left || m_right != m.m_right || m_top != m.m_top || m_bottom != m.m_bottom);
            }

            /**
            *  Provides access to the properties by specifying an index.
            *  Index 0 returns left and accordingly.
            *  @param index The index of the desired element.
            *  @return The desired Element.
            */
            Int16& operator [] (UInt8 index);
            Int16 operator [] (UInt8 index) const;

        private:
            Int16 m_left;
            Int16 m_top;
            Int16 m_right;
            Int16 m_bottom;
    };

    inline Int16& Margin::operator [] (UInt8 index)
    {
        Int16* result = &m_left;
        switch (index) {
        case 0: { result = &m_left;  break; }
        case 1: { result = &m_top;  break; }
        case 2: { result = &m_right;  break; }
        case 3: { result = &m_bottom;  break; }
        default:
            FEATSTD_DEBUG_FAIL();
        }
        return *result;
    }

    inline Int16 Margin::operator [] (UInt8 index) const
    {
        Int16 result = 0;
        switch (index) {
        case 0: { result = m_left;  break; }
        case 1: { result = m_top;  break; }
        case 2: { result = m_right;  break; }
        case 3: { result = m_bottom;  break; }
        default:
            FEATSTD_DEBUG_FAIL();
        }
        return result;
    }

 /** @} */ // end of Layout
 
}   // namespace Candera

#endif  // CANDERA_MARGIN_H
