//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Layouter.h"

#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/Engine2D/Core/RenderDirectionHelper.h>
#endif
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/EngineBase/Layout/DefaultLayouter.h>
#include <FeatStd/Util/Optional.h>
#include <FeatStd/Util/StaticObject.h>

#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/Mesh2D.h>
#include <Candera/Engine2D/Effects/Effect2D.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
#include <Candera/Engine2D/Core/Camera2D.h>
#endif
#endif

#include <Candera/EngineBase/Layout/BoundingRectTraverser.h>

#if defined(CANDERA_GLOBALIZATION_ENABLED)
    #include <CanderaGlobalization/CultureManager.h>
#endif

#ifdef CANDERA_3D_ENABLED
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Group.h>
#ifdef CANDERA_3D_CANVAS_ENABLED
#include <Candera/Engine3D/Canvas/CanvasSprite.h>
#endif
#include <Candera/Engine3D/Mathematics/Math3D.h>
#endif


#include <Candera/EngineBase/Common/AbstractNodePointer.h>
#include <CanderaAssetLoader/AssetLoaderBase/OptionalDataType.h>

#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"

namespace Candera {
    CGI_WIDGET_RTTI_DEFINITION(ArabicLayouterPatchWidget);

namespace Internal { namespace LayoutEvents {
    FEATSTD_RTTI_DEFINITION(LayoutEvent, FeatStd::Event)
    FEATSTD_RTTI_DEFINITION(SizeLayoutEvent, LayoutEvent)
    FEATSTD_RTTI_DEFINITION(ArrangeEvent, SizeLayoutEvent)
} }

    class LayouterPrivateDynamicProperties
    {
    public:
        static bool IsLayoutValid(const Candera::CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(LayoutValid));
        }

        static bool IsLayoutValidSet(const Candera::CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(LayoutValid));
        }

        static bool ClearLayoutValid(Candera::CanderaObject& node)
        {
            return node.ClearValue(CdaDynamicPropertyInstance(LayoutValid));
        }

        static void SetLayoutValid(Candera::CanderaObject& node, bool layoutValid)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(LayoutValid), layoutValid));
        }

        static const Candera::Vector2& GetCachedPreferredSize(const Candera::CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(CachedPreferredSize));
        }

        static bool IsCachedPreferredSizeSet(const Candera::CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(CachedPreferredSize));
        }

        static bool ClearCachedPreferredSize(Candera::CanderaObject& node)
        {
            return node.ClearValue(CdaDynamicPropertyInstance(CachedPreferredSize));
        }

        static void SetCachedPreferredSize(Candera::CanderaObject& node, const Candera::Vector2& cachedPreferredSize)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(CachedPreferredSize), cachedPreferredSize));
        }

        static const Candera::Vector2& GetCachedMeasureSize(const Candera::CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(CachedMeasureSize));
        }

        static bool IsCachedMeasureSizeSet(const Candera::CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(CachedMeasureSize));
        }

        static bool ClearCachedMeasureSize(Candera::CanderaObject& node)
        {
            return node.ClearValue(CdaDynamicPropertyInstance(CachedMeasureSize));
        }

        static void SetCachedMeasureSize(Candera::CanderaObject& node, const Candera::Vector2& cachedMeasureSize)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(CachedMeasureSize), cachedMeasureSize));
        }

        static const Candera::Vector2& GetCachedArrangeActualSize(const Candera::CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(CachedArrangeActualSize));
        }

        static bool IsCachedArrangeActualSizeSet(const Candera::CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(CachedArrangeActualSize));
        }

        static bool ClearCachedArrangeActualSize(Candera::CanderaObject& node)
        {
            return node.ClearValue(CdaDynamicPropertyInstance(CachedArrangeActualSize));
        }

        static void SetCachedArrangeActualSize(Candera::CanderaObject& node, const Candera::Vector2& cachedArrangeActualSize)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(CachedArrangeActualSize), cachedArrangeActualSize));
        }

        static const Candera::Vector2& CachedArrangeActualSizeDefault()
        {
            static Candera::Vector2 s_cachedArrangeActualSizeDefault(-1.0F, -1.0F);
            return s_cachedArrangeActualSizeDefault;
        }

        static void SetActualSize(CanderaObject& node, const Vector2& size)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(ActualSize), size));
        }

        static const Vector2& GetActualSize(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(ActualSize));
        }

        typedef FeatStd::Optional<Vector2> RotatedPreferredSizeType;

        static bool SetRotatedPreferredSize(CanderaObject& node, const RotatedPreferredSizeType& rotatedPreferredSize) {
            return node.SetValue(CdaDynamicPropertyInstance(RotatedPreferredSize), rotatedPreferredSize);
        }

        static const RotatedPreferredSizeType& GetRotatedPreferredSize(const CanderaObject& node) {
            return node.GetValue(CdaDynamicPropertyInstance(RotatedPreferredSize));
        }

        static bool SetOriginalPreferredSize(CanderaObject& node, Vector2 dockSide) {
            return node.SetValue(CdaDynamicPropertyInstance(OriginalPreferredSize), dockSide);
        }

        static const Vector2& GetOriginalPreferredSize(const CanderaObject& node) {
            return node.GetValue(CdaDynamicPropertyInstance(OriginalPreferredSize));
        }

        static bool SetClippingArea(CanderaObject& node, Rectangle clippingArea) {
            return node.SetValue(CdaDynamicPropertyInstance(ClippingArea), clippingArea);
        }

        static const Rectangle& GetClippingArea(const CanderaObject& node) {
            return node.GetValue(CdaDynamicPropertyInstance(ClippingArea));
        }

        static const RotatedPreferredSizeType& RotatedPreferredSizeDefault()
        {
            static RotatedPreferredSizeType s_rotatedPreferredSizeDefault;
            return s_rotatedPreferredSizeDefault;
        }

        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host) {
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
                const CanderaObject* object = static_cast<const CanderaObject*>(host);
#ifdef CANDERA_2D_ENABLED
            const Node2D* node2D = Dynamic_Cast<const Node2D*>(object);
            if (0 != node2D) {
                return node2D->GetParent();
            }
#endif
#ifdef CANDERA_3D_ENABLED
            const Node* node = Dynamic_Cast<const Node*>(object);
            if (0 != node) {
                return node->GetParent();
            }
#endif
            return 0;
        }

    private:
        static Candera::Layouter* LayouterDefault()
        {
            return Candera::DefaultLayouter::GetInstance();
        }

        static const Candera::Vector2& CachedPreferredSizeDefault()
        {
            static Candera::Vector2 s_cachedPreferredSizeDefault(0.0F, 0.0F);
            return s_cachedPreferredSizeDefault;
        }

        static const Candera::Vector2& CachedMeasureSizeDefault()
        {
            static Candera::Vector2 s_cachedMeasureSizeDefault(-1.0F, -1.0F);
            return s_cachedMeasureSizeDefault;
        }

        static const bool& LayoutValidDefault()
        {
            static bool s_layoutValidDefault = false;
            return s_layoutValidDefault;
        }

        static const Candera::Rectangle& ClippingAreaDefault()
        {
            static Candera::Rectangle s_clippingAreaDefault(0.0F, 0.0F, -1.0F, -1.0F);
            return s_clippingAreaDefault;
        }

        CdaDynamicPropertiesDefinition(LayouterPrivateDynamicProperties, Candera::DynamicProperties::DynamicPropertyHost);
            CdaDynamicProperty(Layouter, Candera::Layouter*);
                CdaDynamicPropertyDefaultValue(LayouterPrivateDynamicProperties::LayouterDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(LayoutValid, bool);
                CdaDynamicPropertyDefaultValue(LayouterPrivateDynamicProperties::LayoutValidDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(CachedPreferredSize, Candera::Vector2);
                CdaDynamicPropertyDefaultValue(LayouterPrivateDynamicProperties::CachedPreferredSizeDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(CachedMeasureSize, Candera::Vector2);
                CdaDynamicPropertyDefaultValue(LayouterPrivateDynamicProperties::CachedMeasureSizeDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(CachedArrangeActualSize, Candera::Vector2);
                CdaDynamicPropertyDefaultValue(LayouterPrivateDynamicProperties::CachedArrangeActualSizeDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(ActualSize, Vector2);
                CdaDynamicPropertyDefaultValue(LayouterPrivateDynamicProperties::CachedMeasureSizeDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(RotatedPreferredSize, RotatedPreferredSizeType);
                CdaDynamicPropertyDefaultValue(LayouterPrivateDynamicProperties::RotatedPreferredSizeDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(OriginalPreferredSize, Candera::Vector2);
                CdaDynamicPropertyDefaultValue(CachedPreferredSizeDefault());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(ClippingArea, Rectangle);
                CdaDynamicPropertyDefaultValue(ClippingAreaDefault());
            CdaDynamicPropertyEnd();

        CdaDynamicPropertiesEnd();
    };

    class LayouterDynamicProperties
    {
    private:
        friend class Layouter;

        static void SetVerticalAlignment(CanderaObject& node, VerticalAlignment alignment)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(VerticalAlignment), alignment));
        }

        static VerticalAlignment GetVerticalAlignment(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(VerticalAlignment));
        }

        static void SetHorizontalAlignment(CanderaObject& node, HorizontalAlignment horizontalAlignment)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(HorizontalAlignment), horizontalAlignment));
        }

        static HorizontalAlignment GetHorizontalAlignment(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(HorizontalAlignment));
        }

        static const Vector2& GetSize(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(Size));
        }

        static void SetSize(CanderaObject& node, const Vector2& size)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(Size), size));
        }

        static bool IsSizeSet(const CanderaObject &node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(Size));
        }

        static const Vector2& GetMinimumSize(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(MinimumSize));
        }

        static void SetMinimumSize(CanderaObject& node, const Vector2& minSize)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(MinimumSize), minSize));
        }

        static const Vector2& GetMaximumSize(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(MaximumSize));
        }

        static bool IsMaximumSizeSet(const CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(MaximumSize));
        }

        static void SetMaximumSize(CanderaObject& node, const Vector2& maxSize)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(MaximumSize), maxSize));
        }

        static const Margin& GetMargin(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(Margin));
        }

        static void SetMargin(CanderaObject& node, const Margin& margin)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(Margin), margin));
        }

        static bool IsMarginSet(const CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(Margin));
        }

        static void SetStretchBehavior(CanderaObject& node, LayoutAlignment::StretchBehavior::Enum stretchBehavior)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(StretchBehavior), stretchBehavior));
        }

        static LayoutAlignment::StretchBehavior::Enum GetStretchBehavior(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(StretchBehavior));
        }

        static void SetLayoutDirection(CanderaObject& node, LayoutAlignment::LayoutDirection::Enum direction)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(LayoutDirection), direction));
        }

        static LayoutAlignment::LayoutDirection::Enum GetLayoutDirection(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(LayoutDirection));
        }

        static bool IsLayoutDirectionSet(const CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(LayoutDirection));
        }

        static void ClearLayoutDirection(CanderaObject& node)
        {
            static_cast<void>(node.ClearValue(CdaDynamicPropertyInstance(LayoutDirection)));
        }

        static void SetCollapsible(CanderaObject& node, bool collapsible)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(Collapsible), collapsible));
        }

        static bool IsCollapsible(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(Collapsible));
        }

        static bool IsCachedLayout(const Candera::CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(CachedLayout));
        }

        static bool IsCachedLayoutSet(const Candera::CanderaObject& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(CachedLayout));
        }

        static bool ClearCachedLayout(Candera::CanderaObject& node)
        {
            return node.ClearValue(CdaDynamicPropertyInstance(CachedLayout));
        }

        static void SetCachedLayout(Candera::CanderaObject& node, bool cachedLayout)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(CachedLayout), cachedLayout));
        }

        static const Padding& GetPadding(const CanderaObject& node) {
            return node.GetValue(CdaDynamicPropertyInstance(Padding));
        }

        static void SetPadding(CanderaObject& node, const Padding& padding) {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(Padding), padding));
        }


        CdaDynamicPropertiesDefinition(Candera::Layouter, CanderaObject);
            CdaDynamicProperty(VerticalAlignment, VerticalAlignment);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnVerticalAlignmentChanged);
                CdaDynamicPropertyDescription("The vertical alignment of this element in its parent.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(HorizontalAlignment, HorizontalAlignment);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnHorizontalAlignmentChanged);
                CdaDynamicPropertyDescription("The horizontal alignment of this element in its parent.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(MinimumSize, Vector2);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnSizeChanged);
                CdaDynamicPropertyDefaultValue(Layouter::MinimumSizeDefault());
                CdaDynamicPropertyDescription("The minimum size to use for this element.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(MaximumSize, Vector2);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnSizeChanged);
                CdaDynamicPropertyDefaultValue(Layouter::MaximumSizeDefault());
                CdaDynamicPropertyDescription("The maximum size to use for this element.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(Size, Vector2);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnSizeChanged);
                CdaDynamicPropertyDefaultValue(Layouter::SizeDefault());
                CdaDynamicPropertyDescription("Layout size. -1;-1 indicates that the width and height will be used from the bounding box.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(Margin, Margin);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnMarginChanged);
                CdaDynamicPropertyDefaultValue(Layouter::MarginDefault());
                CdaDynamicPropertyDescription("The margins of this element.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(Padding, Padding);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnPaddingChanged);
                CdaDynamicPropertyDescription("The padding of this element.")
                CdaDynamicPropertyDefaultValue(Layouter::PaddingDefault());
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(StretchBehavior, LayoutAlignment::StretchBehavior::Enum);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnStretchBehaviorChanged);
                CdaDynamicPropertyDescription("The stretching behavior of this element.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(LayoutDirection, LayoutAlignment::LayoutDirection::Enum);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnLayoutDirectionChanged);
                CdaDynamicPropertyDescription("The language direction environment used for layout (left to right, right to left, language sensitive, inherit (automatic).")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(Collapsible, bool);
                CdaDynamicPropertyValueChangedCb(&Layouter::OnCollapsibleChanged);
                CdaDynamicPropertyDefaultValue(false);
                CdaDynamicPropertyDescription("Specifies if the node should be collapsed if the effective rendering is disabled (if the layout is collapsed then no layout will be performed on any child node).")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(CachedLayout, bool);
            CdaDynamicPropertyValueChangedCb(&Layouter::OnCachedChanged);
                CdaDynamicPropertyDefaultValue(false);
                CdaDynamicPropertyDescription("Specifies if the layout of the node should be only performed once until a invalidation inside the subtree occures. Due to the requirement to cache some layout results this has an impact on memory consuption ")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

        CdaDynamicPropertiesEnd();
    };

    void Layouter::SetLayoutValid(Candera::CanderaObject& node, bool layoutValid)
    {
        LayouterPrivateDynamicProperties::SetLayoutValid(node, layoutValid);
    }


    void Layouter::SetActualSize(const AbstractNodePointer& node, const Vector2& size)
    {
        LayouterPrivateDynamicProperties::SetActualSize(*(node.ToCanderaObject()), size);
    }

    const Candera::Vector2& Layouter::GetActualSize(const AbstractNodePointer& node)
    {
        return LayouterPrivateDynamicProperties::GetActualSize(*(node.ToCanderaObject()));
    }


    /******************************************************************************
    * Workaround in order to check, whether a node contains an effect with a NinePatch configured Bitmap.
    * Temporary functionality - in the future every node shall not work with scale anymore but with actual size.
    * then this will become obsolete
    ******************************************************************************/
    bool Layouter::IsActualSizeAvailable(AbstractNodePointer const& node)
    {
#ifdef CANDERA_2D_ENABLED
        RenderNode* renderNode = Dynamic_Cast<RenderNode*>(node.ToNode2D());
        if (0 != renderNode) {
            Effect2D* effect = renderNode->GetEffect(0);
            if (0 != effect) {
                BitmapBrush* bitmapBrush = Dynamic_Cast<BitmapBrush*>(effect->GetBrushEffect2D());
                if (0 != bitmapBrush) {
                    if (renderNode->IsTypeOf(Mesh2D::GetTypeId())) {
                        return false;
                    }
                    else {
                        return bitmapBrush->IsNinePatch();
                    }
                }
            }
        }
#endif

#ifdef CANDERA_3D_CANVAS_ENABLED
        CanvasSprite* canvasSprite = Dynamic_Cast<CanvasSprite*>(node.ToNode());
        if (0 != canvasSprite) {
            const Bitmap::SharedPointer& ninePatchBitmap = canvasSprite->GetNinePatchBitmap();
            return ((ninePatchBitmap != 0) && (ninePatchBitmap->IsNinePatch()));
        }
#endif

        return false;
    }

    const Candera::DynamicProperties::PropertyHierarchyNode&  Layouter::GetPropertyHierarchy() const
    {
        return LayouterDynamicProperties::GetPropertyHierarchy();
    }

    const Rectangle& Layouter::GetClippingArea(const CanderaObject& node)
    {
        return LayouterPrivateDynamicProperties::GetClippingArea(node);

    }

    bool Layouter::SetClippingArea(CanderaObject& node, const Rectangle& clippingArea)
    {
        return LayouterPrivateDynamicProperties::SetClippingArea(node, clippingArea);
    }

    void Layouter::SetVerticalAlignment(CanderaObject& node, VerticalAlignment alignment)
    {
        LayouterDynamicProperties::SetVerticalAlignment(node, alignment);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetVerticalAlignment(Node2D& node, VerticalAlignment alignment)
    {
        LayouterDynamicProperties::SetVerticalAlignment(node, alignment);
    }
#endif

    VerticalAlignment Layouter::GetVerticalAlignment(const CanderaObject& node)
    {
        return LayouterDynamicProperties::GetVerticalAlignment(node);
    }

#ifdef CANDERA_2D_ENABLED
    VerticalAlignment Layouter::GetVerticalAlignment(const Node2D& node)
    {
        return LayouterDynamicProperties::GetVerticalAlignment(node);
    }
#endif

    void Layouter::SetHorizontalAlignment(CanderaObject& node, HorizontalAlignment horizontalAlignment)
    {
        LayouterDynamicProperties::SetHorizontalAlignment(node, horizontalAlignment);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetHorizontalAlignment(Node2D& node, HorizontalAlignment horizontalAlignment)
    {
        LayouterDynamicProperties::SetHorizontalAlignment(node, horizontalAlignment);
    }
#endif

    HorizontalAlignment Layouter::GetHorizontalAlignment(const CanderaObject& node)
    {
        return LayouterDynamicProperties::GetHorizontalAlignment(node);
    }

#ifdef CANDERA_2D_ENABLED
    HorizontalAlignment Layouter::GetHorizontalAlignment(const Node2D& node)
    {
        return LayouterDynamicProperties::GetHorizontalAlignment(node);
    }
#endif

    const Vector2& Layouter::GetSize(const CanderaObject& node)
    {
        return LayouterDynamicProperties::GetSize(node);
    }

#ifdef CANDERA_2D_ENABLED
    const Vector2& Layouter::GetSize(const Node2D& node)
    {
        return LayouterDynamicProperties::GetSize(node);
    }
#endif

    void Layouter::SetSize(CanderaObject& node, const Vector2& size)
    {
        LayouterDynamicProperties::SetSize(node, size);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetSize(Node2D& node, const Vector2& size)
    {
        LayouterDynamicProperties::SetSize(node, size);
    }
    void Layouter::SetSize(Node2D& node, Float width, Float height)
    {
        LayouterDynamicProperties::SetSize(node, Vector2(width, height));
    }
#endif

    bool Layouter::IsSizeSet(const CanderaObject &node)
    {
        return LayouterDynamicProperties::IsSizeSet(node);
    }

#ifdef CANDERA_2D_ENABLED
    bool Layouter::IsSizeSet(const Node2D &node)
    {
        return LayouterDynamicProperties::IsSizeSet(node);
    }
#endif

    const Vector2& Layouter::GetMinimumSize(const CanderaObject& node)
    {
        return LayouterDynamicProperties::GetMinimumSize(node);
    }

#ifdef CANDERA_2D_ENABLED
    const Vector2& Layouter::GetMinimumSize(const Node2D& node)
    {
        return LayouterDynamicProperties::GetMinimumSize(node);
    }
#endif

    void Layouter::SetMinimumSize(CanderaObject& node, const Vector2& minSize)
    {
        LayouterDynamicProperties::SetMinimumSize(node, minSize);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetMinimumSize(Node2D& node, const Vector2& minSize)
    {
        LayouterDynamicProperties::SetMinimumSize(node, minSize);
    }

    void Layouter::SetMinimumSize(Node2D& node, Float width, Float height)
    {
        LayouterDynamicProperties::SetMinimumSize(node, Vector2(width, height));
    }
#endif

    const Vector2& Layouter::GetMaximumSize(const CanderaObject& node)
    {
        return LayouterDynamicProperties::GetMaximumSize(node);
    }

#ifdef CANDERA_2D_ENABLED
    const Vector2& Layouter::GetMaximumSize(const Node2D& node)
    {
        return LayouterDynamicProperties::GetMaximumSize(node);
    }
#endif

    bool Layouter::IsMaximumSizeSet(const CanderaObject& node)
    {
        return LayouterDynamicProperties::IsMaximumSizeSet(node);
    }

#ifdef CANDERA_2D_ENABLED
    bool Layouter::IsMaximumSizeSet(const Node2D& node)
    {
        return LayouterDynamicProperties::IsMaximumSizeSet(node);
    }
#endif

    void Layouter::SetMaximumSize(CanderaObject& node, const Vector2& maxSize)
    {
        LayouterDynamicProperties::SetMaximumSize(node, maxSize);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetMaximumSize(Node2D& node, const Vector2& maxSize)
    {
        LayouterDynamicProperties::SetMaximumSize(node, maxSize);
    }
    void Layouter::SetMaximumSize(Node2D& node, Float width, Float height)
    {
        LayouterDynamicProperties::SetMaximumSize(node, Vector2(width, height));
    }
#endif

    const Margin& Layouter::GetMargin(const CanderaObject& node)
    {
        return LayouterDynamicProperties::GetMargin(node);
    }

#ifdef CANDERA_2D_ENABLED
    const Margin& Layouter::GetMargin(const Node2D& node)
    {
        return LayouterDynamicProperties::GetMargin(node);
    }
#endif

    void Layouter::SetMargin(CanderaObject& node, const Margin& margin)
    {
        LayouterDynamicProperties::SetMargin(node, margin);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetMargin(Node2D& node, const Margin& margin)
    {
        LayouterDynamicProperties::SetMargin(node, margin);
    }

    bool Layouter::IsMarginSet(const CanderaObject& node)
    {
        return LayouterDynamicProperties::IsMarginSet(node);
    }

#endif

    void Layouter::SetStretchBehavior(CanderaObject& node, LayoutAlignment::StretchBehavior::Enum stretchBehavior)
    {
        LayouterDynamicProperties::SetStretchBehavior(node, stretchBehavior);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetStretchBehavior(Node2D& node, LayoutAlignment::StretchBehavior::Enum stretchBehavior)
    {
        LayouterDynamicProperties::SetStretchBehavior(node, stretchBehavior);
    }
#endif

    LayoutAlignment::StretchBehavior::Enum Layouter::GetStretchBehavior(const CanderaObject& node)
    {
        return LayouterDynamicProperties::GetStretchBehavior(node);
    }

#ifdef CANDERA_2D_ENABLED
    LayoutAlignment::StretchBehavior::Enum Layouter::GetStretchBehavior(const Node2D& node)
    {
        return LayouterDynamicProperties::GetStretchBehavior(node);
    }
#endif

    void Layouter::SetLayoutDirection(CanderaObject& node, LayoutAlignment::LayoutDirection::Enum direction)
    {
        LayouterDynamicProperties::SetLayoutDirection(node, direction);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetLayoutDirection(Node2D& node, LayoutAlignment::LayoutDirection::Enum direction)
    {
        LayouterDynamicProperties::SetLayoutDirection(node, direction);
    }
#endif

    LayoutAlignment::LayoutDirection::Enum Layouter::GetLayoutDirection(const CanderaObject& node)
    {
        return LayouterDynamicProperties::GetLayoutDirection(node);
    }

#ifdef CANDERA_2D_ENABLED
    LayoutAlignment::LayoutDirection::Enum Layouter::GetLayoutDirection(const Node2D& node)
    {
        return LayouterDynamicProperties::GetLayoutDirection(node);
    }
#endif

    bool Layouter::IsLayoutDirectionSet(const CanderaObject& node)
    {
        return LayouterDynamicProperties::IsLayoutDirectionSet(node);
    }

#ifdef CANDERA_2D_ENABLED
    bool Layouter::IsLayoutDirectionSet(const Node2D& node)
    {
        return LayouterDynamicProperties::IsLayoutDirectionSet(node);
    }
#endif

    void Layouter::ClearLayoutDirection(CanderaObject& node)
    {
        LayouterDynamicProperties::ClearLayoutDirection(node);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::ClearLayoutDirection(Node2D& node)
    {
        LayouterDynamicProperties::ClearLayoutDirection(node);
    }
#endif

    void Layouter::SetCollapsible(CanderaObject& node, bool collapsible)
    {
        LayouterDynamicProperties::SetCollapsible(node, collapsible);
    }

#ifdef CANDERA_2D_ENABLED
    void Layouter::SetCollapsible(Node2D& node, bool collapsible)
    {
        LayouterDynamicProperties::SetCollapsible(node, collapsible);
    }
#endif

    bool Layouter::IsCollapsible(const CanderaObject& node)
    {
        return LayouterDynamicProperties::IsCollapsible(node);
    }

#ifdef CANDERA_2D_ENABLED
    bool Layouter::IsCollapsible(const Node2D& node)
    {
        return LayouterDynamicProperties::IsCollapsible(node);
    }
#endif

    using namespace Diagnostics;
    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine2D);

    const Float LAYOUT_MAX_VALUE=-1.0F;

    #if defined(CANDERA_GLOBALIZATION_ENABLED)
        using namespace Globalization;
    #endif

    /**************************************************************************************************
     Return node's current max size (w and h) OR 3.4028235e+038 if node's (w and h) are equal to (-1).
     Param node:   The node, whose maximum size.
     Return: 3.4028235e+038 or current value.
    **********************************************************************************************/
    static const Vector2 ReturnNodeMaximumSizeOrMaxFloat(const CanderaObject& node)
    {
            const Vector2& maximumSize = Layouter::GetMaximumSize(const_cast<CanderaObject&>(node));
        Vector2 size((maximumSize.GetX() < 0.0F) ? Math::MaxFloat() : maximumSize.GetX(),
                     (maximumSize.GetY() < 0.0F) ? Math::MaxFloat() : maximumSize.GetY());
        return size;
    }

     /******************************************************************************
     *  AutomaticDirectionBehavior
     ******************************************************************************/
    static LayoutAlignment::LayoutDirection::Enum& AutomaticDirectionBehavior() {
        FEATSTD_UNSYNCED_STATIC_OBJECT(LayoutAlignment::LayoutDirection::Enum, s_automaticDirectionBehavior, LayoutAlignment::LayoutDirection::InheritDirection);
        return s_automaticDirectionBehavior;
    }

    static LayoutAlignment::LayoutDirection::Enum s_forceAutomaticDirectionBehaviorInit = AutomaticDirectionBehavior();

    /******************************************************************************
     *  SetAutomaticDirectionBehavior
     ******************************************************************************/
    void Layouter::SetAutomaticDirectionBehavior(LayoutAlignment::LayoutDirection::Enum layoutDirection)
    {
        FEATSTD_DEBUG_ASSERT(layoutDirection != LayoutAlignment::LayoutDirection::AutomaticDirection);
        if (LayoutAlignment::LayoutDirection::AutomaticDirection == layoutDirection) {
            layoutDirection = LayoutAlignment::LayoutDirection::InheritDirection;
        }
        AutomaticDirectionBehavior() = layoutDirection;
    }

    /******************************************************************************
     *  GetAutomaticDirectionBehavior
     ******************************************************************************/
    LayoutAlignment::LayoutDirection::Enum Layouter::GetAutomaticDirectionBehavior()
    {
        return AutomaticDirectionBehavior();
    }


    /******************************************************************************
     *  GetEventSource
     ******************************************************************************/
    FeatStd::EventSource& Layouter::GetEventSource()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::EventSource, s_eventSource);
        return s_eventSource;
    }

    /******************************************************************************
     *  Layout
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::Layout(Node2D& node)
    {
        AbstractNodePointer abstractNode(&node);
        Layout(abstractNode);
    }
#endif

    void Layouter::Layout(const AbstractNodePointer& node)
    {
        if (node.IsValid()) {
            Vector2 nodeSize(Layouter::GetSize(*node.ToCanderaObject()));
            const Vector2 size((nodeSize.GetX() < 0.0F) ? Math::MaxFloat() : nodeSize.GetX(),
                (nodeSize.GetY() < 0.0F) ? Math::MaxFloat() : nodeSize.GetY());

            Internal::LayoutEvents::SizeLayoutEvent sizeLayoutEventBegin(Internal::LayoutEvents::LayoutEvent::LayoutBegin, node, size);
            GetEventSource().DispatchEvent(sizeLayoutEventBegin);
            Measure(node, size);
            Arrange(node, Rectangle(node.GetPosition(), size), true);
            Internal::LayoutEvents::SizeLayoutEvent sizeLayoutEventEnd(Internal::LayoutEvents::LayoutEvent::LayoutEnd, node, size);
            GetEventSource().DispatchEvent(sizeLayoutEventEnd);
        }
    }

    /******************************************************************************
     *  CalculatetAxisAlignedDimension
     ******************************************************************************/
    static void CalculatetAxisAlignedDimension(const Vector2& size, Float rotation, Float& w1, Float& w2, Float& h1, Float& h2)
    {
        const Float alpha = Math::DegreeToRadian(rotation);
        const Float cosAlpha = Math::Cosine(alpha);
        const Float sinAlpha = Math::Sine(alpha);

        const Float width = size.GetX();
        const Float height = size.GetY();

        w1 = width * cosAlpha;
        w2 = height * sinAlpha;

        h1 = height * cosAlpha;
        h2 = width * sinAlpha;
    }

    static Vector2 RotatedPoint(const Vector2& p, Float cosAlpha, Float sinAlpha)
    {
        return Vector2(p.GetX() * cosAlpha - p.GetY() * sinAlpha, p.GetX() * sinAlpha + p.GetY() * cosAlpha);
    }

    /******************************************************************************
     *  GetAxisAlignedOffset
     ******************************************************************************/
    static Vector2 GetAxisAlignedOffset(const Vector2& size, Float rotation, const Vector2& pivotPosition, const Vector2& scale)
    {
        if (rotation == 0.0F) {
            return Vector2();
        }
        if ((rotation > 180.0F) || (rotation < -180.0F)) {
            if (rotation > 0.0F) {
                rotation = Float(Int32((Int32(rotation) + 180) % 360) - 180) + (rotation - Float(Int32(rotation)));
            }
            else {
                rotation = -Float(Int32((Int32(-rotation) + 180) % 360) - 180) + (rotation - Float(Int32(rotation)));
            }
        }
        Vector2 offset;
        if ((pivotPosition.GetX() == 0.0F) && (pivotPosition.GetY() == 0.0F)) {
            Float w1 = 0.0F;
            Float w2 = 0.0F;
            Float h1 = 0.0F;
            Float h2 = 0.0F;
            CalculatetAxisAlignedDimension(size, rotation, w1, w2, h1, h2);
            if ((rotation >= 0.0F) && (rotation <= +90.0F)) {
                offset.SetX(w2);
                offset.SetY(0.0F);
            }
            else if ((rotation >= -90.0F) && (rotation <= 0.0F)) {
                offset.SetX(0.0F);
                offset.SetY(-h2);
            }
            else if ((rotation >= -180.0F) && (rotation <= -90.0F)) {
                offset.SetX(-w1);
                offset.SetY(-(h1 + h2));
            }
            else { //((rotation >= 180.0F) && (rotation <= 90.0F))
                offset.SetX(w2 - w1);
                offset.SetY(-h1);
            }
        }
        else {
            const Float alpha = Math::DegreeToRadian(rotation);
            const Float cosAlpha = Math::Cosine(alpha);
            const Float sinAlpha = Math::Sine(alpha);
            Float scaledPivotPositionX = pivotPosition.GetX();
            if ((0.0F != scaledPivotPositionX) && (1.0F != scale.GetX())) {
                scaledPivotPositionX *= scale.GetX();
            }
            Float scaledPivotPositionY = pivotPosition.GetY();
            if ((0.0F != scaledPivotPositionY) && (1.0F != scale.GetY())) {
                scaledPivotPositionY *= scale.GetY();
            }
            if ((rotation >= 0.0F) && (rotation <= +90.0F)) {
                Vector2 rotatedLeftBottom = RotatedPoint(Vector2(-scaledPivotPositionX, size.GetY() - scaledPivotPositionY), cosAlpha, sinAlpha);
                Vector2 rotatedLeftTop = RotatedPoint(Vector2(-scaledPivotPositionX, -scaledPivotPositionY), cosAlpha, sinAlpha);
                offset.SetX(-(rotatedLeftBottom.GetX() + pivotPosition.GetX()));
                offset.SetY(-(rotatedLeftTop.GetY() + pivotPosition.GetY()));
            }
            else if ((rotation >= -90.0F) && (rotation <= 0.0F)) {
                Vector2 rotatedLeftTop = RotatedPoint(Vector2(-scaledPivotPositionX, -scaledPivotPositionY), cosAlpha, sinAlpha);
                Vector2 rotatedRightTop = RotatedPoint(Vector2(size.GetX() - scaledPivotPositionX, -scaledPivotPositionY), cosAlpha, sinAlpha);
                offset.SetX(-(rotatedLeftTop.GetX() + pivotPosition.GetX()));
                offset.SetY(-(rotatedRightTop.GetY() + pivotPosition.GetY()));
            }
            else if ((rotation >= -180.0F) && (rotation <= -90.0F)) {
                Vector2 rotatedRightTop = RotatedPoint(Vector2(size.GetX() - scaledPivotPositionX, -scaledPivotPositionY), cosAlpha, sinAlpha);
                Vector2 rotatedRightBottom = RotatedPoint(Vector2(size.GetX() - scaledPivotPositionX, size.GetY() - scaledPivotPositionY), cosAlpha, sinAlpha);
                offset.SetX(-(rotatedRightTop.GetX() + pivotPosition.GetX()));
                offset.SetY(-(rotatedRightBottom.GetY() + pivotPosition.GetY()));
            }
            else { //((rotation >= 180.0F) && (rotation <= 90.0F))
                Vector2 rotatedRightBottom = RotatedPoint(Vector2(size.GetX() - scaledPivotPositionX, size.GetY() - scaledPivotPositionY), cosAlpha, sinAlpha);
                Vector2 rotatedLeftBottom = RotatedPoint(Vector2(-scaledPivotPositionX, size.GetY() - scaledPivotPositionY), cosAlpha, sinAlpha);
                offset.SetX(-(rotatedRightBottom.GetX() + pivotPosition.GetX()));
                offset.SetY(-(rotatedLeftBottom.GetY() + pivotPosition.GetY()));
            }
        }
        return offset;
    }

    /******************************************************************************
     *  GetAxisAlignedSize
     ******************************************************************************/
    static inline Vector2 GetAxisAlignedSize(const Vector2& size, Float rotation)
    {
        Float w1 = 0.0F;
        Float w2 = 0.0F;
        Float h1 = 0.0F;
        Float h2 = 0.0F;
        CalculatetAxisAlignedDimension(size, rotation, w1, w2, h1, h2);
        return Vector2(Math::Absolute(w1) + Math::Absolute(w2), Math::Absolute(h1) + Math::Absolute(h2));
    }

    static inline Vector2 LayouterGetAxisAlignedDimension(const Vector2& size, Float rotation)
    {
        if ((rotation == 0.0F) || (rotation == 180.0F) || (rotation == -180.0F)) {
            return size;
        }
        if ((rotation == 90.0F) || (rotation == -90.0F) || (rotation == 270.0F) || (rotation == -270.0F)) {
            return Vector2(size.GetY(), size.GetX());
        }
        return GetAxisAlignedSize(size, rotation);
    }

    static inline Float NormalizeRotation(Float rotation)
    {
        if (rotation < 0.0F) {
            return 360.0F - NormalizeRotation(-rotation);
        }
        if (rotation < 360.0F) {
            return rotation;
        }
        Float normalizedRotation = FeatStd::Float(FeatStd::UInt(rotation) % FeatStd::UInt(360)) + (rotation - FeatStd::Float(FeatStd::UInt(rotation)));
        if (normalizedRotation > 360.0F) {
            return normalizedRotation - 360.0F;
        }
        return normalizedRotation;
    }

    static inline void LimitAxisAlignedSize(Float rotation, Float layoutWidth, Float layoutHeight, Float& width, Float& height)
    {
        // the algorithm is only valid if layoutHeight is the smaller side.
        if (layoutHeight > layoutWidth) {
            // if this is not the case then we simply call the algorithm with swapped axis
            LimitAxisAlignedSize(rotation, layoutHeight, layoutWidth, height, width);
        }
        else {
            // layoutHeight is the smaller side
            Float alpha = Math::DegreeToRadian(rotation);
            Float sinAlpha = Math::Sine(alpha);
            Float cosAlpha = Math::Cosine(alpha);
            Float halfLayoutHeight = 0.5F * layoutHeight;
            Float layoutWidthCosAlpha = cosAlpha * layoutWidth;
            // as long as the rotated rectangle with maximum area fits into the original rectangle
            // the diagonal is axis parallel to the x axis and in the center of the original rectangle
            // to check that condition the diagonal has to be smaller than the width of the original rectangle
            // (1) rule from trigonometry: tan(alpha) == sin(alpha) / cos(alpha)
            // (2) rule from trigonometry: cot(alpha) == 1 / tan(alpha) == cos(alpha) / sin(alpha)
            // (3) rule from trigonometry: cos^2(alpha) + sin^2(alpha) == 1
            // (4) implication from (1) and (2): cot(alpha) == cos(alpha) / sin(alpha)
            // (5) implication from (1) and (4): cot(alpha) + tan(alpha) == cos(alpha) / sin(alpha) + sin(alpha) / cos(alpha)
            // (6) implication from (5): cot(alpha) + tan(alpha) == (cos^2(alpha) + sin^2(alpha)) / (cos(alpha) * sin(alpha))
            // (7) implication from (6) and (3)cot(alpha) + tan(alpha) == 1 / (cos(alpha) * sin(alpha))
            // (8) condition from trigonometry: layoutWidth >= (cot(alpha) + tan(alpha)) * LayoutHeight * 0.5
            // (9) implication from (8) and (7):layoutWidth >= LayoutHeight * 0.5 / (cos(alpha) * sin(alpha))
            // Hence the best most performant condition to check the length is:
            // halfLayoutHeight <= layoutWidthCosAlpha * sinAlpha
            if (halfLayoutHeight <= layoutWidthCosAlpha * sinAlpha) {
                // only 3 corners touch the original rectangle
                // simply calculate width and height with trigonometry rules
                width = halfLayoutHeight / sinAlpha;
                height = halfLayoutHeight / cosAlpha;
            }
            else {
                // we can not use the simple solution in that case
                // (1) all 4 corners touch the original rectangle
                // (2) implication from (1): layoutHeight == width * sin(alpha) + height * cos(alpha)
                // (3) implication from (1): layoutWidth == width * cos(alpha) + height * sin(alpha)
                // (4) implication from (2): width == (layoutHeight - height * cos(alpha)) / sin(alpha)
                // (5) implication from (3): width == (layoutWidth - height * sin(alpha)) / cos(alpha)
                // (6) implication from (4) and (5): (layoutHeight - height * cos(alpha)) / sin(alpha) == (layoutWidth - height * sin(alpha)) / cos(alpha)
                // (7) implication from (6): height= (layoutHeight * cos(alpha) - layoutWidth * sin(alpha)) / (cos^2(alpha) - sin^2(alpha))
                // (8) implication from (2): height == (layoutHeight - width * sin(alpha)) / cos(alpha)
                // (9) implication from (3): height == (layoutWidth - width * cos(alpha)) / sin(alpha)
                // (10) implication from (8) and (9): (layoutHeight - height * cos(alpha)) / sin(alpha) == (layoutWidth - height * sin(alpha)) / cos(alpha)
                // (11) implication from (10): width= (layoutWidth * cos(alpha) - layoutHeight * sin(alpha)) / (cos^2(alpha) - sin^2(alpha))
                Float denominator = cosAlpha * cosAlpha - sinAlpha * sinAlpha;
                width = (layoutWidthCosAlpha - layoutHeight * sinAlpha) / denominator;
                height = (layoutHeight * cosAlpha - layoutWidth * sinAlpha) / denominator;
            }
        }
    }

    static inline Vector2 GetLimitedAxisAlignedSize(const Vector2& size, Float rotation)
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(777, "Testing floats for equality [MISRA C++ Rule 6-2-2]")
        if ((size.GetX() == Math::MaxFloat()) || (size.GetY() == Math::MaxFloat())) {
            return size;
        }
        // to handle the ranges between 0 and 360 we have to normalize the rotation first to that range.
        Float normalizedRotation = NormalizeRotation(rotation);
        if ((normalizedRotation == 0.0F) || (normalizedRotation == 180.0F)) {
            return size;
        }
        if ((normalizedRotation == 90.0F) || (normalizedRotation == 270.0F)) {
            return Vector2(size.GetY(), size.GetX());
        }
        Float layoutWidth = size.GetX();
        Float layoutHeight = size.GetY();
        Float w = layoutWidth;
        Float h = layoutHeight;
        if (normalizedRotation > 180.0) {
            if (normalizedRotation > 270.0) {
                if (normalizedRotation > 315.0) {
                    // sector 315 to 360
                    // maps to the reverse range 45 to 0 => 360 - normalizedRotation
                    // the axis are normal => no swapping
                    LimitAxisAlignedSize(360.0F - normalizedRotation, layoutWidth, layoutHeight, w, h);
                }
                else {
                    // sector 270 to 315
                    // maps to the range 0 to 45 => normalizedRotation - 270
                    // the axis are swapped => swap w and h
                    LimitAxisAlignedSize(normalizedRotation - 270.0F, layoutWidth, layoutHeight, h, w);
                }
            }
            else {
                if (normalizedRotation > 225.0) {
                    // sector 225 to 270
                    // maps to the reverse range 45 to 0 => 270 - normalizedRotation
                    // the axis are swapped => swap w and h
                    LimitAxisAlignedSize(270.0F - normalizedRotation, layoutWidth, layoutHeight, h, w);
                }
                else {
                    // sector 180 to 225
                    // maps to the range 0 to 45 => normalizedRotation - 180
                    // the axis are normal => no swapping
                    LimitAxisAlignedSize(normalizedRotation - 180.0F, layoutWidth, layoutHeight, w, h);
                }
            }
        }
        else {
            if (normalizedRotation > 90.0) {
                if (normalizedRotation > 135.0) {
                    // sector 135 to 180
                    // maps to the reverse range 45 to 0 => 180 - normalizedRotation
                    // the axis are normal => no swapping
                    LimitAxisAlignedSize(180.0F - normalizedRotation, layoutWidth, layoutHeight, w, h);
                }
                else {
                    // sector 90 to 135
                    // maps to the range 0 to 45 => normalizedRotation - 90
                    // the axis are swapped => swap w and h
                    LimitAxisAlignedSize(normalizedRotation - 90.0F, layoutWidth, layoutHeight, h, w);
                }
            }
            else {
                if (normalizedRotation > 45.0) {
                    // sector 45 to 90
                    // maps to the reverse range 45 to 0 => 90 - normalizedRotation
                    // the axis are swapped => swap w and h
                    LimitAxisAlignedSize(90.0F - normalizedRotation, layoutWidth, layoutHeight, h, w);
                }
                else {
                    // sector 0 to 45
                    // the LimitAxisAlignedSize function is an exact match
                    LimitAxisAlignedSize(normalizedRotation, layoutWidth, layoutHeight, w, h);
                }
            }
        }
        return Vector2(w, h);
    }

    /******************************************************************************
     *  Measure
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::Measure(Node2D& node, const Vector2& clientSize)
    {
        AbstractNodePointer abstractNode(&node);
        Measure(abstractNode, clientSize);
    }
#endif

    void Layouter::Measure(const AbstractNodePointer& node, const Vector2& clientSize)
    {
        if (node.IsValid()) {
            Internal::LayoutEvents::SizeLayoutEvent sizeLayoutEventBegin(Internal::LayoutEvents::LayoutEvent::MeassureBegin, node, clientSize);
            GetEventSource().DispatchEvent(sizeLayoutEventBegin);

            Node2D* node2D = node.ToNode2D();
            if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
                /** ---- Patched Layout ---- **/

                if (node.IsEffectiveRenderingEnabled() || (!IsCollapsible(*node.ToCanderaObject()))) {
                    const Margin& margin = Layouter::GetMargin(*node.ToCanderaObject());

                    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
                        const Vector2& pivotOffset = node.GetPivotOffset();
                    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
                        const Int16 horizonalMargin = margin.GetLeft() + margin.GetRight() + static_cast<Int16>(pivotOffset.GetX());
                    const Int16 verticalMargin = margin.GetTop() + margin.GetBottom() + static_cast<Int16>(pivotOffset.GetY());

                    Vector2 nodeSize(Layouter::GetSize(*node.ToCanderaObject()));
                    Vector2 nodeMaxSize(ReturnNodeMaximumSizeOrMaxFloat(*node.ToCanderaObject()));
                    Vector2 nodeMinSize(Layouter::GetMinimumSize(*node.ToCanderaObject()));
                    Vector2 limittedSize((nodeSize.GetX() < 0.0F) ? (clientSize.GetX() - Float(horizonalMargin)) : nodeSize.GetX(),
                        (nodeSize.GetY() < 0.0F) ? (clientSize.GetY() - Float(verticalMargin)) : nodeSize.GetY());
                    limittedSize = Vector2Min(limittedSize, nodeMaxSize);
                    limittedSize = Vector2Max(limittedSize, nodeMinSize);

                    Float rotation = node.GetRotation();
                    Vector2 preferredSize;
                    if (LayouterDynamicProperties::IsCachedLayout(*node.ToCanderaObject())) {
                        bool layoutValid = LayouterPrivateDynamicProperties::IsLayoutValid(*node.ToCanderaObject());
                        if (layoutValid && (LayouterPrivateDynamicProperties::GetCachedMeasureSize(*node.ToCanderaObject()) != limittedSize)) {
                            LayouterPrivateDynamicProperties::SetLayoutValid(*node.ToCanderaObject(), false);
                            layoutValid = false;
                        }
                        if (!layoutValid) {
                            LayouterPrivateDynamicProperties::SetCachedMeasureSize(*node.ToCanderaObject(), limittedSize);
                            if (0.0F != rotation) {
                                limittedSize = GetLimitedAxisAlignedSize(limittedSize, rotation);
                            }
                            preferredSize = OnMeasure(node, limittedSize);
                            if (!IsHandlingScale()) {
                                preferredSize *= node.GetScale();
                            }
                            LayouterPrivateDynamicProperties::SetCachedPreferredSize(*node.ToCanderaObject(), preferredSize);
                        }
                        else {
                            preferredSize = LayouterPrivateDynamicProperties::GetCachedPreferredSize(*node.ToCanderaObject());
                        }
                    }
                    else {
                        if (0.0F != rotation) {
                            limittedSize = GetLimitedAxisAlignedSize(limittedSize, rotation);
                        }
                        preferredSize = OnMeasure(node, limittedSize);
                        if (!IsHandlingScale()) {
                            preferredSize *= node.GetScale();
                        }
                    }
                    if (0.0F != node.GetRotation()) {
                        static_cast<void>(LayouterPrivateDynamicProperties::SetOriginalPreferredSize(*node.ToCanderaObject(), preferredSize));
                    }
                    const Vector2& nodeScale = node.GetScale();
                    preferredSize += pivotOffset * nodeScale;
                    OnLimitPreferredSize(*node.ToCanderaObject(), preferredSize, nodeSize, nodeMinSize, nodeMaxSize);
                    if (0.0F != rotation) {
                        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(777, "Testing floats for equality [MISRA C++ Rule 6-2-2]")
                            LayouterPrivateDynamicProperties::RotatedPreferredSizeType rotatedPreferredSize;
                        if ((limittedSize.GetX() == Math::MaxFloat()) && (limittedSize.GetY() == Math::MaxFloat())) {
                            Float normalizedRotation = NormalizeRotation(rotation);
                            if (!((normalizedRotation == 0.0F) || (normalizedRotation == 180.0F) || (normalizedRotation == 90.0F) || (normalizedRotation == 270.0F))) {
                                rotatedPreferredSize = LayouterPrivateDynamicProperties::RotatedPreferredSizeType(preferredSize);
                            }
                        }
                        static_cast<void>(LayouterPrivateDynamicProperties::SetRotatedPreferredSize(*node.ToCanderaObject(), rotatedPreferredSize));
                        preferredSize = GetAxisAlignedDimension(preferredSize, rotation);
                        if (((limittedSize.GetX() == Math::MaxFloat()) || (limittedSize.GetY() == Math::MaxFloat())) &&
                            ((limittedSize.GetX() != Math::MaxFloat()) || (limittedSize.GetY() != Math::MaxFloat()))) {
                            // the size of one axis is set and the other one is not set.
                            if (preferredSize.GetX() > limittedSize.GetX()) {
                                preferredSize.SetX(limittedSize.GetX());
                            }
                            if (preferredSize.GetY() > limittedSize.GetY()) {
                                preferredSize.SetY(limittedSize.GetY());
                            }
                            // reduce the dimension of the estimated preferredSize to the minimum requied box to contain the rotated rectangle
                            Vector2 limitedPreferredSize = GetLimitedAxisAlignedSize(preferredSize, rotation);
                            preferredSize = GetAxisAlignedDimension(limitedPreferredSize, rotation);
                        }
                    }
                    node.SetPreferredSize(preferredSize);
                }
                else {
                    node.SetPreferredSize(Vector2());
                }
                /** ---- Patched Layout end ---- **/
            }
            else {
                /** ---- Old Layout ---- **/
                const Margin& margin = Layouter::GetMargin(*node2D);
                const Int16 horizonalMargin = margin.GetLeft() + margin.GetRight();
                const Int16 verticalMargin = margin.GetTop() + margin.GetBottom();

                Vector2 nodeSize(Layouter::GetSize(*node2D));
                Vector2 nodeMaxSize(ReturnNodeMaximumSizeOrMaxFloat(*node2D));
                Vector2 nodeMinSize(Layouter::GetMinimumSize(*node2D));
                Vector2 limittedSize((nodeSize.GetX() < 0.0F) ? (clientSize.GetX() - Float(horizonalMargin)) : nodeSize.GetX(),
                    (nodeSize.GetY() < 0.0F) ? (clientSize.GetY() - Float(verticalMargin)) : nodeSize.GetY());
                limittedSize = Vector2Min(limittedSize, nodeMaxSize);
                limittedSize = Vector2Max(limittedSize, nodeMinSize);

                Vector2 preferredSize;
                if (LayouterDynamicProperties::IsCachedLayout(*node2D)) {
                    bool layoutValid = LayouterPrivateDynamicProperties::IsLayoutValid(*node2D);
                    if (layoutValid && (LayouterPrivateDynamicProperties::GetCachedMeasureSize(*node2D) != limittedSize)) {
                        LayouterPrivateDynamicProperties::SetLayoutValid(*node2D, false);
                        layoutValid = false;
                    }
                    if (!layoutValid) {
                        LayouterPrivateDynamicProperties::SetCachedMeasureSize(*node2D, limittedSize);
                        preferredSize = OnMeasure(node, limittedSize);
                        LayouterPrivateDynamicProperties::SetCachedPreferredSize(*node2D, preferredSize);
                    }
                    else {
                        preferredSize = LayouterPrivateDynamicProperties::GetCachedPreferredSize(*node2D);
                    }
                }
                else {
                    preferredSize = OnMeasure(node, limittedSize);
                }
                OnLimitPreferredSize(*node2D, preferredSize, nodeSize, nodeMinSize, nodeMaxSize);
                node.SetPreferredSize(preferredSize);

                /** ---- Old Layout end ---- **/
            }
            Internal::LayoutEvents::SizeLayoutEvent sizeLayoutEventEnd(Internal::LayoutEvents::LayoutEvent::MeassureEnd, node, clientSize);
            GetEventSource().DispatchEvent(sizeLayoutEventEnd);
        }
    }

   /******************************************************************************
    *  SetArrangeActualSize
    ******************************************************************************/
    static FeatStd::Optional<Vector2>& OnArrangeResultActualSize()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Optional<Vector2>, s_arrangeActualSize);
        return s_arrangeActualSize;
    }

    void Layouter::SetArrangeActualSize(const Vector2& size)
    {
        OnArrangeResultActualSize() = FeatStd::Optional<Vector2>(size);
    }

    /******************************************************************************
    *  SetArrangeActualSize
    ******************************************************************************/

    static Vector2& ArrangeResultActualSize()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(Vector2, s_arrangeResultActualSize);
        return s_arrangeResultActualSize;
    }

    const Vector2& Layouter::GetArrangeActualSize()
    {
        return ArrangeResultActualSize();
    }

    /******************************************************************************
     *  Arrange
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::Arrange(Node2D& node, const Rectangle& clientArea, bool topLevel /* = false */)
    {
        AbstractNodePointer abstractNode(&node);
        Arrange(abstractNode, clientArea, topLevel);
    }
#endif

    void Layouter::Arrange(const AbstractNodePointer& node, const Rectangle& clientArea, bool topLevel /* = false */)
    {
        if (node.IsValid()) {
            const Vector2 arrangePosition = clientArea.GetPosition();
            const Vector2 arrangeSize = clientArea.GetSize();
            Internal::LayoutEvents::ArrangeEvent arrangeEventBegin(Internal::LayoutEvents::LayoutEvent::ArrangeBegin, node, arrangePosition, arrangeSize);
            GetEventSource().DispatchEvent(arrangeEventBegin);
            Node2D* node2D = node.ToNode2D();
            if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
                /** ---- Patched Layout ---- **/

                if (node.IsEffectiveRenderingEnabled() || (!IsCollapsible(*node.ToCanderaObject()))) {
                    Float rotation = node.GetRotation();
                    Vector2 pivotPosition = node.GetPivotPoint();
                    const Vector2& preferredSize = node.GetPreferredSize();
                    LayouterPrivateDynamicProperties::RotatedPreferredSizeType rotatedPreferredSize;
                    if (0.0F != rotation) {
                        rotatedPreferredSize = LayouterPrivateDynamicProperties::GetRotatedPreferredSize(*node.ToCanderaObject());
                    }
                    const Margin& margin = Layouter::GetMargin(*node.ToCanderaObject());

                    const bool isNodeTransformationAllowed = IsNodeTransformationAllowed(node);

                    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
                        const Vector2& pivotOffset = node.GetPivotOffset();
                    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
                        const Int16 hMargin = (isNodeTransformationAllowed ? (margin.GetLeft() + margin.GetRight()) : 0) + static_cast<Int16>(pivotOffset.GetX());
                    const Int16 vMargin = (isNodeTransformationAllowed ? (margin.GetTop() + margin.GetBottom()) : 0) + static_cast<Int16>(pivotOffset.GetY());


                    const Float clientWidth = clientArea.GetWidth();
                    const Float clientHeight = clientArea.GetHeight();
                    Float width = preferredSize.GetX() + static_cast<Float>(hMargin);
                    Float height = preferredSize.GetY() + static_cast<Float>(vMargin);
                    if (clientWidth < Math::MaxFloat()) {
                        if ((GetHorizontalAlignment(*node.ToCanderaObject()) == HStretch) || (clientWidth < width)) {
                            width = clientWidth;
                        }
                    }
                    if (clientHeight < Math::MaxFloat()) {
                        if ((GetVerticalAlignment(*node.ToCanderaObject()) == VStretch) || (clientHeight < height)) {
                            height = clientHeight;
                        }
                    }
                    const Vector2 nodeSize(Layouter::GetSize(*node.ToCanderaObject()));
                    const Vector2 nodeMaxSize(ReturnNodeMaximumSizeOrMaxFloat(*node.ToCanderaObject()));
                    const Vector2 nodeMinSize(Layouter::GetMinimumSize(*node.ToCanderaObject()));
                    if (width < Math::MaxFloat()) {
                        width -= static_cast<Float>(hMargin);
                    }
                    if (height < Math::MaxFloat()) {
                        height -= static_cast<Float>(vMargin);
                    }
                    if (width < 0.0F) {
                        width = 0.0F;
                    }
                    if (height < 0.0F) {
                        height = 0.0F;
                    }
                    bool isRightToLeft = GetParentLanguageSensitiveDirection(node) == LayoutAlignment::LayoutDirection::RightToLeftDirection;
                    Rectangle arrangeArea(node.GetPosition().GetX(), node.GetPosition().GetY(), width, height);
                    OnArrangeResultActualSize() = FeatStd::Optional<Vector2>();
                    if (LayouterDynamicProperties::IsCachedLayout(*node.ToCanderaObject())) {
                        if (!LayouterPrivateDynamicProperties::IsLayoutValid(*node.ToCanderaObject())) {
                            LayouterPrivateDynamicProperties::SetLayoutValid(*node.ToCanderaObject(), true);
                            if (0.0F != rotation) {
                                if (rotatedPreferredSize.IsSet()) {
                                    arrangeArea.SetSize(*rotatedPreferredSize);
                                }
                                else {
                                    arrangeArea.SetSize(GetLimitedAxisAlignedSize(arrangeArea.GetSize(), rotation));
                                }
                            }
                            if (!IsHandlingScale()) {
                                arrangeArea.SetSize(arrangeArea.GetSize() / node.GetScale());
                            }
                            if (nodeSize.GetX() > 0.0F) {
                                arrangeArea.SetWidth(nodeSize.GetX());
                            }
                            if (nodeSize.GetY() > 0.0F) {
                                arrangeArea.SetHeight(nodeSize.GetY());
                            }
                            if (arrangeArea.GetWidth() < nodeMinSize.GetX()) {
                                arrangeArea.SetWidth(nodeMinSize.GetX());
                            }
                            if (arrangeArea.GetHeight() < nodeMinSize.GetY()) {
                                arrangeArea.SetHeight(nodeMinSize.GetY());
                            }
                            if (arrangeArea.GetWidth() > nodeMaxSize.GetX()) {
                                arrangeArea.SetWidth(nodeMaxSize.GetX());
                            }
                            if (arrangeArea.GetHeight() > nodeMaxSize.GetY()) {
                                arrangeArea.SetHeight(nodeMaxSize.GetY());
                            }
                            OnArrange(node, arrangeArea);
                            if (OnArrangeResultActualSize().IsSet()) {
                                LayouterPrivateDynamicProperties::SetCachedArrangeActualSize(*node.ToCanderaObject(), *OnArrangeResultActualSize());
                            }
                            else {
                                bool ok = LayouterPrivateDynamicProperties::ClearCachedArrangeActualSize(*node.ToCanderaObject());
                                FEATSTD_UNUSED(ok);
                            }
                        }
                        else {
                            Vector2 cachedArrangeActualSize = LayouterPrivateDynamicProperties::GetCachedArrangeActualSize(*node.ToCanderaObject());
                            if (LayouterPrivateDynamicProperties::CachedArrangeActualSizeDefault() != cachedArrangeActualSize) {
                                OnArrangeResultActualSize() = cachedArrangeActualSize;
                            }
                        }
                    }
                    else {
                        if (0.0F != rotation) {
                            if (rotatedPreferredSize.IsSet()) {
                                arrangeArea.SetSize(*rotatedPreferredSize);
                            }
                            else {
                                arrangeArea.SetSize(GetLimitedAxisAlignedSize(arrangeArea.GetSize(), rotation));
                            }
                        }
                        if (!IsHandlingScale()) {
                            arrangeArea.SetSize(arrangeArea.GetSize() / node.GetScale());
                        }
                        if (nodeSize.GetX() > 0.0F) {
                            arrangeArea.SetWidth(nodeSize.GetX());
                        }
                        if (nodeSize.GetY() > 0.0F) {
                            arrangeArea.SetHeight(nodeSize.GetY());
                        }
                        if (arrangeArea.GetWidth() < nodeMinSize.GetX()) {
                            arrangeArea.SetWidth(nodeMinSize.GetX());
                        }
                        if (arrangeArea.GetHeight() < nodeMinSize.GetY()) {
                            arrangeArea.SetHeight(nodeMinSize.GetY());
                        }
                        if (arrangeArea.GetWidth() > nodeMaxSize.GetX()) {
                            arrangeArea.SetWidth(nodeMaxSize.GetX());
                        }
                        if (arrangeArea.GetHeight() > nodeMaxSize.GetY()) {
                            arrangeArea.SetHeight(nodeMaxSize.GetY());
                        }
                        OnArrange(node, arrangeArea);
                    }
                    Vector2 actualPosition;
                    Vector2 actualSize(width, height);
                    if (!topLevel) {
                        Vector2 scale = node.GetScale();
                        if (OnArrangeResultActualSize().IsSet()) {
                            actualSize = *OnArrangeResultActualSize();
                        }
                        else {
                            FEATSTD_LOG_ERROR("custom layouter did not set ArrangeActualSize within OnArrange! ArrangeActualSize is required for a correct RightToLeft layout!");
                        }
                        const Float outerLeft = clientArea.GetLeft() + pivotOffset.GetX() + static_cast<Float>((isRightToLeft) ? margin.GetRight() : margin.GetLeft());
                        const Float outerTop = clientArea.GetTop() + pivotOffset.GetY() + static_cast<Float>(margin.GetTop());
                        const Float outerWidth = (clientWidth >= Math::MaxFloat()) ? preferredSize.GetX() : ((clientWidth)-(static_cast<Float>(hMargin)+pivotOffset.GetX()));
                        const Float outerHeight = (clientHeight >= Math::MaxFloat()) ? preferredSize.GetY() : ((clientHeight)-(static_cast<Float>(vMargin)+pivotOffset.GetY()));
                        const Float null(0.0F);
                        const Rectangle outerRect(outerLeft, outerTop, Math::Maximum(null, outerWidth), Math::Maximum(null, outerHeight));
                        Rectangle innerRect(0.0F, 0.0F, actualSize.GetX(), actualSize.GetY());
                        Vector2 axisAlignedOffsetSize = actualSize;
                        if (0.0F != rotation) {
                            innerRect.SetSize(GetAxisAlignedDimension(actualSize, rotation));
                        }
                        AlignRectangle(outerRect, innerRect, node);
                        Vector2 rightBottomMargin(static_cast<Float>((isRightToLeft) ? margin.GetLeft() : margin.GetRight()), static_cast<Float>(margin.GetBottom()));
                        actualSize = innerRect.GetPosition() + innerRect.GetSize() + rightBottomMargin;
                        actualPosition = innerRect.GetPosition();
                        if (0.0F != rotation) {
                            innerRect.SetPosition(innerRect.GetPosition() + GetAxisAlignedOffset(axisAlignedOffsetSize, rotation, pivotPosition, scale));
                        }
                        else {
                            if ((1.0F != scale.GetX()) && (0.0F != pivotPosition.GetX())) {
                                innerRect.SetLeft(innerRect.GetLeft() + pivotPosition.GetX() * (scale.GetX() - 1.0F));
                            }
                            if ((1.0F != scale.GetY()) && (0.0F != pivotPosition.GetY())) {
                                innerRect.SetTop(innerRect.GetTop() + pivotPosition.GetY() * (scale.GetY() - 1.0F));
                            }
                        }
                        OnSetPosition(node, innerRect.GetPosition());
                    }
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
                    Rectangle clippingRectangle(clientArea);
                    clippingRectangle.SetLeft(clippingRectangle.GetLeft() + static_cast<Float>(margin.GetLeft()));
                    clippingRectangle.SetTop(clippingRectangle.GetTop() + static_cast<Float>(margin.GetTop()));
                    clippingRectangle.SetWidth(clippingRectangle.GetWidth() - static_cast<Float>((margin.GetLeft() + margin.GetRight())));
                    clippingRectangle.SetHeight(clippingRectangle.GetHeight() - static_cast<Float>((margin.GetTop() + margin.GetBottom())));
                    OnClipping(node, clippingRectangle);
#endif
                    if (actualSize.GetX() > clientArea.GetLeft() + clientArea.GetWidth()) {
                        actualSize.SetX(clientArea.GetLeft() + clientArea.GetWidth());
                    }
                    if (actualSize.GetY() > clientArea.GetTop() + clientArea.GetHeight()) {
                        actualSize.SetY(clientArea.GetTop() + clientArea.GetHeight());
                    }
                    Internal::LayoutEvents::ArrangeEvent arrangeEvent(Internal::LayoutEvents::LayoutEvent::ArrangeSetActualSize, node, actualPosition, actualSize - actualPosition);
                    GetEventSource().DispatchEvent(arrangeEvent);
                    ArrangeResultActualSize() = actualSize;
                    OnArrangeResultActualSize() = FeatStd::Optional<Vector2>();
                    // the preferred size is no longer needed for the node, so it can be cleared
                    if (0.0F != rotation) {
                        static_cast<void>(LayouterPrivateDynamicProperties::SetOriginalPreferredSize(*node.ToCanderaObject(), Vector2()));
                    }
                    node.SetPreferredSize(Vector2());
#ifdef CANDERA_2D_ENABLED
                    RenderNode* renderNode = Dynamic_Cast<RenderNode*>(node.ToNode2D());
                    if (0 != renderNode) {
                        bool useRightToLeft = false;
                        if (isRightToLeft) {
                            Effect2D* effect = renderNode->GetEffect(0);
                            if (0 != effect) {
                                InPlaceEffect2D* inPlaceEffect2D = effect->GetInPlaceEffect2D(0);
                                if (0 != inPlaceEffect2D) {
                                    if (Internal::RenderDirectionHelper::GetUseNodeLayoutDirection(*inPlaceEffect2D)) {
                                        useRightToLeft = true;
                                    }
                                }
                            }
                        }
                        Internal::RenderDirectionHelper::SetRightToLeft(*renderNode, useRightToLeft);
                    }
#endif
                }
                else {
                    Vector2 actualSize;
                    Internal::LayoutEvents::SizeLayoutEvent sizeLayoutEvent(Internal::LayoutEvents::LayoutEvent::ArrangeSetActualSize, node, actualSize);
                    GetEventSource().DispatchEvent(sizeLayoutEvent);
                    ArrangeResultActualSize() = actualSize;
                    node.SetPreferredSize(Vector2());
                }
                /** ---- Patched Layout end ---- **/
            }
            else {
                /** ---- Old Layout ---- **/
                const Float null(0.0F);
                const Vector2& preferredSize = node.GetPreferredSize();
                Rectangle innerRect(node.GetPosition(), preferredSize);

                if (!topLevel) {
                    innerRect.SetPosition(0.0F, 0.0F);
                    const Margin& margin = Layouter::GetMargin(*node2D);
                    const Int16 hMargin = margin.GetLeft() + margin.GetRight();
                    const Int16 vMargin = margin.GetTop() + margin.GetBottom();

                    const Float clientWidth = clientArea.GetWidth();
                    const Float clientHeight = clientArea.GetHeight();
                    const Float width = (clientWidth >= Math::MaxFloat()) ? preferredSize.GetX() : ((clientWidth)-static_cast<Float>(hMargin));
                    const Float height = (clientHeight >= Math::MaxFloat()) ? preferredSize.GetY() : ((clientHeight)-static_cast<Float>(vMargin));

                    const Float left = clientArea.GetLeft() + static_cast<Float>((GetLanguageSensitiveDirection(node) == LayoutAlignment::LayoutDirection::RightToLeftDirection) ? margin.GetRight() : margin.GetLeft());
                    const Float top = clientArea.GetTop() + static_cast<Float>(margin.GetTop());

                    Rectangle outerRect(left, top, Math::Maximum(null, width), Math::Maximum(null, height));
                    AlignRectangle(outerRect, innerRect, node);
                }

                if (LayouterDynamicProperties::IsCachedLayout(*node2D)) {
                    if (!LayouterPrivateDynamicProperties::IsLayoutValid(*node2D)) {
                        LayouterPrivateDynamicProperties::SetLayoutValid(*node2D, true);
                        OnArrange(node, innerRect);
                        if (OnArrangeResultActualSize().IsSet()) {
                            LayouterPrivateDynamicProperties::SetCachedArrangeActualSize(*node2D, *OnArrangeResultActualSize());
                        }
                        else {
                            LayouterPrivateDynamicProperties::ClearCachedArrangeActualSize(*node2D);
                        }
                    }
                    else {
                        Vector2 cachedArrangeActualSize = LayouterPrivateDynamicProperties::GetCachedArrangeActualSize(*node2D);
                        if (LayouterPrivateDynamicProperties::CachedArrangeActualSizeDefault() != cachedArrangeActualSize) {
                            OnArrangeResultActualSize() = cachedArrangeActualSize;
                        }
                    }
                }
                else {
                    OnArrange(node, innerRect);
                }

                // the preferred size is no longer needed for the node, so it can be cleared
                node.SetPreferredSize(Vector2());
                /** ---- Old Layout end ---- **/

            }
            Internal::LayoutEvents::ArrangeEvent arrangeEventEnd(Internal::LayoutEvents::LayoutEvent::ArrangeEnd, node, arrangePosition, arrangeSize);
            GetEventSource().DispatchEvent(arrangeEventEnd);
        }
    }

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
    /******************************************************************************
     *  OnClipping
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::OnClipping(Node2D& node, const Rectangle& clientArea)
    {
        OnClipping(AbstractNodePointer(&node), clientArea);
    }
#endif

    void Layouter::OnClipping(const AbstractNodePointer& node, const Rectangle& clientArea)
    {
#ifdef CANDERA_2D_ENABLED
        RenderNode* renderNode = Dynamic_Cast<RenderNode*>(node.ToNode2D());
        if (0 != renderNode) {
            AbstractNodePointer parent = node.GetParent();
            if (parent.IsValid()) {
                Rectangle boundingRect;
                BoundingRectTraverser traverser(*parent.ToNode2D());
                traverser.Traverse(*node.ToNode2D());
                traverser.GetBoundingRectangle(boundingRect);
                if ((boundingRect.GetLeft() < clientArea.GetLeft()) ||
                    (boundingRect.GetTop() < clientArea.GetTop()) ||
                    (boundingRect.GetLeft() + boundingRect.GetWidth()> node.GetPosition().GetX() + clientArea.GetWidth()) ||
                    (boundingRect.GetTop() + boundingRect.GetHeight() > node.GetPosition().GetY() + clientArea.GetHeight())) {
                    static_cast<void>(SetClippingArea(*renderNode, clientArea));
                    renderNode->SetClippingRect(clientArea);
                }
                else {
                    static_cast<void>(SetClippingArea(*renderNode, Rectangle(0.0F, 0.0F, -1.0F, -1.0F)));
                    renderNode->SetClippingRect();
                }
            }
        }
#else
        FEATSTD_UNUSED2(node, clientArea);
#endif
    }
#endif

    /******************************************************************************
     *  OnSetPosition
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::OnSetPosition(Node2D& node, const Vector2& position)
    {
        AbstractNodePointer abstractNode(&node);
        OnSetPosition(abstractNode, position);
    }
#endif

    void Layouter::OnSetPosition(const AbstractNodePointer& node, const Vector2& position)
    {
        SetNodePosition(node, position);
    }

    /******************************************************************************
     *  GetClientSize
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    Vector2 Layouter::GetClientSize(const Node2D& node) const
    {
        return GetClientSize(AbstractNodePointer(const_cast<Node2D*>(&node)));
    }
#endif

    Vector2 Layouter::GetClientSize(const AbstractNodePointer& node)
    {
        if ((!node.IsEffectiveRenderingEnabled()) && Layouter::IsCollapsible(*node.ToCanderaObject()))
        {
            // If rendering is disable and node is collapsible, don't take into account the margins
            return node.GetPreferredSize();
        }

        const Margin& margin = Layouter::GetMargin(*node.ToCanderaObject());
        const Int16 horizontalMargin = margin.GetLeft() + margin.GetRight();
        const Int16 verticalMargin = margin.GetTop() + margin.GetBottom();

        return Vector2(node.GetPreferredSize() + Vector2(Float(horizontalMargin), Float(verticalMargin)));
    }

    void Layouter::SetCached(CanderaObject& node, bool cached)
    {
        LayouterDynamicProperties::SetCachedLayout(node, cached);
    }

    bool Layouter::IsCached(const CanderaObject& node)
    {
        return LayouterDynamicProperties::IsCachedLayout(node);
    }

    const Candera::DynamicProperties::DynamicPropertyHost* Layouter::ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host) {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, "Layouter interface only allows CanderaObject for dynamic properties")
        const CanderaObject* object = static_cast<const CanderaObject*>(host);
#ifdef CANDERA_2D_ENABLED
        const Node2D* node2D = Dynamic_Cast<const Node2D*>(object);
        if (0 != node2D) {
            return node2D->GetParent();
        }
#endif
#ifdef CANDERA_3D_ENABLED
        const Node* node = Dynamic_Cast<const Node*>(object);
        if (0 != node) {
            return node->GetParent();
        }
#endif
        return 0;
    }

    /******************************************************************************
    *  GetParentSpaceAxisAlignedBoundingRect
    ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::GetParentSpaceAxisAlignedBoundingRect(const Node2D& node, Rectangle& rect)
    {
        GetParentSpaceAxisAlignedBoundingRect(AbstractNodePointer(const_cast<Node2D*>(&node)), rect);
    }
#endif

#ifdef CANDERA_3D_ENABLED
    class LayoutBoundingBoxTraverser : public ConstTreeTraverser {
        typedef ConstTreeTraverser Base;

    public:
        LayoutBoundingBoxTraverser(const Node& root) :
            Base(),
            m_minBounds(Math::MaxFloat(), Math::MaxFloat(), Math::MaxFloat()),    // Init to invalid min bounds.
            m_maxBounds(-Math::MaxFloat(), -Math::MaxFloat(), -Math::MaxFloat())  // Init to invalid max bounds.
        {
            m_inverseGroupTransform = root.GetWorldTransform();
            m_inverseGroupTransform.Inverse();
        }

        void GetBoundingRectangle(Rectangle& rect) const
        {
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
            if ((m_minBounds.GetX() != Math::MaxFloat()) && (m_minBounds.GetY() != Math::MaxFloat())) {
                rect.SetPosition(m_minBounds.GetX(), m_minBounds.GetY());
                rect.SetSize(m_maxBounds.GetX() - m_minBounds.GetX(), m_maxBounds.GetY() - m_minBounds.GetY());
            }
            else {
                rect.SetPosition(0.0F, 0.0F);
                rect.SetSize(0.0F, 0.0F);
            }
        }

    protected:
        virtual TraverserAction ProcessNode(const Node& node) override
        {
            if ((Layouter::IsSizeSet(node)) || ((!node.IsTypeOf(Group::GetTypeId())) && (!node.IsTypeOf(Camera::GetTypeId())) && (!node.IsTypeOf(Light::GetTypeId()))) ) {
                Vector3 minBounds;
                Vector3 maxBounds;  //Node-oriented bounding box.
                Vector3 groupMinBounds;
                Vector3 groupMaxBounds;  //Group-oriented bounding box.
                node.GetBoundingBox(minBounds, maxBounds);

                // Compute transformation of given node into group's local space.
                Matrix4 transform = node.GetWorldTransform() * m_inverseGroupTransform;

                // Transform node's bounding box into group's local space.
                Math3D::TransformBoundingBox(minBounds, maxBounds, transform, groupMinBounds, groupMaxBounds);

                // Accumulate boundaries if they exceed current group's axis aligned bounding box.
                if (groupMinBounds.GetX() < m_minBounds.GetX()) { m_minBounds.SetX(groupMinBounds.GetX()); }
                if (groupMinBounds.GetY() < m_minBounds.GetY()) { m_minBounds.SetY(groupMinBounds.GetY()); }
                if (groupMinBounds.GetZ() < m_minBounds.GetZ()) { m_minBounds.SetZ(groupMinBounds.GetZ()); }

                if (groupMaxBounds.GetX() > m_maxBounds.GetX()) { m_maxBounds.SetX(groupMaxBounds.GetX()); }
                if (groupMaxBounds.GetY() > m_maxBounds.GetY()) { m_maxBounds.SetY(groupMaxBounds.GetY()); }
                if (groupMaxBounds.GetZ() > m_maxBounds.GetZ()) { m_maxBounds.SetZ(groupMaxBounds.GetZ()); }
            }

            return ProceedTraversing;
        }
    private:
        Matrix4 m_inverseGroupTransform;
        Vector3 m_minBounds;
        Vector3 m_maxBounds;
    };
#endif

    void Layouter::GetParentSpaceAxisAlignedBoundingRect(const AbstractNodePointer& node, Rectangle& rect)
    {
        AbstractNodePointer parent = node.GetParent();
        if (parent.IsValid()) {
            switch (parent.GetType()) {
#ifdef CANDERA_2D_ENABLED
            case AbstractNodePointer::Candera2D: {
                    BoundingRectTraverser traverser(*parent.ToNode2D());
                    traverser.Traverse(*node.ToNode2D());
                    traverser.GetBoundingRectangle(rect);
                }
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case AbstractNodePointer::Candera3D: {
                    LayoutBoundingBoxTraverser traverser(*parent.ToNode());
                    traverser.Traverse(*node.ToNode());
                    traverser.GetBoundingRectangle(rect);
                }
                break;
#endif
            case AbstractNodePointer::Undefined:
                break;

            default:
                break;
            }
        }
    }

    /******************************************************************************
    *  OnLimitPreferredSize
    ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::OnLimitPreferredSize(Node2D& node, Vector2& preferredSize, const Vector2& nodeSize, const Vector2& nodeMinSize, const Vector2& nodeMaxSize)
    {
        OnLimitPreferredSize(static_cast<CanderaObject&>(node), preferredSize, nodeSize, nodeMinSize, nodeMaxSize);
    }
#endif

    void Layouter::OnLimitPreferredSize(CanderaObject& /*node*/, Vector2& preferredSize, const Vector2& nodeSize, const Vector2& nodeMinSize, const Vector2& nodeMaxSize)
    {
        preferredSize.SetX(Math::Minimum((nodeSize.GetX() < 0.0F) ? preferredSize.GetX() : nodeSize.GetX(), nodeMaxSize.GetX()));
        preferredSize.SetY(Math::Minimum((nodeSize.GetY() < 0.0F) ? preferredSize.GetY() : nodeSize.GetY(), nodeMaxSize.GetY()));
        preferredSize.SetX(Math::Maximum((nodeSize.GetX() < 0.0F) ? preferredSize.GetX() : nodeSize.GetX(), nodeMinSize.GetX()));
        preferredSize.SetY(Math::Maximum((nodeSize.GetY() < 0.0F) ? preferredSize.GetY() : nodeSize.GetY(), nodeMinSize.GetY()));
    }

    /******************************************************************************
    *  AlignRectangle
    ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::AlignRectangle(const Rectangle& outerRect, Rectangle& innerRect, Node2D& node)
    {
        AbstractNodePointer abstractNode(&node);
        AlignRectangle(outerRect, innerRect, abstractNode);
    }
#endif

    void Layouter::AlignRectangle(const Rectangle& outerRect, Rectangle& innerRect, const AbstractNodePointer& node)
    {
        if (node.IsValid()) {
            Node2D* node2D = node.ToNode2D();
            if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
                /** ---- Patched Layout ---- **/
                const Vector2 nodeSize(Layouter::GetSize(*node.ToCanderaObject()));
                const Vector2 nodeMaxSize(ReturnNodeMaximumSizeOrMaxFloat(*node.ToCanderaObject()));
                const Vector2 nodeMinSize(Layouter::GetMinimumSize(*node.ToCanderaObject()));

                /* left, width
                 * ------------------- */
                HorizontalAlignment horizontalAlignment = GetHorizontalAlignment(*node.ToCanderaObject());
                LayoutAlignment::LayoutDirection::Enum layoutDirection = GetParentLanguageSensitiveDirection(node);
                if (layoutDirection == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                    switch (horizontalAlignment)
                    {
                    case Candera::HLeft:
                        horizontalAlignment = HRight;
                        break;
                    case Candera::HRight:
                        horizontalAlignment = HLeft;
                        break;
                    default:
                        break;
                    }
                }
                Float left = outerRect.GetLeft();
                switch (horizontalAlignment) {
                case HRight:
                    left += (outerRect.GetWidth() - innerRect.GetWidth());
                    break;
                case HCenter:
                    left += 0.5F * (outerRect.GetWidth() - innerRect.GetWidth());
                    break;
                case HLeft:
                    break;
                case HStretch: {
                                   Float width = outerRect.GetWidth();
                                   if (nodeSize.GetX() >= 0.0F) {
                                       width = nodeSize.GetX();
                                       innerRect.SetWidth(width);
                                   }
                                   if (nodeMaxSize.GetX() < width) {
                                       width = nodeMaxSize.GetX();
                                       innerRect.SetWidth(width);
                                   }
                                   if (nodeMinSize.GetX() > width) {
                                       width = nodeMinSize.GetX();
                                       innerRect.SetWidth(width);
                                   }
                                   if (outerRect.GetWidth() > width) {
                                       left += (0.5F * (outerRect.GetWidth() - width));
                                   }
                                   else {
                                       if (layoutDirection == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                                           left += outerRect.GetWidth() - innerRect.GetWidth();
                                       }
                                   }
                } break;

                default:
                    break;
                }
                innerRect.SetLeft(left);

                /* top, height
                 * ------------------- */
                Float top = outerRect.GetTop();
                switch (GetVerticalAlignment(*node.ToCanderaObject())) {
                case VBottom:
                    top += (outerRect.GetHeight() - innerRect.GetHeight());
                    break;
                case VCenter:
                    top += 0.5F * (outerRect.GetHeight() - innerRect.GetHeight());
                    break;
                case VTop:
                    break;
                case VStretch: {
                                   Float height = outerRect.GetHeight();
                                   if (nodeSize.GetY() >= 0.0F) {
                                       height = nodeSize.GetY();
                                       innerRect.SetHeight(height);
                                   }
                                   if (nodeMaxSize.GetY() < height) {
                                       height = nodeMaxSize.GetY();
                                       innerRect.SetHeight(height);
                                   }
                                   if (nodeMinSize.GetY() > height) {
                                       height = nodeMinSize.GetY();
                                       innerRect.SetHeight(height);
                                   }
                                   top = outerRect.GetTop();
                                   if (outerRect.GetHeight() > height) {
                                       top += (0.5F * (outerRect.GetHeight() - height));
                                   }
                } break;

                default:
                    break;
                }
                innerRect.SetTop(top);
                /** ---- Patched Layout end ---- **/
            }
            else {
                /** ---- Old Layout ---- **/

                const Vector2 nodeSize(Layouter::GetSize(*node2D));
                const Vector2 nodeMaxSize(ReturnNodeMaximumSizeOrMaxFloat(*node2D));
                const Vector2 nodeMinSize(Layouter::GetMinimumSize(*node2D));

                /* left, width
                * ------------------- */
                Float left = innerRect.GetLeft();
                Float factor = 0.0F;
                HorizontalAlignment horizontalAlignment = GetHorizontalAlignment(*node2D);
                LayoutAlignment::LayoutDirection::Enum layoutDirection = GetLanguageSensitiveDirection(node);
                if (layoutDirection == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                    switch (horizontalAlignment) {
                    case Candera::HLeft:
                        horizontalAlignment = HRight;
                        break;
                    case Candera::HRight:
                        horizontalAlignment = HLeft;
                        break;
                    default:
                        break;
                    }
                }
                switch (horizontalAlignment) {
                case HRight:
                    factor += 1.0F;
                    CANDERA_LINT_FALLTHROUGH()
                case HCenter:
                    factor += 1.0F;
                    CANDERA_LINT_FALLTHROUGH()
                case HLeft:
                    if (layoutDirection == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                        factor = 2.0F - factor;
                    }

                    left = outerRect.GetLeft() + (factor * 0.5F * (outerRect.GetWidth() - innerRect.GetWidth()));
                    break;

                case HStretch: {
                                   Float width = (nodeSize.GetX() >= 0.0F) ? nodeSize.GetX() : outerRect.GetWidth();
                                   if (nodeMaxSize.GetX() < width) {
                                       width = nodeMaxSize.GetX();
                                   }
                                   if (nodeMinSize.GetX() > width) {
                                       width = nodeMinSize.GetX();
                                   }
                                   left = outerRect.GetLeft();
                                   if (outerRect.GetWidth() > width) {
                                       left += (0.5F * (outerRect.GetWidth() - width));
                                   }
                                   else {
                                       if (layoutDirection == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                                           left += outerRect.GetWidth() - innerRect.GetWidth();
                                       }
                                   }
                                   innerRect.SetWidth(width);
                } break;

                default:
                    break;
                }

                innerRect.SetLeft(left);

                /* top, height
                * ------------------- */
                Float top = innerRect.GetTop();
                factor = 0.0F;

                switch (GetVerticalAlignment(*node2D)) {
                case VBottom:
                    factor += 1.0F;
                    CANDERA_LINT_FALLTHROUGH()
                case VCenter:
                    factor += 1.0F;
                    CANDERA_LINT_FALLTHROUGH()
                case VTop:
                    top = outerRect.GetTop() + (factor * 0.5F * (outerRect.GetHeight() - innerRect.GetHeight()));
                    break;

                case VStretch: {
                                   Float height = (nodeSize.GetY() >= 0.0F) ? nodeSize.GetY() : outerRect.GetHeight();
                                   if (nodeMaxSize.GetY() < height) {
                                       height = nodeMaxSize.GetY();
                                   }
                                   if (nodeMinSize.GetY() > height) {
                                       height = nodeMinSize.GetY();
                                   }
                                   top = outerRect.GetTop();
                                   if (outerRect.GetHeight() > height) {
                                       top += (0.5F * (outerRect.GetHeight() - height));
                                   }
                                   innerRect.SetHeight(height);
                } break;

                default:
                    break;
                }

                innerRect.SetTop(top);
                /** ---- Old Layout end ---- **/
            }
        }
    }

    /******************************************************************************
     *  SetNodePosition
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::SetNodePosition(Node2D& node, const Vector2& position)
    {
        AbstractNodePointer abstractNode(&node);
        SetNodePosition(abstractNode, position);
    }
#endif

    void Layouter::SetNodePosition(const AbstractNodePointer& node, const Vector2& position)
    {
        if (IsNodeTransformationAllowed(node)) {
            Internal::LayoutEvents::SizeLayoutEvent preSetNodePositionEvent(Internal::LayoutEvents::LayoutEvent::PreSetNodePosition, node, position);
            GetEventSource().DispatchEvent(preSetNodePositionEvent);
            node.SetPosition(position);
        }
    }

    /******************************************************************************
     *  SetNodeScale
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::SetNodeScale(Node2D& node, const Vector2& scale)
    {
        AbstractNodePointer abstractNode(&node);
        Layouter::SetNodeScale(abstractNode, scale);
    }
#endif

    void Layouter::SetNodeScale(const AbstractNodePointer& node, const Vector2& scale)
    {
        if ((node.GetLayouter() == DefaultLayouter::GetInstance())) {
            Internal::LayoutEvents::SizeLayoutEvent preSetNodeScaleEvent(Internal::LayoutEvents::LayoutEvent::PreSetNodeScale, node, scale);
            GetEventSource().DispatchEvent(preSetNodeScaleEvent);
            node.SetScale(scale);
        }
    }

    /******************************************************************************
    *  Vector2Min
    ******************************************************************************/
    Vector2 Layouter::Vector2Min(const Vector2& size1, const Vector2& size2)
    {
        const Float width1 = size1.GetX();
        const Float height1 = size1.GetY();
        const Float width2 = size2.GetX();
        const Float height2 = size2.GetY();
        const Vector2 lMin(Math::Minimum(width1, width2),
                           Math::Minimum(height1, height2));
        return lMin;
    }

    /******************************************************************************
    *  Vector2Max
    ******************************************************************************/
    Vector2 Layouter::Vector2Max(const Vector2& size1, const Vector2& size2)
    {
        const Float width1 = size1.GetX();
        const Float height1 = size1.GetY();
        const Float width2 = size2.GetX();
        const Float height2 = size2.GetY();
        const Vector2 lMax(Math::Maximum(width1, width2),
                           Math::Maximum(height1, height2));
        return lMax;
    }

    /******************************************************************************
     *  MinimumSizeDefault
     ******************************************************************************/
    const Vector2& Layouter::MinimumSizeDefault()
    {
        static Vector2 defaultMinimumSize(0.0F, 0.0F);
        return defaultMinimumSize;
    }

    /******************************************************************************
     *  MaximumSizeDefault
     ******************************************************************************/
    const Vector2& Layouter::MaximumSizeDefault()
    {
        static Vector2 defaultMaximumSize(LAYOUT_MAX_VALUE, LAYOUT_MAX_VALUE);
        return defaultMaximumSize;
    }

    /******************************************************************************
     *  SizeDefault
     ******************************************************************************/
    const Vector2& Layouter::SizeDefault()
    {
        static Vector2 defaultSize(-1.0F, -1.0F);
        return defaultSize;
    }

    /******************************************************************************
     *  MarginDefault
     ******************************************************************************/
    const Margin& Layouter::MarginDefault()
    {
        static Margin defaultMargin(0,0);
        return defaultMargin;
    }

    /******************************************************************************
     *  PaddingDefault
     ******************************************************************************/
    const Padding& Layouter::PaddingDefault()
    {
        static Padding defaultPadding(0,0);
        return defaultPadding;
    }

    /******************************************************************************
     *  GetLanguageSensitiveDirection
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    LayoutAlignment::LayoutDirection::Enum Layouter::GetLanguageSensitiveDirection(const Node2D& node)
    {
        AbstractNodePointer abstractNode(const_cast<Node2D*>(&node));
        return GetLanguageSensitiveDirection(abstractNode);
    }
#endif

    LayoutAlignment::LayoutDirection::Enum Layouter::GetLanguageSensitiveDirection(const AbstractNodePointer& node)
    {
        AbstractNodePointer current = node;
        LayoutAlignment::LayoutDirection::Enum direction = LayoutAlignment::LayoutDirection::InheritDirection;
        while (current.IsValid() && (direction == LayoutAlignment::LayoutDirection::InheritDirection)) {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, current, CANDERA_LINT_REASON_NONCONST)
            direction = GetLayoutDirection(*current.ToCanderaObject());
            if (direction == LayoutAlignment::LayoutDirection::AutomaticDirection) {
                direction = GetAutomaticDirectionBehavior();
            }
            if (direction == LayoutAlignment::LayoutDirection::InheritDirection) {
                current = current.GetParent();
                if (!current.IsValid()) {
                    direction = LayoutAlignment::LayoutDirection::CultureDirection;
                }
            }
        }

        if (direction == LayoutAlignment::LayoutDirection::CultureDirection) {
#if defined(CANDERA_GLOBALIZATION_ENABLED)
            Culture::SharedPointer currentCulture = CultureManager::GetInstance().GetCurrentCulture();
            if (currentCulture != 0) {
                direction = (currentCulture->GetTextDirection() == Globalization::LeftToRight) ? LayoutAlignment::LayoutDirection::LeftToRightDirection : LayoutAlignment::LayoutDirection::RightToLeftDirection;
            }
            else
#endif
            {
                direction = LayoutAlignment::LayoutDirection::LeftToRightDirection;
            }
        }

        return direction;
    }

    /************************************************************************
     * GetParentLanguageSensitiveDirection
     ************************************************************************/
#ifdef CANDERA_2D_ENABLED
    LayoutAlignment::LayoutDirection::Enum Layouter::GetParentLanguageSensitiveDirection(const Node2D& node)
    {
        AbstractNodePointer abstractNode(const_cast<Node2D*>(&node));
        return GetParentLanguageSensitiveDirection(abstractNode);
    }
#endif

    LayoutAlignment::LayoutDirection::Enum Layouter::GetParentLanguageSensitiveDirection(const AbstractNodePointer& node)
    {
        AbstractNodePointer parent = node.GetParent();
        if (parent.IsValid()) {
            return GetLanguageSensitiveDirection(parent);
        }
        return GetLanguageSensitiveDirection(node);
    }

    /************************************************************************
     * GetLanguageSensitiveLeftMargin
     ************************************************************************/
#ifdef CANDERA_2D_ENABLED
    Candera::Int16 Layouter::GetLanguageSensitiveLeftMargin(const Node2D& node, const Margin& margin)
    {
        AbstractNodePointer abstractNode(const_cast<Node2D*>(&node));
        return GetLanguageSensitiveLeftMargin(abstractNode, margin);
    }
#endif

    Candera::Int16 Layouter::GetLanguageSensitiveLeftMargin(const AbstractNodePointer& node, const Margin& margin)
    {
        return (GetLanguageSensitiveDirection(node) == LayoutAlignment::LayoutDirection::RightToLeftDirection) ? margin.GetRight() : margin.GetLeft();
    }

    /************************************************************************
     * OnStretchBehaviorChanged
     ************************************************************************/
    void Layouter::OnStretchBehaviorChanged(DynamicPropertyHost* host, const DynamicProperties::ValueChangedArgs<LayoutAlignment::StretchBehavior::Enum>& args)
    {
        if ((0 != host) && (args.OldValue() != LayoutAlignment::StretchBehavior::None)) {
            CanderaObject* object = static_cast<CanderaObject*>(host);
#ifdef CANDERA_2D_ENABLED
            Node2D* node2D = Dynamic_Cast<Node2D*>(object);
            if (0 != node2D) {
                if (ArabicLayouterPatch::IsSceneEnabled(*node2D)) {
                    /** ---- Patched Layout ---- **/
                    node2D->SetScale(1.0F, 1.0F);
                    /** ---- Patched Layout end ---- **/
                }
            }
#endif
#ifdef CANDERA_3D_ENABLED
#ifdef CANDERA_2D_ENABLED
            else
#endif
            {
                Node* node = Dynamic_Cast<Node*>(object);
                if (0 != node) {
                    node->SetScale(1.0F, 1.0F, 1.0F);
                }
            }
#endif
            InvalidateLayout(host);
        }
    }


    /******************************************************************************
     *  InvalidateLayout
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void Layouter::InvalidateLayout(const Node2D* node)
    {
        AbstractNodePointer abstractNode(const_cast<Node2D*>(node));
        InvalidateLayout(abstractNode);
    }
#endif

    void Layouter::InvalidateLayout(const AbstractNodePointer& node)
    {
        node.InvalidateLayout();
    }

    /******************************************************************************
     *  InvalidateLayout
     ******************************************************************************/
    void Layouter::InvalidateLayout(DynamicPropertyHost* host)
    {
        CanderaObject* object = static_cast<CanderaObject*>(host);
#ifdef CANDERA_2D_ENABLED
        Node2D* node2D = Dynamic_Cast<Node2D*>(object);
        if (0 != node2D) {
            AbstractNodePointer(node2D).InvalidateLayout();
        }
#endif
#ifdef CANDERA_3D_ENABLED
#ifdef CANDERA_2D_ENABLED
        else{
#endif
            Node* node = Dynamic_Cast<Node*>(object);
            if (0 != node) {
                AbstractNodePointer(node).InvalidateLayout();
            }
#ifdef CANDERA_2D_ENABLED
    }
#endif
#endif
    }

    /******************************************************************************
     *  IsNodeTransformationAllowed
     ******************************************************************************/
    bool Layouter::IsNodeTransformationAllowed(const AbstractNodePointer& node)
    {
        bool hasParentDefaultLayouter = false;

        AbstractNodePointer parent = node.GetParent();
        if (parent.IsValid()) {
            hasParentDefaultLayouter = (parent.GetLayouter() == DefaultLayouter::GetInstance());
        }

        return !hasParentDefaultLayouter;
    }


    Vector2 Layouter::GetAxisAlignedDimension(const Vector2& size, Float rotation)
    {
        return LayouterGetAxisAlignedDimension(size, rotation);
    }

    const Vector2& Layouter::GetOriginalPreferredSize(const CanderaObject& node) {
        return LayouterPrivateDynamicProperties::GetOriginalPreferredSize(node);
    }

}   // namespace Candera

