//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GridLayouter.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Diagnostics/Log.h>

#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"

namespace Candera {
using namespace Diagnostics;
    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine2D);

    class GridLayouterDynamicProperties
    {
    private:
        static UInt8 GetRow(const DynamicProperties::DynamicPropertyHost &object)
        {
            return object.GetValue(CdaDynamicPropertyInstance(Row));
        }

        static void SetRow(DynamicProperties::DynamicPropertyHost &object, UInt8 row)
        {
            static_cast<void>(object.SetValue(CdaDynamicPropertyInstance(Row), row));
        }

        static UInt8 GetColumn(const DynamicProperties::DynamicPropertyHost &object)
        {
            return object.GetValue(CdaDynamicPropertyInstance(Column));
        }

        static void SetColumn(DynamicProperties::DynamicPropertyHost &object, UInt8 column)
        {
            static_cast<void>(object.SetValue(CdaDynamicPropertyInstance(Column), column));
        }

        static UInt8 GetRowSpan(const DynamicProperties::DynamicPropertyHost& object) {
            return object.GetValue(CdaDynamicPropertyInstance(RowSpan));
        }

        static void SetRowSpan(DynamicProperties::DynamicPropertyHost& object, UInt8 rowSpan) {
            static_cast<void>(object.SetValue(CdaDynamicPropertyInstance(RowSpan), rowSpan));
        }

        static UInt8 GetColumnSpan(const DynamicProperties::DynamicPropertyHost& object) {
            return object.GetValue(CdaDynamicPropertyInstance(ColumnSpan));
        }

        static void SetColumnSpan(DynamicProperties::DynamicPropertyHost& object, UInt8 columnSpan) {
            static_cast<void>(object.SetValue(CdaDynamicPropertyInstance(ColumnSpan), columnSpan));
        }

        static void SetGridAutoArrangement(CanderaObject& node, GridAutoArrangement arrangement)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(GridAutoArrangement), arrangement));
        }

        static GridAutoArrangement GetGridAutoArrangement(CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(GridAutoArrangement));
        }

        friend class GridLayouter;
        /**
         * Dynamic property definition.
         */
        CdaDynamicPropertiesDefinition(Candera::GridLayouter, Candera::Layouter);

            CdaDynamicProperty(Row, UInt8);
                CdaDynamicPropertyValueChangedCb(&GridLayouter::OnCellChanged);
                CdaDynamicPropertyFlags(Candera::DynamicProperties::BrowsableForDescendants); 
                CdaDynamicPropertyDescription("The row of this element in the grid.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(Column, UInt8);
                CdaDynamicPropertyValueChangedCb(&GridLayouter::OnCellChanged);
                CdaDynamicPropertyFlags(Candera::DynamicProperties::BrowsableForDescendants);
                CdaDynamicPropertyDescription("The column of this element in the grid.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(RowSpan, UInt8);
                CdaDynamicPropertyValueChangedCb(&GridLayouter::OnCellChanged);
                CdaDynamicPropertyFlags(Candera::DynamicProperties::BrowsableForDescendants);
                CdaDynamicPropertyDefaultValue(1);
                CdaDynamicPropertyDescription("The row span of this element in the grid.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(ColumnSpan, UInt8);
                CdaDynamicPropertyValueChangedCb(&GridLayouter::OnCellChanged);
                CdaDynamicPropertyFlags(Candera::DynamicProperties::BrowsableForDescendants);
                CdaDynamicPropertyDefaultValue(1);
                CdaDynamicPropertyDescription("The column span of this element in the grid.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(GridAutoArrangement, GridAutoArrangement);
                CdaDynamicPropertyValueChangedCb(&GridLayouter::OnAutoArrangementChanged);
                CdaDynamicPropertyDescription("Auto arrangement setting for this layouter.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

        CdaDynamicPropertiesEnd();
    };

    const Candera::DynamicProperties::PropertyHierarchyNode&  GridLayouter::GetPropertyHierarchy() const
    {
        return GridLayouterDynamicProperties::GetPropertyHierarchy();
    }


    UInt8 GridLayouter::GetRow(const DynamicPropertyHost &object)
    {
        return GridLayouterDynamicProperties::GetRow(object);
    }

    void GridLayouter::SetRow(DynamicPropertyHost &object, UInt8 row)
    {
        GridLayouterDynamicProperties::SetRow(object, row);
    }

    UInt8 GridLayouter::GetColumn(const DynamicPropertyHost &object)
    {
        return GridLayouterDynamicProperties::GetColumn(object);
    }

    void GridLayouter::SetColumn(DynamicPropertyHost &object, UInt8 column)
    {
        GridLayouterDynamicProperties::SetColumn(object, column);
    }

    UInt8 GridLayouter::GetRowSpan(const DynamicPropertyHost& object) {
        return GridLayouterDynamicProperties::GetRowSpan(object);
    }

    void GridLayouter::SetRowSpan(DynamicPropertyHost& object, UInt8 rowSpan) {
        GridLayouterDynamicProperties::SetRowSpan(object, rowSpan);
    }

    UInt8 GridLayouter::GetColumnSpan(const DynamicPropertyHost& object) {
        return GridLayouterDynamicProperties::GetColumnSpan(object);
    }

    void GridLayouter::SetColumnSpan(DynamicPropertyHost& object, UInt8 columnSpan) {
        GridLayouterDynamicProperties::SetColumnSpan(object, columnSpan);
    }

#ifdef CANDERA_2D_ENABLED
    void GridLayouter::SetGridAutoArrangement(Node2D& node, GridAutoArrangement arrangement)
    {
        GridLayouterDynamicProperties::SetGridAutoArrangement(node, arrangement);
    }
#endif

    void GridLayouter::SetGridAutoArrangement(CanderaObject& node, GridAutoArrangement arrangement)
    {
        GridLayouterDynamicProperties::SetGridAutoArrangement(node, arrangement);
    }

#ifdef CANDERA_2D_ENABLED
    GridAutoArrangement GridLayouter::GetGridAutoArrangement(Node2D& node)
    {
        return GridLayouterDynamicProperties::GetGridAutoArrangement(node);
    }
#endif

    GridAutoArrangement GridLayouter::GetGridAutoArrangement(CanderaObject& node)
    {
        return GridLayouterDynamicProperties::GetGridAutoArrangement(node);
    }

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GridLayouter::GridLayouter() :
        m_rows(2),
        m_columns(2),
        m_rowData(0),
        m_columnData(0),
        m_cellData(0),
        m_cellPoolHead(0)
    {
        ClearCache();
    }

    GridLayouter::GridLayouter(const GridLayouter& rhs) :
        Base(rhs),
        m_rows(rhs.m_rows),
        m_columns(rhs.m_columns),
        m_rowData(0),
        m_columnData(0),
        m_cellData(0),
        m_cellPoolHead(0)
    {
        ClearCache();
        for (UInt8 i = 0; i < m_rows; i++) {
            m_rowData[i] = rhs.m_rowData[i];
        }

        for (UInt8 i = 0; i < m_columns; i++) {
            m_columnData[i] = rhs.m_columnData[i];
        }
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GridLayouter::~GridLayouter()
    {
        // cleanup is done in UpdateCache()
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1579, Candera::GridLayouter::m_columnData, CANDERA_LINT_REASON_DESTRUCTORMISINTERPRETATION)
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1579, Candera::GridLayouter::m_rowData, CANDERA_LINT_REASON_DESTRUCTORMISINTERPRETATION)

        m_rows = 0;
        m_columns = 0;
        ClearCache();
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GridLayouter* GridLayouter::Create()
    {
        return FEATSTD_NEW(GridLayouter);
    }

    /******************************************************************************
     *  SetLayout
     ******************************************************************************/
    void GridLayouter::SetLayout(UInt8 rows, UInt8 columns)
    {
        m_rows = rows;
        m_columns = columns;

        ClearCache();
    }

    UInt8 GridLayouter::GetLayoutRowIndex(const CanderaObject& node, UInt32 iterationIndex, GridAutoArrangement arrangement) const
    {
        UInt32 row;
        switch (arrangement) {
        case GridAutoArrangementDisabled:
            row = GetRow(node);
            break;
        case GridAutoArrangementHorizontal:
            row = iterationIndex / m_columns; 
            break;
        case GridAutoArrangementVertical:
            row = iterationIndex % m_rows;
            break;
        default:
            row = 0;
            FEATSTD_LOG_WARN("Grid layout arrangement not understood.");
        }

        return static_cast<UInt8>(row);
    }

    UInt8 GridLayouter::GetLayoutColumnIndex(const CanderaObject& node, UInt32 iterationIndex, GridAutoArrangement arrangement) const
    {
        UInt32 column;
        switch (arrangement) {
            case GridAutoArrangementDisabled:
                column = GetColumn(node);
                break;
            case GridAutoArrangementHorizontal:
                column = iterationIndex % m_columns; 
                break;
            case GridAutoArrangementVertical:
                column = iterationIndex / m_rows;
                break;
            default:
                column = 0;
                FEATSTD_LOG_WARN("Grid layout arrangement not understood.");
         }

        return static_cast<UInt8>(column);
    }

    /******************************************************************************
     *  PreprocessCells
     ******************************************************************************/
    void GridLayouter::PreprocessCells(const AbstractNodePointer& node)
    {
        if (node.IsValid()) {
            AbstractNodePointer child = node.GetFirstChild();
            UInt32 nodeCounter = 0;
            GridAutoArrangement arrangement = GetGridAutoArrangement(*node.ToCanderaObject());
            // build cellData grid
            while (child.IsValid()) {
                Layouter* childLayouter = child.GetLayouter();
                if (0 != childLayouter) {
                    UInt8 row = GetLayoutRowIndex(*child.ToCanderaObject(), nodeCounter, arrangement);
                    UInt8 col = GetLayoutColumnIndex(*child.ToCanderaObject(), nodeCounter, arrangement);
                    if ((row < m_rows) && (col < m_columns)) {
                        CellData* cell = &m_cellData[row * m_columns + col];
                        if (cell->m_node.IsValid()) {
                            // ensure at least one new CellData is in the pool
                            if (0 == m_cellPoolHead) {
                                m_cellPoolHead = FEATSTD_NEW(CellData)();
                            }
                            // get CellData from the pool;
                            CellData* cellData = m_cellPoolHead;
                            if (0 != m_cellPoolHead) {
                                m_cellPoolHead->m_measured = false;
                                m_cellPoolHead = m_cellPoolHead->m_next;
                                cellData->m_next = 0;
                            }
                            // add to the end of the list
                            if (0 != cellData) {
                                CellData* tail = cell;
                                while (0 != tail->m_next) {
                                    // search for the list tail
                                    tail = tail->m_next;
                                }
                                tail->m_next = cellData;
                                cell = cellData;
                            }
                        }
                        // set node at the last CellData
                        cell->m_node = child;
                    }
                    ++nodeCounter;
                }
                child = child.GetNextSibling();
            }
        }
    }

    /******************************************************************************
     *  PreprocessData
     ******************************************************************************/
    void GridLayouter::PreprocessData(GridLayouter::Data* data, UInt8 dataLength)
    {
        for (UInt8 i = 0; i < dataLength; ++i) {
            if (IsFixedData(data, i)) {
                data[i].m_size = data[i].m_param;
            }
            else {
                data[i].m_size = -1.0F;
                data[i].m_tempSize = -1.0F;
            }
        }
    }

    /******************************************************************************
     *  UpdateAutoData
     ******************************************************************************/
    void GridLayouter::UpdateAutoData(GridLayouter::Data* data, UInt8 dataLength)
    {
        for (UInt8 i = 0; i < dataLength; ++i) {
            if (IsAutoData(data, i)) {
                data[i].m_size = data[i].m_tempSize;
            }
        }
    }

    /******************************************************************************
     *  IsDataConfigurationComplete
     ******************************************************************************/
    bool GridLayouter::IsDataConfigurationComplete(const GridLayouter::Data* data, UInt8 dataLength)
    {
        for (UInt8 i = 0; i < dataLength; ++i) {
            if (IsAutoData(data, i) && (data[i].m_size < 0.0F)) {
                return false;
            }
        }
        return true;
    }

    /******************************************************************************
     *  UpdateFractionData
     ******************************************************************************/
    bool GridLayouter::UpdateFractionData(GridLayouter::Data* data, UInt8 dataLength, Float availableSize)
    {
        if (IsDataConfigurationComplete(data, dataLength)) {
            Float usedSize = 0.0F;
            Float overallFraction = 0.0F;
            for (UInt8 i = 0; i < dataLength; ++i) {
                if (IsFractionData(data, i)) {
                    overallFraction += data[i].m_param;
                }
                else {
                    usedSize += data[i].m_size;
                }
            }
            FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
            Float remainingSize = (availableSize == Math::MaxFloat()) ? 0.0F : Math::Maximum(Float(0.0F), availableSize - usedSize);
            Float sizeFactor = ((remainingSize == 0.0F) || (overallFraction == 0.0F)) ? 0.0F : (remainingSize / overallFraction);
            for (UInt8 j = 0; j < dataLength; ++j) {
                if (IsFractionData(data, j)) {
                    data[j].m_size = (sizeFactor == 0.0F) ? 0.0F : data[j].m_param * sizeFactor;
                }
            }
            return true;
        }
        return false;
    }

    /******************************************************************************
     *  UpdateFractionDataForInfiniteSize
     ******************************************************************************/
    void GridLayouter::UpdateFractionDataForInfiniteSize(GridLayouter::Data* data, UInt8 dataLength)
    {
        Float overallFraction = 0.0F;
        for (UInt8 k = 0; k < dataLength; ++k) {
            if (IsFractionData(data, k) && (data[k].m_tempSize > 0.0F)) {
                overallFraction += data[k].m_param;
            }
        }
        Float maxSize = 0.0F;
        for (UInt8 i = 0; i < dataLength; ++i) {
            if (IsFractionData(data, i) && (data[i].m_param > 0.0F) && (data[i].m_tempSize > 0.0F)) {
                Float currentSize = data[i].m_tempSize * (overallFraction / data[i].m_param);
                if (currentSize > maxSize) {
                    maxSize = currentSize;
                }
            }
        }
        Float sizeFactor = ((maxSize == 0.0F) || (overallFraction == 0.0F)) ? 0.0F : (maxSize / overallFraction);
        for (UInt8 j = 0; j < dataLength; ++j) {
            if (IsFractionData(data, j)) {
                data[j].m_size = (sizeFactor == 0.0F) ? 0.0F : data[j].m_param * sizeFactor;
            }
        }
    }

    /******************************************************************************
     *  TryGetDataSize
     ******************************************************************************/
    bool GridLayouter::TryGetDataSize(const Data* data, UInt8 index, Float& size)
    {
        if (IsAutoData(data, index)) {
            if (data[index].m_size >= 0.0F) {
                size = data[index].m_size;
            }
            else {
                size = Math::MaxFloat();
            }
            return true;
        }
        else if (IsFixedData(data, index)) {
            size = data[index].m_size;
            return true;
        }
        else {
            if (data[index].m_size >= 0.0F) {
                size = data[index].m_size;
                return true;
            }
        }
        return false;
    }

    /******************************************************************************
     *  TryGetDataSpanSize
     ******************************************************************************/
    bool GridLayouter::TryGetDataSpanSize(const Data* data, UInt8 dataLength, UInt8 index, UInt8 span, Float& size)
    {
        for (UInt8 i = 1; i < span; ++i) {
            if (index + i < dataLength) {
                Float nextSize = Math::MaxFloat();
                if (TryGetDataSize(data, index + i, nextSize)) {
                    size += nextSize;
                    if (size > Math::MaxFloat()) {
                        size = Math::MaxFloat();
                    }
                }
                else {
                    return false;
                }
            }
        }
        return true;
    }

    /******************************************************************************
     *  GetFixedDataSpanSize
     ******************************************************************************/
    Float GridLayouter::GetFixedDataSpanSize(const Data* data, UInt8 dataLength, UInt8 index, UInt8 span)
    {
        Float size = 0.0F;
        for (UInt8 i = 0; i < span; ++i) {
            if ((index + i < dataLength) && IsFixedData(data, index + i)) {
                size += data[index + i].m_size;
            }
        }
        return size;
    }

    /******************************************************************************
     *  CountAutoDataSpan
     ******************************************************************************/
    UInt8 GridLayouter::CountAutoDataSpan(const Data* data, UInt8 dataLength, UInt8 index, UInt8 span)
    {
        UInt8 count = 0;
        for (UInt8 i = 0; i < span; ++i) {
            if ((index + i < dataLength) && IsAutoData(data, index + i)) {
                ++count;
            }
        }
        return count;
    }

    /******************************************************************************
     *  UpdateAutoDataMaxSize
     ******************************************************************************/
    void GridLayouter::UpdateAutoDataMaxSize(Data* data, UInt8 index, Float size)
    {
        // also fraction data needs the max size information if the measure size is infinite
        if ((IsAutoData(data, index) || IsFractionData(data, index)) && (data[index].m_tempSize < size)) {
            data[index].m_tempSize = size;
        }
    }

    /******************************************************************************
     *  UpdateSpanAutoDataMaxSize
     ******************************************************************************/
    void GridLayouter::UpdateSpanAutoDataMaxSize(Data* data, UInt8 dataLength, UInt8 index, UInt8 span, Float size)
    {
        if (span > 1) {
            UInt8 autoData = CountAutoDataSpan(data, dataLength, index, span);
            if (autoData > 0) {
                Float autoRemainingSize = size - GetFixedDataSpanSize(data, dataLength, index, span);
                Float autoSize = autoRemainingSize / Float(autoData);
                for (UInt8 i = 0; i < span; ++i) {
                    // also fraction data needs the max size information if the measure size is infinite
                    if ((index + i < dataLength) && (IsAutoData(data, index + i) || IsFractionData(data, index + i))) {
                        --autoData;
                        if (data[index].m_size > autoSize) {
                            // only auto data is allowed to reduce the remaining size
                            if (IsAutoData(data, index + i)) {
                                autoRemainingSize -= data[index].m_size;
                                autoSize = ((autoData > 0) && (autoRemainingSize > 0.0F)) ? autoRemainingSize / Float(autoData) : 0.0F;
                            }
                        }
                        else if (data[index].m_tempSize > autoSize) {
                            // only auto data is allowed to reduce the remaining size
                            if (IsAutoData(data, index + i)) {
                                autoRemainingSize -= data[index].m_tempSize;
                                autoSize = ((autoData > 0) && (autoRemainingSize > 0.0F)) ? autoRemainingSize / Float(autoData) : 0.0F;
                            }
                        }
                        else {
                            UpdateAutoDataMaxSize(data, index + i, autoSize);
                            // only auto data is allowed to reduce the remaining size
                            if (IsAutoData(data, index + i)) {
                                autoRemainingSize -= autoSize;
                            }
                        }
                    }
                }
            }
        }
        else {
            UpdateAutoDataMaxSize(data, index, size);
        }
    }

    /******************************************************************************
     *  GetCompleteDataSize
     ******************************************************************************/
    Float GridLayouter::GetCompleteDataSize(const Data* data, UInt8 dataLength)
    {
        Float size = 0.0F;
        for (UInt8 i = 0; i < dataLength; ++i) {
            if (data[i].m_size > 0.0F) {
                size += data[i].m_size;
            }
        }
        return size;
    }

    /******************************************************************************
     *  OnMeasure
     ******************************************************************************/
    Vector2 GridLayouter::OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea)
    {
        bool complete = false;
        bool isRowSizeComplete = false;
        bool isColumnSizeComplete = false;
        bool isSimpleAutoComplete = false;
        bool isSpanAutoComplete = false;
        bool isRowFractionRemainingSizeInfinite = false;
        bool isColumnFractionRemainingSizeInfinite = false;
        PreprocessCells(node);
        PreprocessData(m_rowData, m_rows);
        PreprocessData(m_columnData, m_columns);
        do {
            bool itemSkipped = false;
            bool spanSkipped = false;
            // only update the fraction after all automatic configured rows and columns are measured (including the row/column span items)
            // located at the to of the loop to prevent unnecessary checks for simple grids without fractions
            if (isSpanAutoComplete) {
                if ((!isRowSizeComplete) && UpdateFractionData(m_rowData, m_rows, clientArea.GetY())) {
                    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
                    isRowFractionRemainingSizeInfinite = clientArea.GetY() == Math::MaxFloat();
                    isRowSizeComplete = true;
                }
                if ((!isColumnSizeComplete) && UpdateFractionData(m_columnData, m_columns, clientArea.GetX())) {
                    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
                    isColumnFractionRemainingSizeInfinite = clientArea.GetX() == Math::MaxFloat();
                    isColumnSizeComplete = true;
                }
            }
            for (UInt8 row = 0; row < m_rows; ++row) {
                Float height = Math::MaxFloat();
                // skip row for this pass if not yet configurable due to fraction configuration
                if (TryGetDataSize(m_rowData, row, height)) {
                    for (UInt8 column = 0; column < m_columns; ++column) {
                        Float width = Math::MaxFloat();
                        // skip column for this pass if not yet configurable due to fraction configuration
                        if (TryGetDataSize(m_columnData, column, width)) {
                            // process all items in the cell
                            CellData* cellData = &m_cellData[row * m_columns + column];
                            while (0 != cellData) {
                                AbstractNodePointer child = cellData->m_node;
                                if (child.IsValid()) {
                                    if (!cellData->m_measured) {
                                        Layouter* childLayouter = child.GetLayouter();
                                        if (0 != childLayouter) {
                                            Float childHeight = height;
                                            Float childWidth = width;
                                            UInt8 rowSpan = GetRowSpan(*child.ToCanderaObject());
                                            UInt8 columnSpan = GetColumnSpan(*child.ToCanderaObject());
                                            bool measure = true;
                                            // first pass will finish the simple automatic configuration without row/column span
                                            // the optional second pass will finish the complex automatic configuration with row/column span
                                            // the optional third pass will finish the fraction configuration
                                            if ((rowSpan > 1) || (columnSpan > 1)) {
                                                if (!isSimpleAutoComplete) {
                                                    measure = false;
                                                    spanSkipped = true;
                                                }
                                            }
                                            if (measure &&
                                                TryGetDataSpanSize(m_rowData, m_rows, row, rowSpan, childHeight) &&
                                                TryGetDataSpanSize(m_columnData, m_columns, column, columnSpan, childWidth)) {
                                                FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
                                                if (IsFractionData(m_rowData, row) && (clientArea.GetY() == Math::MaxFloat())) {
                                                    childHeight = Math::MaxFloat();
                                                }
                                                FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
                                                if (IsFractionData(m_columnData, column) && (clientArea.GetX() == Math::MaxFloat())) {
                                                    childWidth = Math::MaxFloat();
                                                }
                                                childLayouter->Measure(child, Vector2(childWidth, childHeight));
                                                cellData->m_measured = true;
                                                const Vector2 preferredChildSize = childLayouter->GetClientSize(child);
                                                // prepare the update of the automatic configuration of the row/column size if bigger than the current value
                                                // the actual value is not allowed to be changed because it is needed for the remaining items of that pass
                                                // therefore a temporary value is used.
                                                UpdateSpanAutoDataMaxSize(m_rowData, m_rows, row, rowSpan, preferredChildSize.GetY());
                                                UpdateSpanAutoDataMaxSize(m_columnData, m_columns, column, columnSpan, preferredChildSize.GetX());
                                            }
                                            else {
                                                itemSkipped = true;
                                            }
                                        }
                                    }
                                }
                                else {
                                    // empty cell detected => potential row/column with a size of 0
                                    UpdateAutoDataMaxSize(m_rowData, row, 0.0F);
                                    UpdateAutoDataMaxSize(m_columnData, column, 0.0F);

                                }
                                cellData = cellData->m_next;
                            }
                        }
                        else {
                            itemSkipped = true;
                        }
                    }
                }
                else {
                    itemSkipped = true;
                }
            }
            // update the actual automatic configuration with the temporary results of this pass
            UpdateAutoData(m_rowData, m_rows);
            UpdateAutoData(m_columnData, m_columns);
            isSimpleAutoComplete = true;
            if (!spanSkipped) {
                // if no item with a row/column span has been skipped in this pass then the fraction configuration can be updated in the next pass (if a next pass is required)
                isSpanAutoComplete = true;
                if (!itemSkipped) {
                    // if no item was skipped then we are finished
                    complete = true;
                }
            }
        } while (!complete);
        if (isRowFractionRemainingSizeInfinite) {
            UpdateFractionDataForInfiniteSize(m_rowData, m_rows);
        }
        if (isColumnFractionRemainingSizeInfinite) {
            UpdateFractionDataForInfiniteSize(m_columnData, m_columns);
        }
        for (UInt8 i = 0; i < m_rows; ++i) {
            for (UInt8 j = 0; j < m_columns; ++j) {
                // clean up the cells CellData
                CellData& cell = m_cellData[i * m_columns + j];
                cell.m_measured = false;
                cell.m_node = AbstractNodePointer();
                if (0 != cell.m_next) {
                    // clean up the CellData pool
                    CellData* current = cell.m_next;
                    while (0 != current->m_next) {
                        current->m_measured = false;
                        current->m_node = AbstractNodePointer();
                        current = current->m_next;
                    }
                    current->m_measured = false;
                    current->m_node = AbstractNodePointer();
                    // add existing pool at tail of the cell list
                    current->m_next = m_cellPoolHead;
                    // put cell list back into the pool
                    m_cellPoolHead = cell.m_next;
                    cell.m_next = 0;
                }
            }
        }
        // get the size configuration as preferred size
        Float height = GetCompleteDataSize(m_rowData, m_rows);
        Float width = GetCompleteDataSize(m_columnData, m_columns);
        return Vector2(width, height);
    }

    /******************************************************************************
     *  OnArrange
     ******************************************************************************/
    void GridLayouter::OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea)
    {
        if (node.IsValid()) {
            // update the row/column fraction configuration with the concrete client area
            (void)UpdateFractionData(m_rowData, m_rows, clientArea.GetHeight());
            (void)UpdateFractionData(m_columnData, m_columns, clientArea.GetWidth());
            const LayoutAlignment::LayoutDirection::Enum orientation = GetLanguageSensitiveDirection(node);
            const Float clientWidth = clientArea.GetWidth();
            Float width = 0.0F;
            Float height = 0.0F;
            //Width has to be calculated here in order to cope with the configured orientation.
            //Height was already computed in the measure phase.
            // calculate the position according to the row/column configuration
            for (UInt8 i = 0; i < m_columns; i++) {
                if (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                    width += m_columnData[i].m_size;
                    m_columnData[i].m_pos = clientWidth - width;
                }
                else {
                    m_columnData[i].m_pos = width;
                    width += m_columnData[i].m_size;
                }
            }

            for (UInt8 j = 0; j < m_rows; j++) {
                m_rowData[j].m_pos = height;
                height += m_rowData[j].m_size;
            }

            AbstractNodePointer child = node.GetFirstChild();
            UInt32 nodeCounter = 0;
            GridAutoArrangement arrangement = GetGridAutoArrangement(*node.ToCanderaObject());

            while (child.IsValid()) {
                Layouter* childLayouter = child.GetLayouter();
                if (childLayouter != 0) {
                    UInt32 row = GetLayoutRowIndex(*child.ToCanderaObject(), nodeCounter, arrangement);
                    UInt32 col = GetLayoutColumnIndex(*child.ToCanderaObject(), nodeCounter, arrangement);
                    UInt8 rowSpan = GetRowSpan(*child.ToCanderaObject());
                    UInt8 colSpan = GetColumnSpan(*child.ToCanderaObject());

                    ++nodeCounter;

                    if ((row < m_rows) && (col < m_columns)) {
                        const Float posX = m_columnData[col].m_pos;
                        const Float posY = m_rowData[row].m_pos;

                        Float cellWidth = 0.0F;
                        Float cellHeight = 0.0F;
                        // calculate the width and height according to the row/column configuration and the row/column span
                        for (UInt8 x = 0; x < colSpan; ++x) {
                            if ((col + x) < m_columns) {
                                cellWidth += m_columnData[col + x].m_size;
                            }
                        }

                        for (UInt8 y = 0; y < rowSpan; ++y) {
                            if ((row + y) < m_rows) {
                                cellHeight += m_rowData[row + y].m_size;
                            }
                        }

                        Rectangle childArea((orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection) ? (posX + m_columnData[col].m_size - cellWidth) : posX, posY, cellWidth, cellHeight);
                        childLayouter->Arrange(child, childArea);
                    }
                }

                child = child.GetNextSibling();
            }
            Node2D* node2D = node.ToNode2D();
            if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
                /** ---- Patched Layout ---- **/
                SetArrangeActualSize(Vector2(width, height));
                /** ---- Patched Layout end ---- **/
            }
            else {
                /** ---- Old Layout ---- **/
                SetNodePosition(node, Vector2(clientArea.GetLeft(), clientArea.GetTop()));
                /** ---- Old Layout end ---- **/
            }
        }
    }


    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Layouter* GridLayouter::Clone() const
    {
        return CANDERA_NEW(GridLayouter)(*this);
    }

    /******************************************************************************
    *  UpdateCache
    ******************************************************************************/
    void GridLayouter::ClearCache()
    {
        FEATSTD_DELETE_ARRAY(m_rowData);
        m_rowData = 0;
        FEATSTD_DELETE_ARRAY(m_columnData);
        m_columnData = 0;
        FEATSTD_DELETE_ARRAY(m_cellData);
        m_cellData = 0;

        CellData* current = m_cellPoolHead;
        while (0 != current) {
            // delete the CellData pool 
            CellData* next = current->m_next;
            FEATSTD_DELETE(current);
            current = next;
        }
        m_cellPoolHead = 0;

        if (m_rows > 0) {
            m_rowData = FEATSTD_NEW_ARRAY(Data, m_rows);
        }
        if (m_columns > 0) {
            m_columnData = FEATSTD_NEW_ARRAY(Data, m_columns);
        }
        if ((m_columns > 0) && (m_rows > 0)) {
            m_cellData = FEATSTD_NEW_ARRAY(CellData, static_cast<SizeType>(m_columns) * static_cast<SizeType>(m_rows));
        }
    }
}   // namespace Candera
