//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DOCKPANEL_LAYOUTER_H)
#define CANDERA_DOCKPANEL_LAYOUTER_H

#include <Candera/EngineBase/Layout/Layouter.h>

#define ENUM_DATA_TYPE \
    ENUM_DATA_TYPE_BEGIN(DockSide) \
        ENUM_DATA_TYPE_ITEM(DockLeft) \
        ENUM_DATA_TYPE_ITEM(DockTop) \
        ENUM_DATA_TYPE_ITEM(DockRight) \
        ENUM_DATA_TYPE_ITEM(DockBottom) \
    ENUM_DATA_TYPE_END(DockSide)

#include <Candera/System/MetaInfo/EnumDataType.h>

namespace Candera {
    /** @addtogroup Layout
     *  @{
     */

    /**
     *  @brief The Dock Panel Layouter provides an easy docking of elements to the left,
     *  right, top, bottom or center of the panel. The dock side of an element is defined
     *  by the attached dynamic DockSide property. To dock an element to the center of the
     *  panel, it must be the last child of the panel.
     */
    class DockPanelLayouter : public Layouter {
        public:
            /**
             *  Gets the instance of this class.
             *  @return A pointer to the instance of this class.
             */
            static DockPanelLayouter* GetInstance();

            /**
             *  This is a singleton so dispose does nothing.
             */
            virtual void Dispose() override {}

            /**
             * Sets the value that represents the dock side.
             * @param   node        The node for which the dynamic property is set.
             * @param   dockSide    Desired dock side.
             * @return Whether the property could be set or not
             */
            static bool SetDockSide(CanderaObject& node, DockSide dockSide);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static bool SetDockSide(Node2D& node, DockSide dockSide));
#endif

            /**
             * Gets the value that represents the dock side.
             * @param   node        The node for which the dynamic property is set.
             * @return  The dock side which was set for the node.
             */
            static DockSide GetDockSide(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static DockSide GetDockSide(const Node2D& node));
#endif

            /**
             * Gets the language sensitive value that represents the dock side.
             * @param   node        The node for which the dynamic property is set.
             * @return  The dock side which was set for the node.
             */
            static DockSide GetLanguageSensitiveDockSide(const AbstractNodePointer& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static DockSide GetLanguageSensitiveDockSide(const Node2D& node));
#endif

            /**
            * This is a singleton class - cloning returns the singleton instance.
            * @return singleton instance of DockPanelLayouter
            */
            virtual Layouter* Clone() const override { return GetInstance(); }

    protected:

        /**
        *  This method is called as first layout pass to collect all the data which is needed to layout the child nodes.
        *  @param node The node which child nodes shall be layouted.
        *  @param clientArea Area where layout shall be applied.
        *  @return Vector describing height and width of whole layouted rectangle.
        */
        virtual Candera::Vector2 OnMeasure(const AbstractNodePointer& node, const Candera::Vector2& clientArea) override;
        /**
        *  This method is called as second layout pass to arrange the child nodes (position and scale).
        *  @param node Node which children shall be arranged.
        *  @param clientArea Rectangle where layout shall be applied.
        */
            virtual void OnArrange(const AbstractNodePointer& node, const Candera::Rectangle& clientArea) override;
            void MeasureChildren(const AbstractNodePointer& firstChild, const Candera::Vector2& clientArea, Candera::Vector2& usedArea) const;

#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                void MeasureChildren(Candera::Node2D* firstChild, const Candera::Vector2& clientArea, Candera::Vector2& usedArea) const);
#endif

            void ArrangeChildren(LayoutAlignment::LayoutDirection::Enum layoutDirection, const AbstractNodePointer& firstChild, const Candera::Vector2& contentAreaPosition, const Candera::Vector2& contentAreaSize) const;

#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                void ArrangeChildren(LayoutAlignment::LayoutDirection::Enum layoutDirection, Candera::Node2D* firstChild, const Candera::Vector2& contentAreaPosition, const Candera::Vector2& contentAreaSize) const);
#endif

        private:
            friend class DockPanelLayouterDynamicProperties;

            DockPanelLayouter();
            DockPanelLayouter(const DockPanelLayouter&);

            static const DockSide& DockSideDefault() {
                static DockSide defaultAlignment = DockTop;
                return defaultAlignment;
            }

            static void OnDockSideChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<DockSide>& /*args*/) {
                InvalidateLayout(obj);
            }

            CdaDynamicPropertiesDeclaration();
    };

    /** @} */ // end of Layout

}   // namespace Candera

#endif  // CANDERA_LAYOUTER_EXT_H
