//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BoundingRectTraverser.h"

#include <Candera/EngineBase/Layout/Layouter.h>
#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Group2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#endif

namespace Candera {

#ifdef CANDERA_2D_ENABLED
    BoundingRectTraverser::BoundingRectTraverser(const Node2D& root) :
        m_initializedBounds(false),
        m_initializedFallbackPosition(false),
        m_root(root)
    {
        m_bounds.minX = Math::MaxFloat();
        m_bounds.minY = Math::MaxFloat();
        m_bounds.maxX = -Math::MaxFloat();
        m_bounds.maxY = -Math::MaxFloat();
    }


    void BoundingRectTraverser::GetBoundingRectangle(Rectangle& rect) const
    {
        if (m_initializedBounds) {
            rect.SetPosition(m_bounds.minX, m_bounds.minY);
            rect.SetSize((m_bounds.maxX - m_bounds.minX), (m_bounds.maxY - m_bounds.minY));
        }
        else {
            rect.SetPosition(m_fallbackPosition);
            rect.SetSize(0.0F, 0.0F);
        }
    }
    ConstTreeTraverser2D::TraverserAction BoundingRectTraverser::ProcessNode(const Node2D& node) 
    {
        if ((Layouter::IsSizeSet(static_cast<const CanderaObject&>(node))) || ((!node.IsTypeOf(Group2D::GetTypeId())) && (!node.IsTypeOf(Camera2D::GetTypeId())))) {
            Vector2 point[4];

            Rectangle boundingRect;
            node.GetComputedLayoutRectangle(boundingRect);

            Matrix3x2 transform(node.GetCompositeTransform());

            const Node2D* parent = node.GetParent();
            while ((parent != 0) && (parent != &m_root)) {
                transform *= parent->GetCompositeTransform();
                parent = parent->GetParent();
            }

            const Float left = boundingRect.GetLeft();
            const Float top = boundingRect.GetTop();
            const Float width = boundingRect.GetWidth();
            const Float height = boundingRect.GetHeight();

            const Float right = left + width;
            const Float bottom = top + height;

            point[0] = transform.Multiply(Vector2(left, top));
            point[1] = transform.Multiply(Vector2(right, top));
            point[2] = transform.Multiply(Vector2(right, bottom));
            point[3] = transform.Multiply(Vector2(left, bottom));

            m_initializedBounds = true;
            for (Int i = 0; i < 4; i++) {
                const Float x = point[i].GetX();
                const Float y = point[i].GetY();

                m_bounds.minX = Math::Minimum(m_bounds.minX, x);
                m_bounds.minY = Math::Minimum(m_bounds.minY, y);
                m_bounds.maxX = Math::Maximum(m_bounds.maxX, x);
                m_bounds.maxY = Math::Maximum(m_bounds.maxY, y);
            }
        }
        else {
            if ((!m_initializedBounds) && (!m_initializedFallbackPosition)) {
                m_initializedFallbackPosition = true;
                Matrix3x2 transform(node.GetCompositeTransform());
                const Node2D* parent = node.GetParent();
                while ((parent != 0) && (parent != &m_root)) {
                    transform *= parent->GetCompositeTransform();
                    parent = parent->GetParent();
                }

                m_fallbackPosition = transform.Multiply(node.GetPosition());
            }
        }

        return ProceedTraversing;
    }
#endif

}   // namespace Candera
