//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DynamicProperties_DynamicPropertyHost_H)
#define CANDERA_DynamicProperties_DynamicPropertyHost_H

#include <Candera/EngineBase/DynamicProperties/TypeInfo.h>
#include <Candera/EngineBase/DynamicProperties/CallbackArgs.h>
#include <Candera/System/EntityComponentSystem/Entity.h>

#define    CdaDynamicPropertyHost(specName, type, baseType)    \
    virtual const Candera::DynamicProperties::PropertyHierarchyNode& GetPropertyHierarchy() const {    \
        return Candera::DynamicProperties::DynamicPropertyRegistry<type>::GetHierarchy();    \
    }    \
    typedef Candera::DynamicProperties::HostSpecification<type, baseType> specName

namespace Candera {
    namespace DynamicProperties {
        /// @addtogroup DynamicPropertiesBase
        /// @{
        template<typename ValueType, typename SpecType> class DynamicProperty;
        class DynamicPropertyBase;
        class PropertyHierarchyNode;
        class ValueNodeBase;

        /**
            * @brief Represents one host of dynamic properties. Inheriting from this class enables the definition of
            *        dynamic properties for a class.
            */
        class DynamicPropertyHost : public EntityComponentSystem::Entity {

            public:

                /**
                    *  Constructor
                    */
                DynamicPropertyHost();

                /**
                    *  Copy constructor.
                    *  @param obj Source of copy.
                    */
                DynamicPropertyHost(const DynamicPropertyHost& obj);

                /**
                    *  Destructor.
                    */
                virtual ~DynamicPropertyHost();

                /**
                    *  Retrieves the value of a given dynamic property.
                    *  The retrieval works as follows:
                    *    + The value of the property set in this host is returned, if set in this host.
                    *    + The value of the property in the parent host is returned, if inherited and set in the parent host.
                    *    + The default value is returned otherwise.
                    *  @param prop The property, whose value is sought.
                    *  @return The value of prop.
                    */
                template<typename ValueType, typename SpecType> typename TypeInfo<ValueType>::ConstRefType GetValue(const DynamicProperty<ValueType, SpecType>& prop) const;

                /**
                    *  Sets the value of a given dynamic property.
                    *  The setting of values works as follows:
                    *    + The value is coerced using the coerce value callback
                    *    + The value is validated using the validate callback (default values are NOT validated).
                    *    + If the validation was successful, the coerced value is set, otherwise it is discarded.
                    *  @param prop The property, whose value shall be set.
                    *  @param value The value to be set.
                    *  @return true, if setting of the value was successful, false otherwise.
                    */
                template<typename ValueType, typename SpecType> bool SetValue(DynamicProperty<ValueType, SpecType>& prop, typename TypeInfo<ValueType>::ConstRefType value);

                /**
                    *  Clears the value of the given property to the default value.
                    *  @param prop The property that should be cleared.
                    *  @return true, if successful, false otherwise.
                    */
                template<typename ValueType, typename SpecType> bool ClearValue(DynamicProperty<ValueType, SpecType>& prop);

                /**
                    *  Determines, whether a value was set for a specific dynamic property on this
                    *  host.
                    *  @param prop The property to check for a set value.
                    *  @return true, if a value for prop was set for this dynamic property host, false otherwise.
                    */
                bool IsValueSet(const DynamicPropertyBase& prop) const;

                /**
                    *  Assignment operator.
                    *  @param obj R-value of assignment.
                    *  @return This for chaining.
                    */
                DynamicPropertyHost& operator=(const DynamicPropertyHost& obj);

                /**
                    *  The properties associated with this host.
                    *  @return The properties associated with this host.
                    */
                virtual const PropertyHierarchyNode& GetPropertyHierarchy() const;

            private:
                ValueNodeBase* m_values;

                typedef const DynamicPropertyHost* (*ParentProviderFn)(const DynamicPropertyHost*);

                void AddValue(ValueNodeBase* newVal);
                void FreeValues();
                void CloneValues(const DynamicPropertyHost& srcObj);
                ValueNodeBase* FindValue(const DynamicPropertyBase& prop, ParentProviderFn ParentProvider, bool& searchedInAncestor) const;
        };
        /// @}
    }
}

#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>

namespace Candera {
    namespace DynamicProperties {
        template<typename ValueType, typename SpecType>
        typename TypeInfo<ValueType>::ConstRefType DynamicPropertyHost::GetValue(const DynamicProperty<ValueType, SpecType>& prop) const
        {
            bool foundInAncestor;
            typename DynamicProperty<ValueType, SpecType>::Value* val =
                static_cast<typename DynamicProperty<ValueType, SpecType>::Value*>(FindValue(prop, &SpecType::HostType::ParentProvider, foundInAncestor));
            return (val == 0) ? prop.GetDefaultValue() : val->Value();
        }

        template<typename ValueType, typename SpecType>
        bool DynamicPropertyHost::SetValue(DynamicProperty<ValueType, SpecType>& prop, typename TypeInfo<ValueType>::ConstRefType value)
        {
            bool ok = true;
            bool searchedInAncestor = false;

            typename DynamicProperty<ValueType, SpecType>::Value* val =
                static_cast<typename DynamicProperty<ValueType, SpecType>::Value*>(FindValue(prop, &SpecType::HostType::ParentProvider, searchedInAncestor));

            // determine old value
            ValueType oldValue;
            if (val == 0) {
                oldValue = prop.GetDefaultValue();
            }
            else {
                oldValue = val->Value();
            }

            if ((!searchedInAncestor) && (oldValue == value)) {
                return true;
            }

            // compute new value
            ValueType newValue(value);

            // default values are NOT coerced
            if (!(value == prop.GetDefaultValue())) {
                CoerceValueArgs<typename SpecType::ValueType> coerceArgs(prop, newValue);
                SpecType::CoerceValueCallback()(this, coerceArgs);
            }

            // values are always validated before being set
            ok = SpecType::ValidateValueCallback()(newValue);
            if (!ok) {
                return false;
            }

            if (!prop.IsValuePropagated() && newValue == prop.GetDefaultValue()) {
                ClearValue(prop);
                return true;
            }

            if (val == 0 || searchedInAncestor) {
                val = prop.New(newValue);
                ok = val != 0;
                if (ok) {
                    AddValue(val);
                }
                else {
                    return false;
                }
            }
            else {
                val->SetValue(newValue);
            }

            ValueChangedArgs<typename SpecType::ValueType> valueChanged(prop, newValue, oldValue);
            SpecType::ValueChangedCallback()(this, valueChanged);
            return ok;
        }

        template<typename ValueType, typename SpecType> bool DynamicPropertyHost::ClearValue(DynamicProperty<ValueType, SpecType>& prop)
        {
            ValueNodeBase* prev;
            ValueNodeBase* val = prop.FindValue(m_values, prev);
            bool ok = val != 0;
            if (ok) {
                typename DynamicProperty<ValueType, SpecType>::Value* value = static_cast<typename DynamicProperty<ValueType, SpecType>::Value*>(val);

                ValueType oldValue = value->Value();
                ValueType newValue = prop.GetDefaultValue();

                if (prev == 0) {
                    m_values = val->Next();
                }
                else {
                    prev->SetNext(val->Next());
                }
                prop.Free(val);

                if (!(oldValue == newValue)) {
                    ValueChangedArgs<typename SpecType::ValueType> valueChanged(prop, newValue, oldValue);
                    SpecType::ValueChangedCallback()(this, valueChanged);
                }
            }
            return ok;
        }
    }
}
#endif    // !defined(CANDERA_DynamicProperties_DynamicPropertyHost_H)
