//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DynamicProperties_CallbackArgs_H)
#define CANDERA_DynamicProperties_CallbackArgs_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/DynamicProperties/TypeInfo.h>

namespace Candera {
    namespace DynamicProperties {

        class DynamicPropertyHost;
        class DynamicPropertyBase;

        /**
            * @brief Base class for the representation of the arguments of a callback function.
            */
        class CallbackArgsBase {
            public:
                /**
                    *  Constructor
                    *  @param prop The property concerned in the callback
                    */
                CallbackArgsBase(DynamicPropertyBase& prop) :
                    m_property(prop)
                {
                }

                /**
                    *  Retrieves the property concerned in the callback.
                    *  @return The property concerned in the callback.
                    */
                DynamicPropertyBase& Property() const
                {
                    return m_property;
                }

            private:
                DynamicPropertyBase& m_property;

                /**
                    *  Copy Constructor.
                    */
                CallbackArgsBase(const CallbackArgsBase& obj) : m_property(obj.m_property)
                {
                }

                /**
                    *  Assignment operator.
                    */
                CallbackArgsBase& operator=(const CallbackArgsBase&)
                {
                    return *this;
                }
        };

        /**
            * @brief Class for arguments of value changed callback functions.
            */
        template<typename T> class ValueChangedArgs : public CallbackArgsBase {
            public:
                // const& type definition for value
                typedef typename TypeInfo<T>::ConstRefType ValueConstRefType;

                /**
                    *  Signature for value changed callback functions.
                    *  @param host The host, whose value has changed.
                    *  @param args Information about the changed property.
                    */
                typedef void (*Signature)(DynamicPropertyHost* host, const ValueChangedArgs<T>& args);

                /**
                    *  Constructor.
                    *  @param prop The property concerned.
                    *  @param newValue The new value of prop.
                    *  @param oldValue The old value of prop.
                    */
                ValueChangedArgs(DynamicPropertyBase& prop, ValueConstRefType newValue, ValueConstRefType oldValue);

                /**
                    *  Retrieves the new value of the property.
                    *  @return The new value of the property.
                    */
                ValueConstRefType NewValue() const;

                /**
                    *  Retrieves the old value of the property.
                    *  @return The old value of the property.
                    */
                ValueConstRefType OldValue() const;

                /**
                    *  The default callback function to use for property changes, if
                    *  no callback was defined explicitly. By default, nothing happens,
                    *  if the value of a dynamic property changes.
                    *  @param host The host, whose property change should be handled.
                    *  @param changeArgs The changes in this->Property() (old and new value).
                    */
                static void DefaultCallback(DynamicPropertyHost* host, const ValueChangedArgs<T>& changeArgs);

            private:
                ValueConstRefType m_newValue;
                ValueConstRefType m_oldValue;
        };

        template<typename T>
        ValueChangedArgs<T>::ValueChangedArgs(DynamicPropertyBase& prop, ValueConstRefType newValue, ValueConstRefType oldValue) : 
            CallbackArgsBase(prop), 
            m_newValue(newValue), 
            m_oldValue(oldValue)
        {
        }

        template<typename T>
        inline typename ValueChangedArgs<T>::ValueConstRefType ValueChangedArgs<T>::NewValue() const
        {
            return m_newValue;
        }

        template<typename T>
        inline typename ValueChangedArgs<T>::ValueConstRefType ValueChangedArgs<T>::OldValue() const
        {
            return m_oldValue;
        }

        template<typename T>
        void ValueChangedArgs<T>::DefaultCallback(DynamicPropertyHost* /*host*/, const ValueChangedArgs<T>& /*changeArgs*/)
        {
        }

        /**
            *  @brief Class for arguments of coerce value callback functions.
            */
        template<typename T> class CoerceValueArgs : public CallbackArgsBase {
            public:
                /**
                    *  Signature for coerce value callback functions.
                    *  @param host The host, whose value is to be coerced.
                    *  @param args Information about the property and value to coerce.
                    */
                typedef void (*Signature)(DynamicPropertyHost* host, CoerceValueArgs& args);

                /**
                    *  Constructor.
                    *  @param prop The property whose value is to be coerced.
                    *  @param value The value to be coerced.
                    */
                CoerceValueArgs(DynamicPropertyBase& prop, T& value);

                /**
                    *  Retrieves a reference to the value to be coerced. The value can be
                    *  set directly in the coerce value callback function via this reference.
                    *  @return A reference to the value to be coerced.
                    */
                typename TypeInfo<T>::ConstRefType Value() const;

                /**
                    *  Sets the value to be coerced.
                    *  @param value The new value.
                    */
                void SetValue(typename TypeInfo<T>::ConstRefType value) const;

                /**
                    *  The default callback function to use for value coercion, if no callback
                    *  was defined explicitly. By default, a value is not coerced (stays the same).
                    *  @param host The host of the property, whose value should be coerced.
                    *  @param coerceArgs Contains the value to coerce. Change this value, if
                    *                    you want it coerced.
                    */
                static void DefaultCallback(DynamicPropertyHost* host, CoerceValueArgs& coerceArgs);

            private:
                T& m_value;
        };

        template<typename T>
        CoerceValueArgs<T>::CoerceValueArgs(DynamicPropertyBase& prop, T& value) : 
            CallbackArgsBase(prop), 
            m_value(value)
        {
        }

        template<typename T>
        inline typename TypeInfo<T>::ConstRefType CoerceValueArgs<T>::Value() const
        {
            return m_value;
        }

        template<typename T>
        inline void CoerceValueArgs<T>::SetValue(typename TypeInfo<T>::ConstRefType value) const
        {
            m_value = value;
        }

        template<typename T>
        void CoerceValueArgs<T>::DefaultCallback(DynamicPropertyHost* /*host*/, CoerceValueArgs& /*coerceArgs*/)
        {
        }

        /**
            *  Class representing the arguments to a validate value callback function.
            */
        template<typename T> struct ValidateValue {
            /**
                *  Signature for a validate value callback function.
                *  @param value The value to validate.
                *  @return true, if the value is valid, false otherwise. If a validation callback
                *          returns false for a value, the value will not be set.
                */
            typedef bool (*Signature)(typename TypeInfo<T>::ConstRefType value);

            /**
                *  The default callback function to use for validation , if no callback was defined
                *  explicitly. By default, there is no validation.
                *  @param value The value to check for validity.
                *  @return Always true, since by default there is no validation.
                */
            static bool DefaultCallback(typename TypeInfo<T>::ConstRefType value);
        };

        template<typename T>
        bool ValidateValue<T>::DefaultCallback(typename TypeInfo<T>::ConstRefType /*value*/)
        {
            return true;
        }

    }
}
#endif    // !defined(CANDERA_DynamicProperties_CallbackArgs_H)
