//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TreeTraverserBase_H)
#define CANDERA_TreeTraverserBase_H

#include <Candera/Environment.h>

namespace Candera {

    /** @addtogroup CommonBase
     *  @{
     */

    /**
     * @brief Traverses a Node tree depth-first.
     * What is done at each node is customized in the ProcessNode virtual methods.
     */
    template<typename T>
    class TreeTraverserBase {
        public:
            /**
             * Traverser Action.
             */
            enum TraverserAction {
                ProceedTraversing,              ///< Traversing continues with the next child (if the current node has any) or sibling.
                StopTraversingForDescendants,   ///< Traversing will stop for all child nodes and will continue with the next sibling.
                StopTraversing                  ///< Traversing will stop here.
            };

            /**
            *   Constructor
            */
            TreeTraverserBase() {}

            /**
             *  Destructor
             */
            virtual ~TreeTraverserBase();

            /**
             * Depth-first traversing of the given tree.
             * Parents are visited before children.
             *
             * @param root of the tree to traverse.
             */
            void Traverse(T& root);

        protected:
            /**
             *  To be implemented by the derived class. Holds the code what to do for each node. The return value decides if (or how)
             *  traversing will continue. See TreeTraverserBase::TraverserAction.
             *  @return TraverserAction enum value describing how traversing continues.
             */
            virtual TraverserAction ProcessNode(T& node) = 0;

        private:
    };

    template<typename T>
    TreeTraverserBase<T>::~TreeTraverserBase()
    {
    }

    template<typename T>
    void TreeTraverserBase<T>::Traverse(T& root)
    {
        T* currentNode = &root;
        bool allChildrenOfRootProcessed = false;
        T* nextChild = currentNode->GetFirstChild();
        TraverserAction action = ProceedTraversing;

        while ((!allChildrenOfRootProcessed) && (action != StopTraversing)) {
            action = ProceedTraversing;
            // Process the current node before its children.
            if (nextChild == currentNode->GetFirstChild()) {
                // Const cast: see function comment.
                action = ProcessNode(*currentNode);
            }

            // Process the children (if traversal has not been stopped by ProcessNode()'s return at the parent).
            if (action != StopTraversing) {
                const bool allChildrenAlreadyProcessed = (nextChild == 0);

                if (allChildrenAlreadyProcessed || action == StopTraversingForDescendants) {
                    // Up to parent, with next of parent's children (== currentNode's next sibling).
                    if (currentNode == &root) {
                        allChildrenOfRootProcessed = true;
                    }
                    else {
                        nextChild = currentNode->GetNextSibling();
                    }
                    currentNode = currentNode->GetParent();
                }
                else {
                    // down to child
                    currentNode = nextChild;
                    nextChild = currentNode->GetFirstChild();
                }
            }
        }
    }


    /** @} */ // end of CommonBase

} // namespace Candera

#endif  // CANDERA_TreeTraverserBase_H
