//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TREE_ITERATOR_H)
#define CANDERA_TREE_ITERATOR_H

#include <Candera/Environment.h>

namespace Candera {
    /** @addtogroup CommonBase
     *  @{
     */

    /**
     * Iterates over a tree with the possibility to skip iteration of sub-trees.
     *
     * The iteration starts with the root. Then, in each step, the first child
     *  of the current node is iterated over. When a node with no children is
     *  reached, the next sibling of the current node is iterated over. When a
     *  node with no children and no siblings is reached, the iteration goes
     *  to back to the sibling of the first ancestor that has siblings. If the
     *  root node is reached while going back to the ancestors, the iteration is
     *  considered to be finished.
     *
     * @tparam T Node type.
     */
    template <typename T>
    class TreeIterator {
        public:
            /**
             * Path to take during iteration advance.
             */
            enum AdvancePath {
                AdvanceToDescendant, ///< Continue by advancing to descendants.
                AdvanceToSibling     ///< Continue by skipping descendants and advance to siblings or ancestors.
            };

            /**
             * Constructor.
             */
            TreeIterator(T* node);

            /**
             * @brief Get current node.
             *
             * @return Node of current iteration.
             */
            T* GetNode() const;

            /**
             * @brief Advance iteration.
             *
             * @param path Path to advance on (descendants/siblings).
             */
            void Advance(AdvancePath path = AdvanceToDescendant);

            /**
             * Retrieve whether the iterator is valid and GetNode will return a valid
             *  iterated Node or iteration is finished in which case GetNode() will
             *  return 0.
             *
             * @return true if iterator is valid, false otherwise.
             */
            bool IsValid() const;

        private:
            T* m_root;
            T* m_current;
    };

    template <typename T>
    Candera::TreeIterator<T>::TreeIterator(T* node)
        : m_root(node),
          m_current(node)
    {
    }

    template <typename T>
    inline T* Candera::TreeIterator<T>::GetNode() const
    {
        return m_current;
    }

    template <typename T>
    inline bool Candera::TreeIterator<T>::IsValid() const
    {
        return (m_current != 0);
    }

    template <typename T>
    void Candera::TreeIterator<T>::Advance(AdvancePath path)
    {
        T* result = 0;
        if (IsValid()) {
            if (path == AdvanceToDescendant) {
                result = m_current->GetFirstChild();
            }
            while ((result == 0) && (m_current != 0) && (m_current != m_root)) {
                result = m_current->GetNextSibling();
                m_current = m_current->GetParent();
            }
        }

        m_current = result;
    }

    /** @} */ // end of CommonBase

} // namespace Candera

#endif  // CANDERA_TREE_ITERATOR_H
