//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SearchTreeTraverser_H)
    #define CANDERA_SearchTreeTraverser_H

#include <Candera/Environment.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <Candera/EngineBase/Common/TreeTraverserBase.h>
#include <Candera/EngineBase/Common/SearchCriterion.h>

namespace Candera {

    /** @addtogroup CommonBase
     *  @{
     */

    /**
     * @brief Search nodes that meet a search criterion within a node tree.
     *
     * The search tree traverser is a TreeTraverserBase implementation that evaluates each node with a given
     *  SearchCriterion. To start a search within a node tree, the base method Traverse() needs to be called.
     *  The nodes that are fulfilled by the set search criterion are retained and are available to be later retrieved.
     */
    template<typename T>
    class SearchTreeTraverser: public TreeTraverserBase<T>
    {
        FEATSTD_TYPEDEF_BASE(TreeTraverserBase<T>);
        public:
            /**
             *  Constructor
             */
            SearchTreeTraverser();

            /**
             *  Destructor
             */
            virtual ~SearchTreeTraverser() override;

            /**
             * Set SearchCriterion.
             *
             * The search criterion validates nodes based on the implemented criterion.
             *
             * @param searchCriterion Pointer to SearchCriterion.
             */
            void SetSearchCriterion(const SearchCriterion<T>* searchCriterion) { m_searchCriterion = searchCriterion; }

            /**
             * Get SearchCriterion.
             *
             * @return Pointer to SearchCriterion.
             */
            const SearchCriterion<T>* GetSearchCriterion() const { return m_searchCriterion; }

            /**
             * Set OnFoundAction.
             *
             * The OnFoundAction property defines if the search should stop after the first node that fulfills the search criterion
             * is found, or if it should continue to gather all valid nodes.
             * @param onFoundAction Action to be taken by the traverser after a node that fulfills the search criterion is found.
             */
            void SetOnFoundAction(typename Base::TraverserAction onFoundAction) { m_onFoundAction = onFoundAction; }

            /**
             * Get OnFoundAction.
             *
             * @return Action taken by the traverser after a valid node is found.
             * @see SetOnFoundAction.
             */
            typename Base::TraverserAction GetOnFoundAction() const { return m_onFoundAction; }

            /**
             * Get search result.
             *
             * This method returns, one by one, all the nodes that were validated by the SearchTreeTraverserCriterion,
             *  after the tree traversal. The order of the nodes is not defined and each node can be retrieved only once.
             *  When no more nodes are available, the method returns 0.
             * The method returns results only if the search was triggered by the Traverse() method call.
             *
             * @return pointer to node validated by the SearchTreeTraverserCriterion.
             */
            T* GetSearchResult();

        protected:
            virtual typename Base::TraverserAction ProcessNode(T& node) override;

        private:
            Internal::SingleLinkedList<T*> m_foundNodes;
            const SearchCriterion<T>* m_searchCriterion;
            typename Base::TraverserAction m_onFoundAction;
    };

    template<typename T>
    SearchTreeTraverser<T>::SearchTreeTraverser() :
        m_searchCriterion(0), 
        m_onFoundAction(Base::ProceedTraversing) 
    {
    }

    template<typename T>
    SearchTreeTraverser<T>::~SearchTreeTraverser()
    {
    }

    template<typename T>
    T* SearchTreeTraverser<T>::GetSearchResult() 
    {
        if (m_foundNodes.IsEmpty()) {
            return 0;
        }

        T* node = *m_foundNodes.Begin();
        m_foundNodes.Remove(node);
        return node;
    }

    template<typename T>
    typename TreeTraverserBase<T>::TraverserAction SearchTreeTraverser<T>::ProcessNode(T& node) 
    {
        typename Base::TraverserAction result = Base::ProceedTraversing;

        if (m_searchCriterion == 0) {
            result = Base::StopTraversing;
        }
        else {
            if (m_searchCriterion->IsFulfilled(node)) {
                m_foundNodes.Append(&node);
                result = m_onFoundAction;
            }
        }

        return result;
    }


     /** @} */ // end of CommonBase

} // namespace Candera

#endif  // CANDERA_SearchTreeTraverser_H
