//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_ScopeMask_h)
#define Candera_ScopeMask_h

#include <Candera/Environment.h>
#include <Candera/Macros.h>

namespace Candera {

/** @addtogroup CommonBase
 *  @{
 */

/** @brief The class ScopeMask allows scene graph nodes to form conceptual groups independent of the scene graph hierarchy.
 * Two objects are considered to be in the same scope if they have at least one ScopeMask index in common.
 * Known applications (not necessarily in Candera) of Scopes serve for instance following purposes:
 * Camera Scopes (Visibility Culling), Lighting Scopes, and Picking Scopes.
 * ScopeMask is deliberately not inherited or otherwise propagated through a scene graph automatically.
 * By default, a Node has an all-encompassing ScopeMask (all Scopes are enabled).
*/
class ScopeMask
{
    public:
        /**
         * Constructs a default ScopeMask objects with all scopes enabled.
         */
        ScopeMask();

        /**
         * Constructs a ScopeMask object either with all Scopes enabled or disabled.
         * @param enableAllScopes is true if all scopes shall be enabled, false if all scopes shall be disabled.
         */
        explicit ScopeMask(bool enableAllScopes);

        /**
         * Constructs a ScopeMask object with a 32-bit bitfield.
         * @param scopeBits defines a bitfield where each bit specifies a scope index from 0 to 31.
         * A Scope is enabled, if the scope index bit is set to one, disabled if zero.
         */
        explicit ScopeMask(UInt32 scopeBits);

        /**
         * Copy Constructor.
         * @param other ScopeMask to copy from.
         */
        ScopeMask(const ScopeMask& other);

        /**
         * Destructor is non-virtual; Not intended as base class.
         */
        ~ScopeMask();

        /**
         * Enables or disables all scopes in ScopeMask.
         * @param enable is true if all scopes shall be enabled, false otherwise.
         */
        void SetAllScopesEnabled(bool enable);

        /**
         * Enable or disable a certain scope index of this ScopeMask.
         * @param scopeIndex specifies the scope index to be updated. Allowed Scope indices reach from 0 to 31.
         * @param enable is true if scope at index specified shall be enabled, false otherwise.
         * @return True if scopeIndex is in range 0 to 31. False otherwise.
         */
        bool SetScopeEnabled(UInt scopeIndex, bool enable);

        /**
         * Checks if scope at index specified is enabled or not.
         * @param scopeIndex specifies which scope status shall be retrieved.
         * @return true if scope at index given is enabled, false otherwise.
         * Note: If index is out of range, the return value is false. Allowed Scope indices reach from 0 to 31.
         */
        bool IsScopeEnabled(UInt scopeIndex) const;

        /**
         * Checks if this ScopeMask object shares at least one Scope with the ScopeMask given.
         * @param other defines the ScopeMask to test against this ScopeMask object, if any scope enabled overlaps.
         *        Formally, this means ScopeMask A & ScopeMask B != 0;
         * @return true if both ScopeMasks share at least one Scope, false otherwise.
         */
        bool OverlapsWith(const ScopeMask& other) const;

        /**
         * Assigns all scopes of other ScopeMask to this ScopeMask.
         * @param other The other ScopeMask that is copied.
         * @return The result of the operation as a ScopeMask.
         */
        ScopeMask& operator = (const ScopeMask& other);

        /**
         * Compares this ScopeMask with other ScopeMaks
         * @param other The other ScopeMask that is compared.
         * @return true if ALL scopes of both ScopeMasks are identical, false otherwise.
         */
        bool operator == (const ScopeMask& other) const;

        /**
         * Compares this ScopeMask with other ScopeMask.
         * @param other The other ScopeMask that is compared.
         * @return true if scopes of both ScopeMasks are different, false if they are identical.
         */
        bool operator != (const ScopeMask& other) const;

        /**
         *  Unifies the other scope mask with this ScopeMask;
         *  @param other The other ScopeMask to unify with.
         *  @return The result of the operation as a new ScopeMask.
         */
        ScopeMask operator | (const ScopeMask& other) const;


         /**
         *  Intersects the other scope mask with this ScopeMask;
         *  @param other The other ScopeMask to intersect with.
         *  @return The result of the operation as this ScopeMask.
         */
        ScopeMask& operator &= (const ScopeMask& other);

         /**
         *  Unifies the other scope mask with this ScopeMask;
         *  @param other The other ScopeMask to unify with.
         *  @return The result of the operation as this ScopeMask.
         */
        ScopeMask& operator |= (const ScopeMask& other);


    private:
        UInt32 m_bits;

        bool IsScopeIndexValid(UInt32 scopeIndex) const { return scopeIndex < 32; }
};

/**
*  Intersects two ScopeMasks and returns the result;
*  (Implemented as non-member function to avoid Lint confusing it with the unary "address-of" operator.)
*  @return The result of the operation as a new ScopeMask.
*/
inline ScopeMask operator & (ScopeMask lhs,const ScopeMask& rhs){  //lhs is copied, to prevent modifying the original. rhs is unmodified.
    lhs &= rhs;  
    return lhs;
}

 /** @} */ // end of CommonBase

} // namespace Candera

#endif
