//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ScopeMask.h"

namespace Candera {

ScopeMask::ScopeMask() :
    m_bits(0xFFffFFffU)
{
    // Nothing to do.
}


ScopeMask::ScopeMask(bool enableAllScopes) :
    m_bits(enableAllScopes ? 0xFFffFFffU : 0U)
{
}


ScopeMask::ScopeMask(UInt32 scopeBits) :
    m_bits(scopeBits)
{
}


ScopeMask::ScopeMask(const ScopeMask &other) :
    m_bits(other.m_bits)
{
    // Nothing to do.
}


ScopeMask::~ScopeMask()
{
    // Nothing to do.
}


void ScopeMask::SetAllScopesEnabled(bool enable)
{
    m_bits = enable ? 0xFFffFFffU : 0U;
}


bool ScopeMask::SetScopeEnabled(UInt scopeIndex, bool enable)
{
    bool isScopeIndexValid = IsScopeIndexValid(scopeIndex);

    if (isScopeIndexValid) {
        if (enable) {
            m_bits |= (UInt32(1) << scopeIndex);
        }
        else {
            m_bits &= ~(UInt32(1) << scopeIndex);
        }
    }

    return isScopeIndexValid;
}


bool ScopeMask::IsScopeEnabled(UInt scopeIndex) const
{
    bool isScopeIndexValid = IsScopeIndexValid(scopeIndex);
    bool result = false;

    if (isScopeIndexValid) {
        result = (m_bits & (UInt32(1) << scopeIndex)) != 0;
    }

    return result;
}


bool ScopeMask::OverlapsWith(const ScopeMask &other) const
{
    return (m_bits & other.m_bits) != 0;
}


ScopeMask& ScopeMask::operator=(const ScopeMask& other)
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1529, self-assignment of UInt32 m_bits (the only operation here) is harmless)
    m_bits = other.m_bits;
    return *this;
}

bool ScopeMask::operator==(const ScopeMask& other) const
{
    return m_bits == other.m_bits;
}

bool ScopeMask::operator!=(const ScopeMask& other) const
{
    return !(*this == other);
}

ScopeMask ScopeMask::operator | (const ScopeMask& other) const
{
    return ScopeMask(*this) |= other;
}

ScopeMask& ScopeMask::operator &= (const ScopeMask& other)
{
    m_bits &= other.m_bits;
    return *this;
}

ScopeMask& ScopeMask::operator |= (const ScopeMask& other)
{
    m_bits |= other.m_bits;
    return *this;
}

} // namespace Candera
