//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RESOURCEPROVIDER_H)
#define CANDERA_RESOURCEPROVIDER_H

#include <Candera/Environment.h>

namespace Candera {
    struct ResourceDataHandle;
    /** @addtogroup CommonBase
     *  @{
     */

    /**
     *  ResourceProvider provides and guaranties availability of data access for resources referenced by ResourceDataHandle handles.
     */
    class ResourceProvider {
        public:
        /**
         *  @param resourceDataHandle Handle of the resource to be locked.
         *  @param [out] isPersistent Returns whether resource data is persistent. If resource is persistent, data is supposed to be
         * accessible at any time and Retain/Release should have no effect.
         *  @return Pointer to the resource data. Resource data is guaranteed to be valid until Release is called.
         */
        virtual const void* Lock(const ResourceDataHandle& resourceDataHandle, bool& isPersistent) = 0;

        /**
         *  Increase the lock count. It is guaranteed that data is persistent at least until the number of Release operations matches
         * the summed number of Lock and Retain operations on the same resource.
         *  @param data Pointer to resource data that was retrieved by a previous Lock operation.
         */
        virtual void Retain(const void* data) = 0;

        /**
         *  Decrease the lock count. If the lock count reaches 0, it is expected that resource data is not needed anymore and memory can
         * be reused by the ResourceProvider. Successive lock operations on the same resource might return different addresses for the
         * same resource.
         *  @param data Pointer to resource data that was retrieved by a previous Lock operation.
         */
        virtual void Release(const void* data) = 0;

        /**
         *  Read resource chunk without locking it.
         *  @param buffer Buffer where resource chunk shall be read.
         *  @param handle Handle of the resource to be read.
         *  @param offset Offset to the chunk to be read.
         *  @param count Number of bytes to read.
         *  @return Number of bytes actually read.
         */
        virtual SizeType ReadData(void* buffer, const ResourceDataHandle& handle, SizeType offset, SizeType count) = 0;

        /**
         *  ResourceProvider instance that will be used by all ResourceObject objects to lock the resource data. Using CanderaAssetLoader,
         * DefaultAssetProvider will automatically initialize the ResourceProvider instance with its own implementation for providing
         * resources to objects constructed from compact file format (CFF) assets.
         */
        static ResourceProvider*& Instance();
    };

    inline ResourceProvider*& ResourceProvider::Instance()
    {
        static ResourceProvider* resourceProvider = 0;
        return resourceProvider;
    }

    /** @} */ // end of CommonBase
} // namespace Candera

#endif  // CANDERA_RESOURCEPROVIDER_H
