//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RESOURCEOBJECT_H)
#define CANDERA_RESOURCEOBJECT_H

#include <Candera/Environment.h>
#include <FeatStd/MemoryManagement/Disposer.h>
#include <Candera/EngineBase/Common/ResourceDataHandle.h>
#include <Candera/EngineBase/Common/ResourceData.h>

namespace Candera {
    /** @addtogroup CommonBase
     *  @{
     */

    /**
     * @brief ResourceObject provides access to data of ResourceDataHandle objects.
     * 
     * The lifetime of ResourceObjcet objects should be as short as possible, for an 
     *  optimal memory management of asset stored data buffers.
     *
     * @tparam DataType Type of the resource data.
     */
    template <typename DataType>
    class ResourceObject {
        friend bool operator== (const ResourceObject& left, const ResourceObject&right) 
        { return left.m_resource == right.m_resource;}
        public:
            typedef typename FeatStd::TypeTraits::Internal::UnConstType<DataType>::Type MutableData;
            typedef typename FeatStd::TypeTraits::Internal::ConstType<DataType>::Type ConstData;
            typedef typename FeatStd::MemoryManagement::DisposerBase<ConstData*> Disposer;
            typedef typename Disposer::DisposerFunction DisposerFunction;

            /**
             * Create a ResourceDataHandle that stores constant data. The data stored in
             * this object will be available for retrieval at any time until DisposeData()
             * is called. Trying to retrieve mutable data from this ResourceObjcet will fail.
             * 
             * @param data Data to be stored.
             * @param disposerFn Disposer function that will dispose the data when
             *  DisposeData method is called.
             * @param size Size of the data buffer. Should be limited to 2^30.
             * @return Newly ResourceDataHandle that can be used to access the data.
             */
            static ResourceDataHandle CreateHandle(ConstData* data, DisposerFunction disposerFn, UInt32 size);
        
            /**
             * Create a ResourceDataHandle that stores mutable data. The data stored in
             * this object will be available for retrieval at any time until DisposeData()
             * is called. Constant data can always be retrieved from this ResourceDataHandle,
             * while mutable access is defined by the value of the isMutable parameter.
             * 
             * @param data Data to be stored.
             * @param disposerFn Disposer function that will dispose the data when
             *  DisposeData method is called.
             * @param size Size of the data buffer. Should be limited to 2^30.
             * @param isMutable Flag that specifies if the data should be considered
             *  mutable or not. 
             * @return Newly ResourceDataHandle that can be used to access the data.
             */
            static ResourceDataHandle CreateHandle(MutableData* data, DisposerFunction disposerFn, UInt32 size, bool isMutable = true);

            /**
             * @param handle Handle from which data should be retrieved.
             */
            explicit ResourceObject(ResourceDataHandle& handle);

            /**
             * @param handle Handle from which data should be retrieved.
             */
            explicit ResourceObject(const ResourceDataHandle& handle);

            /**
             * @return Constant resource data. Returns 0 if:
             * - direct addressable data was disposed before the creation of this ResourceObject 
             * - asset referenced data is not found in the asset.
             */
            ConstData* GetData() const;
        
            /**
             * @return Mutable resource data. Returns 0 if:
             * - direct addressable data was disposed before the creation of this ResourceObject 
             * - asset referenced data is not found in the asset.
             * - data is marked as constant.
             */
            MutableData* GetMutableData();

            /**
             * @return whether resource data is persistent, in which case data its lifetime is assure after ResourceObject
             *  destruction, or it is transient, in which case resource should not be used after the ResourceObject is destroyed.
             */
            bool IsPersistent() const;
        
            /**
             * Dispose data buffer. Only direct access data, not from asset, can be
             *  disposed, if a valid disposer has been set. As soon as the data is disposed, 
             *  data and the disposer will be set to 0.
             */
            static void DisposeData(ResourceDataHandle& handle);

            /**
             * Read data directly into a given buffer.
             *
             * @param buffer External buffer where data will be read to.
             * @param handle Handle that provides access information for data to be read.
             * @param offset Data offset to start the reading from.
             * @param count Size of data to be read.
             * @return True in case buffer is successfully read, false otherwise.
             */
            static SizeType CopyData(MutableData* buffer, const ResourceDataHandle& handle, OffsetType offset, SizeType count);
            
            /**
             * Get resource address in case data is persistent.
             *
             * @param handle Handle that provides access information for data to be accessed.
             * @return Address of the persistent resource, or 0 if resource is transient.
             */
            static ConstData* GetAddress(const ResourceDataHandle& handle);
        private:
            Candera::Internal::ResourceData m_resource;
    };

    template <typename DataType>
    inline ResourceDataHandle ResourceObject<DataType>::CreateHandle(ConstData* data, DisposerFunction disposerFn, UInt32 size)
    {
        ResourceDataHandle handle = {{{data, reinterpret_cast<ResourceDataHandle::Accessor::Memory::DisposeFn>(disposerFn)}}, size, 0, 0};
        return handle;
    }

    template <typename DataType>
    inline ResourceDataHandle ResourceObject<DataType>::CreateHandle(MutableData* data, DisposerFunction disposerFn, UInt32 size, bool isMutable)
    {
        ResourceDataHandle handle = {{{data, reinterpret_cast<ResourceDataHandle::Accessor::Memory::DisposeFn>(disposerFn)}}, size, static_cast<UInt32>(isMutable ? 1 : 0), 0};
        return handle;
    }

    template <typename DataType>
    inline ResourceObject<DataType>::ResourceObject(ResourceDataHandle& handle)
        :m_resource(handle)
    {
    }

    template <typename DataType>
    inline ResourceObject<DataType>::ResourceObject(const ResourceDataHandle& handle)
        :m_resource(handle)
    {
    }

    template <typename DataType>
    inline typename ResourceObject<DataType>::ConstData* ResourceObject<DataType>::GetData() const
    {
        return static_cast<ConstData*>(m_resource.GetData());
    }

    template <typename DataType>
    inline typename ResourceObject<DataType>::MutableData* ResourceObject<DataType>::GetMutableData()
    {
        return static_cast<MutableData*>(m_resource.GetMutableData());
    }

    template <typename DataType>
    inline bool ResourceObject<DataType>::IsPersistent() const
    {
        return m_resource.IsPersistent();
    }


    template <typename DataType>
    inline void ResourceObject<DataType>::DisposeData(ResourceDataHandle& handle)
    {
        if (handle.m_isFromAsset == 0) {
            if ((handle.m_accessor.m_memory.m_disposerFunction != 0) && (handle.m_accessor.m_memory.m_data != 0)) {
                reinterpret_cast<DisposerFunction>(handle.m_accessor.m_memory.m_disposerFunction)(static_cast<ConstData*>(handle.m_accessor.m_memory.m_data));
            }
            handle.m_accessor.m_memory.m_data = 0;
            handle.m_accessor.m_memory.m_disposerFunction = 0;
        }
    }

    template <typename DataType>
    inline SizeType ResourceObject<DataType>::CopyData(MutableData* buffer, const ResourceDataHandle& handle, OffsetType offset, SizeType count)
    {
        return Candera::Internal::ResourceData::CopyData(buffer, handle, offset, count);
    }

    template <typename DataType>
    inline typename ResourceObject<DataType>::ConstData* ResourceObject<DataType>::GetAddress(const ResourceDataHandle& handle)
    {
        return static_cast<ConstData*>(Candera::Internal::ResourceData::GetAddress(handle));
    }

    /** @} */ // end of CommonBase
} // namespace Candera

#endif  // CANDERA_RESOURCEOBJECT_H
